<?php
/**
 * Plugin Name: CloudScale Free Backup and Restore
 * Description: No-nonsense WordPress backup and restore. Backs up database, media, plugins and themes into a single zip. Scheduled or manual, with safe restore and maintenance mode.
 * Version: 2.29.0
 * Author: CloudScale
 * License: GPL2
 */

defined('ABSPATH') || exit;

define('CS_VERSION',    '2.29.0');
define('CS_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CS_BACKUP_DIR', WP_CONTENT_DIR . '/cloudscale-backups/');
define('CS_MAINT_FILE', ABSPATH . '.maintenance');

// ============================================================
// Bootstrap
// ============================================================

register_activation_hook(__FILE__, 'cs_activate');
register_deactivation_hook(__FILE__, 'cs_deactivate');

function cs_activate() {
    cs_ensure_backup_dir();
    cs_reschedule();
}

function cs_deactivate() {
    wp_clear_scheduled_hook('cs_scheduled_backup');
    cs_maintenance_off();
}

function cs_ensure_backup_dir(): void {
    if (!file_exists(CS_BACKUP_DIR)) {
        wp_mkdir_p(CS_BACKUP_DIR);
    }
    $htaccess = CS_BACKUP_DIR . '.htaccess';
    if (!file_exists($htaccess)) {
        file_put_contents($htaccess, "Deny from all\n");
    }
    $index = CS_BACKUP_DIR . 'index.php';
    if (!file_exists($index)) {
        file_put_contents($index, "<?php // Silence\n");
    }
}

// ============================================================
// Scheduling — day-of-week based
// ============================================================

// Ensure daily interval exists
add_filter('cron_schedules', function (array $schedules): array {
    if (!isset($schedules['daily'])) {
        $schedules['daily'] = [
            'interval' => DAY_IN_SECONDS,
            'display'  => 'Once Daily',
        ];
    }
    return $schedules;
});

function cs_get_run_days(): array {
    global $wpdb;
    // Read directly from DB — bypasses object cache entirely
    $raw = $wpdb->get_var("SELECT option_value FROM {$wpdb->options} WHERE option_name = 'cs_run_days'");
    if ($raw === null) {
        return [1, 3, 5]; // option row doesn't exist yet — install default
    }
    $saved = maybe_unserialize($raw);
    // Empty array in DB means no days saved yet — return defaults
    if (!is_array($saved) || empty($saved)) {
        return [1, 3, 5];
    }
    return array_map('intval', $saved);
}

function cs_reschedule(): void {
    wp_clear_scheduled_hook('cs_scheduled_backup');

    $enabled = get_option('cs_schedule_enabled', false);
    if (!$enabled) return;

    $hour     = intval(get_option('cs_run_hour', 3));
    $timezone = wp_timezone();
    $now      = new DateTime('now', $timezone);
    $run_days = cs_get_run_days();

    // Walk forward up to 8 days to find next matching day/time
    $candidate = clone $now;
    $candidate->setTime($hour, 0, 0);

    for ($i = 0; $i <= 7; $i++) {
        $dow = intval($candidate->format('N')); // 1=Mon...7=Sun
        if (in_array($dow, $run_days, true) && $candidate > $now) {
            break;
        }
        $candidate->modify('+1 day');
        $candidate->setTime($hour, 0, 0);
    }

    wp_schedule_event($candidate->getTimestamp(), 'daily', 'cs_scheduled_backup');
}

add_action('cs_scheduled_backup', function () {
    // Skip days not in the configured run-day list
    $run_days = cs_get_run_days();
    $today    = intval((new DateTime('now', wp_timezone()))->format('N'));
    if (!in_array($today, $run_days, true)) return;

    set_time_limit(0);
    ignore_user_abort(true);
    cs_ensure_backup_dir();
    cs_create_backup(true, true, true, true);
    cs_enforce_retention();
});

// ============================================================
// Admin menu
// ============================================================

add_action('admin_menu', function () {
    add_management_page(
        'CloudScale Free Backup and Restore',
        'CloudScale Backup & Restore',
        'manage_options',
        'cloudscale-backup',
        'cs_admin_page'
    );
});

add_action('admin_enqueue_scripts', function (string $hook): void {
    if ($hook !== 'tools_page_cloudscale-backup') return;
    wp_enqueue_style('cs-style',  plugin_dir_url(__FILE__) . 'assets/style.css',  [], CS_VERSION);
    wp_enqueue_script('cs-script', plugin_dir_url(__FILE__) . 'assets/script.js', ['jquery'], CS_VERSION, true);
    wp_localize_script('cs-script', 'CS', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('cs_nonce'),
        'site_url' => get_site_url(),
    ]);
});

// ============================================================
// Admin page
// ============================================================

function cs_admin_page(): void {
    cs_ensure_backup_dir();

    $backups      = cs_list_backups();
    $upload_dir   = wp_upload_dir();
    $upload_size   = cs_dir_size($upload_dir['basedir']);
    $plugins_size  = cs_dir_size(WP_PLUGIN_DIR);
    $themes_size   = cs_dir_size(get_theme_root());
    // "Other" backup targets
    $mu_path       = defined('WPMU_PLUGIN_DIR') ? WPMU_PLUGIN_DIR : WP_CONTENT_DIR . '/mu-plugins';
    $mu_size       = is_dir($mu_path) ? cs_dir_size($mu_path) : 0;
    $lang_path     = WP_CONTENT_DIR . '/languages';
    $lang_size     = is_dir($lang_path) ? cs_dir_size($lang_path) : 0;
    $htaccess_path = ABSPATH . '.htaccess';
    $htaccess_size = file_exists($htaccess_path) ? (int) filesize($htaccess_path) : 0;
    $wpconfig_path = ABSPATH . 'wp-config.php';
    $wpconfig_size = file_exists($wpconfig_path) ? (int) filesize($wpconfig_path) : 0;
    $dropins_size  = 0;
    foreach (glob(WP_CONTENT_DIR . '/*.php') ?: [] as $_f) { $dropins_size += (int) filesize($_f); }

    // Estimate database size from information_schema
    global $wpdb;
    $db_size = (int) $wpdb->get_var($wpdb->prepare(
        "SELECT SUM(data_length + index_length)
         FROM information_schema.TABLES
         WHERE table_schema = %s",
        DB_NAME
    ));
    $next_run     = wp_next_scheduled('cs_scheduled_backup');
    $maint_active = file_exists(CS_MAINT_FILE);

    // Disk space
    $free_bytes  = disk_free_space(CS_BACKUP_DIR);
    if ($free_bytes === false) $free_bytes = disk_free_space(ABSPATH);
    $total_bytes = disk_total_space(CS_BACKUP_DIR);
    if ($total_bytes === false) $total_bytes = disk_total_space(ABSPATH);
    $latest_size = !empty($backups) ? $backups[0]['size'] : 0;

    // Traffic-light: how many more backups fit in free space?
    // Red < 4, Amber < 10, Green >= 10. Fallback to size-relative if no backups yet.
    $baseline     = $latest_size > 0 ? $latest_size : 100 * 1024 * 1024;
    $backups_fit  = ($free_bytes !== false && $baseline > 0) ? (int) floor($free_bytes / $baseline) : null;
    $disk_status  = 'green';
    if ($free_bytes !== false) {
        if ($backups_fit !== null && $backups_fit < 4)   $disk_status = 'red';
        elseif ($backups_fit !== null && $backups_fit < 10) $disk_status = 'amber';
    }

    // Show banner for amber or red
    $show_disk_alert = ($disk_status !== 'green');

    // Handle form POST for schedule save (plain form, no redirect)
    $cs_schedule_saved_msg = '';
    if (isset($_POST['cs_action']) && $_POST['cs_action'] === 'save_schedule') {
        check_admin_referer('cs_nonce', 'nonce');
        $raw_days   = isset($_POST['run_days']) && is_array($_POST['run_days']) ? $_POST['run_days'] : [];
        $clean_days = array_values(array_filter(array_map('intval', $raw_days), fn($d) => $d >= 1 && $d <= 7));
        update_option('cs_run_days',         $clean_days);
        update_option('cs_run_days_saved',   '1');
        update_option('cs_schedule_enabled', !empty($_POST['schedule_enabled']));
        update_option('cs_run_hour',         max(0, min(23, intval($_POST['run_hour'] ?? 3))));
        wp_cache_delete('cs_run_days',         'options');
        wp_cache_delete('cs_run_days_saved',   'options');
        wp_cache_delete('cs_schedule_enabled', 'options');
        wp_cache_delete('alloptions',          'options');
        cs_reschedule();
        $day_names = ['','Mon','Tue','Wed','Thu','Fri','Sat','Sun'];
        $saved_labels = implode(', ', array_map(fn($d) => $day_names[$d] ?? $d, $clean_days));
        $cs_schedule_saved_msg = $saved_labels ?: 'No days selected';
    }

    // Settings — read after any POST save so page shows updated values
    $enabled      = isset($_POST['cs_action']) ? !empty($_POST['schedule_enabled']) : (bool) get_option('cs_schedule_enabled', false);
    $run_days_sel = cs_get_run_days();
    $hour         = intval(get_option('cs_run_hour', 3));
    $retention    = intval(get_option('cs_retention', 8));
    $dump_method  = cs_mysqldump_available() ? 'mysqldump (native — fast)' : 'PHP streamed (compatible)';
    $restore_method = cs_mysql_cli_available() ? 'mysql CLI (native — fast)' : 'PHP streamed (compatible)';

    // MySQL version
    $mysql_version = $wpdb->get_var('SELECT VERSION()') ?: 'Unknown';
    $db_label      = 'MySQL ' . $mysql_version . ' — ' . DB_NAME;

    // Disk usage percentage for bar fill
    $free_pct = ($free_bytes !== false && $total_bytes > 0)
        ? min(100, round(($free_bytes / $total_bytes) * 100))
        : null;

    $days_map = [1 => 'Mon', 2 => 'Tue', 3 => 'Wed', 4 => 'Thu', 5 => 'Fri', 6 => 'Sat', 7 => 'Sun'];
    ?>
    <div class="wrap cs-wrap">

        <!-- ===================== HEADER ===================== -->
        <div class="cs-header">
            <div class="cs-header-title">
                <h1>☁ CloudScale Free Backup &amp; Restore</h1>
                <p class="cs-header-sub">Database · media · plugins · themes. No timeouts, no external services.</p>
                <p class="cs-header-free">✅ 100% free forever — no licence, no premium tier, no feature restrictions. Everything is included.</p>
            </div>
            <div class="cs-header-status">
                <?php if ($maint_active): ?>
                    <span class="cs-badge cs-badge-warn">⚠ Maintenance Mode Active</span>
                <?php else: ?>
                    <span class="cs-badge cs-badge-ok">● Site Online</span>
                <?php endif; ?>
            </div>
        </div>

        <!-- ===================== DISK SPACE ALERT ===================== -->
        <?php if ($show_disk_alert): ?>
        <div class="cs-alert cs-alert-<?php echo $disk_status; ?>">
            <?php if ($disk_status === 'red'): ?>
                <span class="cs-alert-icon">🔴</span>
                <div><strong>Critical: Very low disk space</strong> —
                <?php echo cs_format_size((int)$free_bytes); ?> free<?php if ($backups_fit !== null): ?>, room for approximately <strong><?php echo $backups_fit; ?> more backup(s)</strong><?php endif; ?>.
                Free up space or move old backups off-server immediately.</div>
            <?php else: ?>
                <span class="cs-alert-icon">🟡</span>
                <div><strong>Warning: Disk space is running low</strong> —
                <?php echo cs_format_size((int)$free_bytes); ?> free<?php if ($backups_fit !== null): ?>, room for approximately <strong><?php echo $backups_fit; ?> more backup(s)</strong><?php endif; ?>.
                Consider freeing space or reducing your retention count.</div>
            <?php endif; ?>
        </div>
        <?php endif; ?>

        <!-- ===================== SETTINGS ===================== -->
        <div class="cs-section-ribbon"><span>Schedule &amp; Settings</span></div>
        <div class="cs-grid cs-grid-3">

            <!-- SCHEDULE CARD -->
            <form method="post" action="" id="cs-schedule-form">
                <?php wp_nonce_field('cs_nonce', 'nonce'); ?>
                <input type="hidden" name="cs_action" value="save_schedule">
            <div class="cs-card cs-card--blue">
                <div class="cs-card-stripe cs-stripe--blue" style="background:linear-gradient(135deg,#1565c0 0%,#2196f3 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">⏰ Backup Schedule</h2></div>

                <!-- Enable/disable checkbox — inline with label -->
                <div class="cs-field-group">
                    <label class="cs-enable-label">
                        <input type="checkbox" id="cs-schedule-enabled" name="schedule_enabled" value="1" <?php checked($enabled); ?>>
                        Enable automatic backups
                    </label>
                </div>

                <!-- fieldset[disabled] is the only reliable cross-browser disable mechanism -->
                <fieldset id="cs-schedule-controls" <?php echo !$enabled ? 'disabled' : ''; ?> class="cs-schedule-fieldset">
                    <legend class="screen-reader-text">Schedule controls</legend>

                    <div class="cs-field-group">
                        <span class="cs-field-label">Run on these days</span>
                        <div class="cs-day-checks">
                            <?php foreach ($days_map as $num => $day_label): ?>
                            <label class="cs-day-check-label">
                                <input type="checkbox"
                                       class="cs-day-check"
                                       name="run_days[]"
                                       value="<?php echo $num; ?>"
                                       <?php checked(in_array($num, $run_days_sel, false)); ?>>
                                <?php echo $day_label; ?>
                            </label>
                            <?php endforeach; ?>
                        </div>
                        <p class="cs-help">Select one or more days. Backup runs once per selected day at the time below.</p>
                    </div>

                    <div class="cs-field-group">
                        <label class="cs-field-label" for="cs-run-hour">Run at time (server)</label>
                        <div class="cs-inline">
                            <select id="cs-run-hour" class="cs-input-sm">
                                <?php for ($h = 0; $h < 24; $h++): ?>
                                    <option value="<?php echo $h; ?>" <?php selected($hour, $h); ?>><?php echo str_pad($h, 2, '0', STR_PAD_LEFT); ?>:00</option>
                                <?php endfor; ?>
                            </select>
                            <span class="cs-muted-text">server time</span>
                        </div>
                        <p class="cs-help">Now: <?php echo current_time('H:i T'); ?> &nbsp;·&nbsp; TZ: <?php echo wp_timezone_string(); ?></p>
                    </div>

                    <?php if ($next_run): ?>
                    <div class="cs-next-run">
                        <span class="cs-next-run-label">Next scheduled run</span>
                        <span class="cs-next-run-time"><?php echo get_date_from_gmt(date('Y-m-d H:i:s', $next_run), 'D j M \a\t H:i'); ?></span>
                    </div>
                    <?php endif; ?>

                </fieldset>

                <div id="cs-off-notice" class="cs-off-notice" <?php echo $enabled ? 'style="display:none"' : ''; ?>>
                    Automatic backups are <strong>off</strong>. Enable the checkbox above to configure a schedule, or run backups manually below.
                </div>

                <button type="submit" name="cs_save_schedule" class="button button-primary cs-mt">Save Schedule</button>
                <?php if ($cs_schedule_saved_msg): ?>
                <span class="cs-saved-msg" style="display:inline;margin-left:10px">✓ <?php echo esc_html($cs_schedule_saved_msg); ?></span>
                <?php endif; ?>

            </div>
            </form>

            <!-- RETENTION CARD -->
            <?php
            // Compute retention storage estimate server-side
            $ret_needed       = $latest_size > 0 ? $retention * $latest_size : 0;
            $ret_over         = $ret_needed > 0 && $free_bytes !== false && $ret_needed > (int)$free_bytes;
            $ret_has_baseline = $latest_size > 0;
            ?>
            <div class="cs-card cs-card--green">
                <div class="cs-card-stripe cs-stripe--green" style="background:linear-gradient(135deg,#2e7d32 0%,#43a047 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">🗂 Retention &amp; Storage</h2></div>

                <div class="cs-field-group">
                    <label class="cs-field-label">Keep last</label>
                    <div class="cs-inline">
                        <input type="number" id="cs-retention"
                               value="<?php echo $retention; ?>" min="1" max="9999"
                               class="cs-input-sm <?php echo $ret_over ? 'cs-retention-over' : ''; ?>">
                        <span class="cs-muted-text">backups</span>
                        <?php if ($ret_has_baseline): ?>
                        <span id="cs-retention-tl"
                              class="cs-ret-tl cs-ret-tl--<?php echo $ret_over ? 'red' : 'green'; ?>">
                            <span class="cs-tl-dot"></span>
                        </span>
                        <?php endif; ?>
                    </div>

                    <div id="cs-retention-storage" class="cs-retention-storage"
                         data-latest-size="<?php echo (int)$latest_size; ?>"
                         data-free-bytes="<?php echo $free_bytes !== false ? (int)$free_bytes : 0; ?>">
                        <?php if ($ret_has_baseline): ?>
                        <div class="cs-ret-row">
                            <span class="cs-ret-label">Estimated storage needed</span>
                            <span id="cs-retention-est"
                                  class="cs-ret-val <?php echo $ret_over ? 'cs-retention-est--over' : ''; ?>">
                                <?php echo cs_format_size($ret_needed); ?>
                            </span>
                        </div>
                        <div class="cs-ret-row">
                            <span class="cs-ret-label">Disk free space</span>
                            <span class="cs-ret-val cs-free-<?php echo $disk_status; ?>">
                                <?php echo $free_bytes !== false ? cs_format_size((int)$free_bytes) : 'Unknown'; ?>
                            </span>
                        </div>
                        <?php else: ?>
                        <div class="cs-ret-row">
                            <span id="cs-retention-est" class="cs-retention-no-baseline">
                                Run your first backup to see a storage estimate.
                            </span>
                        </div>
                        <?php endif; ?>
                        <div id="cs-retention-warn" class="cs-retention-warn"
                             <?php echo $ret_over ? '' : 'style="display:none"'; ?>>
                            ⚠ Estimated storage exceeds available disk space — reduce the retention count or free up space.
                        </div>
                    </div>

                    <p class="cs-help">Oldest backups beyond this limit are deleted automatically after each run.</p>
                </div>

                <div class="cs-storage-grid">
                    <div class="cs-storage-item">
                        <span class="cs-storage-label">Backup storage used</span>
                        <span class="cs-storage-value cs-value--blue"><?php echo cs_format_size(cs_dir_size(CS_BACKUP_DIR)); ?></span>
                    </div>
                    <div class="cs-storage-item">
                        <span class="cs-storage-label">Media uploads</span>
                        <span class="cs-storage-value"><?php echo cs_format_size($upload_size); ?></span>
                    </div>
                    <div class="cs-storage-item">
                        <span class="cs-storage-label">Plugins folder</span>
                        <span class="cs-storage-value"><?php echo cs_format_size($plugins_size); ?></span>
                    </div>
                    <div class="cs-storage-item">
                        <span class="cs-storage-label">Themes folder</span>
                        <span class="cs-storage-value"><?php echo cs_format_size($themes_size); ?></span>
                    </div>
                </div>

                <button id="cs-save-retention" class="button button-primary cs-mt">Save Retention</button>
                <span id="cs-retention-saved" class="cs-saved-msg" style="display:none">✓ Saved</span>
            </div>

            <!-- SYSTEM INFO CARD -->
            <div class="cs-card cs-card--purple">
                <div class="cs-card-stripe cs-stripe--purple" style="background:linear-gradient(135deg,#6a1b9a 0%,#8e24aa 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">⚙ System Info</h2></div>

                <div class="cs-info-row">
                    <span>Backup method</span>
                    <strong><?php echo esc_html($dump_method); ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>Restore method</span>
                    <strong><?php echo esc_html($restore_method); ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>PHP memory limit</span>
                    <strong><?php echo ini_get('memory_limit'); ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>Max execution time</span>
                    <strong><?php echo ini_get('max_execution_time') === '0' ? 'Unlimited' : ini_get('max_execution_time') . 's'; ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>Database</span>
                    <strong><?php echo esc_html($db_label); ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>Total backups stored</span>
                    <strong><?php echo count($backups); ?></strong>
                </div>
                <div class="cs-info-row">
                    <span>Backup directory</span>
                    <strong class="cs-path"><?php echo esc_html(CS_BACKUP_DIR); ?></strong>
                </div>

                <!-- Traffic-light disk row -->
                <div class="cs-info-row cs-disk-row">
                    <span>Disk free space</span>
                    <strong class="cs-tl cs-tl--<?php echo $disk_status; ?>">
                        <span class="cs-tl-dot"></span>
                        <?php echo $free_bytes !== false ? cs_format_size((int)$free_bytes) : 'Unavailable'; ?>
                        <?php if ($total_bytes): ?>
                            <span class="cs-disk-of">/ <?php echo cs_format_size((int)$total_bytes); ?></span>
                        <?php endif; ?>
                    </strong>
                </div>
                <div class="cs-info-row">
                    <span>Latest backup size</span>
                    <strong><?php echo $latest_size > 0 ? cs_format_size($latest_size) : '—'; ?></strong>
                </div>

                <?php if ($free_pct !== null): ?>
                <div class="cs-info-row">
                    <span>Percentage free space</span>
                    <strong class="cs-tl cs-tl--<?php echo $disk_status; ?>">
                        <span class="cs-tl-dot"></span>
                        <?php echo $free_pct; ?>%
                    </strong>
                </div>
                <div class="cs-disk-bar-wrap">
                    <div class="cs-disk-bar">
                        <div class="cs-disk-bar-fill cs-disk-fill--<?php echo $disk_status; ?>"
                             style="width:<?php echo $free_pct; ?>%"></div>
                    </div>
                </div>
                <?php endif; ?>
            </div>

        </div><!-- /cs-grid-3 -->

        <!-- ===================== MANUAL BACKUP ===================== -->
        <div class="cs-section-ribbon"><span>Manual Backup</span></div>
        <div class="cs-card cs-card--orange cs-full">
            <div class="cs-card-stripe cs-stripe--orange" style="background:linear-gradient(135deg,#e65100 0%,#f57c00 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">▶ Run Backup Now</h2></div>
            <?php
            // Pass sizes to JS for live total calculation
            $backup_sizes = [
                'db'        => $db_size,
                'media'     => $upload_size,
                'plugins'   => $plugins_size,
                'themes'    => $themes_size,
                'mu'        => $mu_size,
                'languages' => $lang_size,
                'htaccess'  => $htaccess_size,
                'wpconfig'  => $wpconfig_size,
                'dropins'   => $dropins_size,
                'free'      => $free_bytes !== false ? (int)$free_bytes : 0,
            ];
            ?>
            <script>
            window.CS_BACKUP_SIZES = <?php echo json_encode($backup_sizes); ?>;
            </script>
            <div class="cs-run-grid">
                <div>
                    <div class="cs-options">
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-db" checked data-size="<?php echo $db_size; ?>"> Include database <code><?php echo $db_size > 0 ? cs_format_size($db_size) : '~unknown'; ?></code></label>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-media" checked data-size="<?php echo $upload_size; ?>"> Include media uploads <code><?php echo cs_format_size($upload_size); ?></code></label>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-plugins" checked data-size="<?php echo $plugins_size; ?>"> Include plugins folder <code><?php echo cs_format_size($plugins_size); ?></code></label>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-themes" checked data-size="<?php echo $themes_size; ?>"> Include themes folder <code><?php echo cs_format_size($themes_size); ?></code></label>

                        <div class="cs-options-divider">
                            <span class="cs-options-divider-label">Other</span>
                        </div>

                        <?php if ($mu_size > 0): ?>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-mu" checked data-size="<?php echo $mu_size; ?>"> Must-use plugins <code><?php echo cs_format_size($mu_size); ?></code></label>
                        <?php endif; ?>

                        <?php if ($lang_size > 0): ?>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-languages" checked data-size="<?php echo $lang_size; ?>"> Languages <code><?php echo cs_format_size($lang_size); ?></code></label>
                        <?php endif; ?>

                        <?php if ($dropins_size > 0): ?>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-dropins" data-size="<?php echo $dropins_size; ?>"> wp-content dropin files <small>(object-cache.php, db.php…)</small> <code><?php echo cs_format_size($dropins_size); ?></code></label>
                        <?php endif; ?>

                        <?php if ($htaccess_size > 0): ?>
                        <label class="cs-option-label"><input type="checkbox" id="cs-include-htaccess" checked data-size="<?php echo $htaccess_size; ?>"> .htaccess <code><?php echo cs_format_size($htaccess_size); ?></code></label>
                        <?php endif; ?>

                        <?php if ($wpconfig_size > 0): ?>
                        <label class="cs-option-label cs-option-sensitive">
                            <input type="checkbox" id="cs-include-wpconfig" data-size="<?php echo $wpconfig_size; ?>">
                            wp-config.php <code><?php echo cs_format_size($wpconfig_size); ?></code>
                            <span class="cs-sensitive-badge">⚠ contains DB credentials</span>
                        </label>
                        <?php endif; ?>
                    </div>

                    <!-- Live total + free space summary -->
                    <div class="cs-backup-summary" id="cs-backup-summary">
                        <div class="cs-backup-summary-row">
                            <span class="cs-summary-label">Estimated backup size</span>
                            <span class="cs-summary-value" id="cs-total-size">—</span>
                        </div>
                        <div class="cs-backup-summary-row">
                            <span class="cs-summary-label">Disk free space</span>
                            <span class="cs-summary-value <?php echo 'cs-free-' . $disk_status; ?>" id="cs-free-space">
                                <?php echo $free_bytes !== false ? cs_format_size((int)$free_bytes) : 'Unknown'; ?>
                            </span>
                        </div>
                        <div class="cs-backup-summary-row" id="cs-space-warn-row" style="display:none">
                            <span class="cs-summary-warn" id="cs-space-warn"></span>
                        </div>
                    </div>

                    <button id="cs-run-backup" class="button button-primary cs-btn-lg">▶ Run Backup Now</button>
                </div>
                <div id="cs-backup-progress" style="display:none" class="cs-progress-panel">
                    <p id="cs-backup-msg" class="cs-progress-msg">Starting backup...</p>
                    <div class="cs-progress-bar"><div id="cs-backup-fill" class="cs-progress-fill"></div></div>
                    <p class="cs-help">Do not close this window. Large sites may take several minutes.</p>
                </div>
            </div>
        </div>

        <!-- ===================== BACKUP HISTORY ===================== -->
        <div class="cs-section-ribbon"><span>Backup History</span></div>
        <div class="cs-card cs-card--teal cs-full">
            <div class="cs-card-stripe cs-stripe--teal" style="background:linear-gradient(135deg,#004d40 0%,#00897b 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">📋 Backup History</h2></div>
            <?php if (empty($backups)): ?>
                <p class="cs-empty">No backups yet. Run your first backup above.</p>
            <?php else: ?>
            <table class="widefat cs-table">
                <thead>
                    <tr>
                        <th style="width:36px">#</th>
                        <th>Filename</th>
                        <th>Type</th>
                        <th>Size</th>
                        <th>Created</th>
                        <th>Age</th>
                        <th style="width:280px">Actions</th>
                    </tr>
                </thead>
                <tbody>
                <?php foreach ($backups as $i => $b):
                    $is_db = (str_contains($b['name'], '_db') || str_contains($b['name'], '_full'));
                ?>
                    <tr class="<?php echo $i === 0 ? 'cs-row-latest' : ''; ?>">
                        <td class="cs-idx"><?php echo $i + 1; ?><?php if ($i === 0) echo ' <span class="cs-latest-badge">latest</span>'; ?></td>
                        <td class="cs-filename" title="<?php echo esc_attr($b['name']); ?>"><?php echo esc_html($b['name']); ?></td>
                        <td><span class="cs-type-badge cs-type-<?php echo esc_attr(strtolower(explode(' ', $b['type'])[0])); ?>"><?php echo esc_html($b['type']); ?></span></td>
                        <td><?php echo esc_html(cs_format_size($b['size'])); ?></td>
                        <td><?php echo esc_html($b['date']); ?></td>
                        <td class="cs-age"><?php echo esc_html(cs_human_age($b['mtime'])); ?></td>
                        <td class="cs-actions">
                            <a href="<?php echo esc_url(admin_url('admin-post.php?action=cs_download&file=' . urlencode($b['name']) . '&_wpnonce=' . wp_create_nonce('cs_download'))); ?>" class="button button-small" title="Download">⬇ Download</a>
                            <?php if ($is_db): ?>
                            <button class="button button-small cs-restore-btn"
                                    data-file="<?php echo esc_attr($b['name']); ?>"
                                    data-date="<?php echo esc_attr($b['date']); ?>"
                                    title="Restore database">↩ Restore DB</button>
                            <?php endif; ?>
                            <button class="button button-small cs-delete-btn" data-file="<?php echo esc_attr($b['name']); ?>" title="Delete backup">🗑 Delete</button>
                        </td>
                    </tr>
                <?php endforeach; ?>
                </tbody>
            </table>
            <p class="cs-help cs-mt">Showing <?php echo count($backups); ?> backup(s). Retention limit: <?php echo $retention; ?>. Oldest backups are removed automatically after each run.</p>
            <?php endif; ?>
        </div>

        <!-- ===================== RESTORE FROM UPLOAD ===================== -->
        <div class="cs-section-ribbon"><span>Restore from File</span></div>
        <div class="cs-card cs-card--red cs-full">
            <div class="cs-card-stripe cs-stripe--red" style="background:linear-gradient(135deg,#b71c1c 0%,#e53935 100%);display:flex;align-items:center;padding:0 20px;height:52px;margin:0 -20px 20px -20px;border-radius:10px 10px 0 0;"><h2 class="cs-card-heading" style="color:#fff!important;font-size:0.95rem;font-weight:700;margin:0;padding:0;line-height:1.3;border:none;background:none;text-shadow:0 1px 3px rgba(0,0,0,0.3);">↩ Restore from Uploaded File</h2></div>
            <div class="cs-restore-upload-grid">
                <div>
                    <p>Upload a <code>.zip</code> (from this plugin) or a raw <code>.sql</code> file to restore the database.</p>
                    <input type="file" id="cs-restore-file" accept=".zip,.sql">
                    <button id="cs-restore-upload-btn" class="button button-secondary cs-mt">↩ Restore from Upload</button>
                </div>
                <div id="cs-restore-upload-progress" style="display:none" class="cs-progress-panel">
                    <p id="cs-restore-upload-msg" class="cs-progress-msg">Uploading...</p>
                    <div class="cs-progress-bar"><div id="cs-restore-upload-fill" class="cs-progress-fill"></div></div>
                </div>
            </div>
        </div>

        <!-- ===================== RESTORE MODAL ===================== -->
        <div id="cs-restore-modal" class="cs-modal" style="display:none">
            <div class="cs-modal-box">
                <h2>⚠ Restore Database</h2>
                <div class="cs-modal-body">
                    <div class="cs-warning-box">
                        <strong>BEFORE YOU RESTORE — TAKE A SNAPSHOT</strong>
                        <ul>
                            <li>Take a server/VM snapshot in your hosting control panel or AWS console <strong>right now</strong>.</li>
                            <li>If anything goes wrong you can roll back to the snapshot instantly.</li>
                            <li>This restore will put the site into maintenance mode, drop and recreate all database tables, then bring the site back online.</li>
                            <li>Active sessions will be interrupted. Users will see a maintenance page.</li>
                        </ul>
                    </div>
                    <p>You are about to restore from:</p>
                    <p><strong id="cs-modal-filename"></strong> &nbsp;|&nbsp; Created: <span id="cs-modal-date"></span></p>
                    <div class="cs-confirm-check">
                        <label>
                            <input type="checkbox" id="cs-confirm-snapshot">
                            I have taken a server snapshot and understand this will overwrite the live database.
                        </label>
                    </div>
                </div>
                <div class="cs-modal-footer">
                    <button id="cs-modal-cancel" class="button">Cancel — keep current database</button>
                    <button id="cs-modal-confirm" class="button button-primary cs-btn-danger" disabled>Restore Now</button>
                </div>
                <div id="cs-modal-progress" style="display:none" class="cs-modal-progress">
                    <p id="cs-modal-progress-msg" class="cs-progress-msg">Enabling maintenance mode...</p>
                    <div class="cs-progress-bar"><div id="cs-modal-fill" class="cs-progress-fill"></div></div>
                    <p class="cs-help">Do not close this window.</p>
                </div>
            </div>
        </div>
        <div id="cs-modal-overlay" class="cs-modal-overlay" style="display:none"></div>

    </div><!-- /cs-wrap -->
    <?php
}

// ============================================================
// AJAX — Run backup
// ============================================================

add_action('wp_ajax_cs_run_backup', function (): void {
    cs_verify_nonce();
    cs_ensure_backup_dir();

    $include_db        = !empty($_POST['include_db']);
    $include_media     = !empty($_POST['include_media']);
    $include_plugins   = !empty($_POST['include_plugins']);
    $include_themes    = !empty($_POST['include_themes']);
    $include_mu        = !empty($_POST['include_mu']);
    $include_languages = !empty($_POST['include_languages']);
    $include_dropins   = !empty($_POST['include_dropins']);
    $include_htaccess  = !empty($_POST['include_htaccess']);
    $include_wpconfig  = !empty($_POST['include_wpconfig']);

    if (!$include_db && !$include_media && !$include_plugins && !$include_themes
        && !$include_mu && !$include_languages && !$include_dropins
        && !$include_htaccess && !$include_wpconfig) {
        wp_send_json_error('Select at least one option.');
    }

    set_time_limit(0);
    ignore_user_abort(true);

    try {
        $filename = cs_create_backup(
            $include_db, $include_media, $include_plugins, $include_themes,
            $include_mu, $include_languages, $include_dropins, $include_htaccess, $include_wpconfig
        );
        cs_enforce_retention();
        wp_send_json_success([
            'message'  => 'Backup complete: ' . $filename,
            'filename' => $filename,
        ]);
    } catch (Exception $e) {
        wp_send_json_error($e->getMessage());
    }
});

// ============================================================
// AJAX — Delete backup
// ============================================================

add_action('wp_ajax_cs_delete_backup', function (): void {
    cs_verify_nonce();
    $file = sanitize_file_name($_POST['file'] ?? '');
    $path = CS_BACKUP_DIR . $file;
    if (file_exists($path) && strpos(realpath($path), realpath(CS_BACKUP_DIR)) === 0) {
        unlink($path);
        wp_send_json_success('Deleted.');
    }
    wp_send_json_error('File not found.');
});

// ============================================================
// AJAX — Restore from stored backup
// ============================================================

add_action('wp_ajax_cs_restore_backup', function (): void {
    cs_verify_nonce();
    set_time_limit(0);
    ignore_user_abort(true);

    $file = sanitize_file_name($_POST['file'] ?? '');
    $path = CS_BACKUP_DIR . $file;

    if (!file_exists($path) || strpos(realpath($path), realpath(CS_BACKUP_DIR)) !== 0) {
        wp_send_json_error('Backup file not found.');
    }

    try {
        cs_maintenance_on();
        cs_restore_from_zip($path);
        cs_maintenance_off();
        wp_send_json_success('Database restored successfully. Maintenance mode disabled. Site is back online.');
    } catch (Exception $e) {
        cs_maintenance_off(); // Always remove maintenance mode even on failure
        wp_send_json_error('Restore failed: ' . $e->getMessage() . ' — Maintenance mode has been disabled.');
    }
});

// ============================================================
// AJAX — Restore from upload
// ============================================================

add_action('wp_ajax_cs_restore_upload', function (): void {
    cs_verify_nonce();
    set_time_limit(0);
    ignore_user_abort(true);

    if (empty($_FILES['backup_file'])) {
        wp_send_json_error('No file uploaded.');
    }

    $tmp = $_FILES['backup_file']['tmp_name'];
    $ext = strtolower(pathinfo($_FILES['backup_file']['name'], PATHINFO_EXTENSION));

    if (!in_array($ext, ['zip', 'sql'], true)) {
        wp_send_json_error('Only .zip or .sql files accepted.');
    }

    try {
        cs_maintenance_on();
        if ($ext === 'zip') {
            cs_restore_from_zip($tmp);
        } else {
            cs_restore_sql_file($tmp);
        }
        cs_maintenance_off();
        wp_send_json_success('Database restored. Maintenance mode disabled. Site is back online.');
    } catch (Exception $e) {
        cs_maintenance_off();
        wp_send_json_error('Restore failed: ' . $e->getMessage());
    }
});

// Schedule is now saved via plain HTML form POST (see page handler above)

// ============================================================
// AJAX — Save retention
// ============================================================

add_action('wp_ajax_cs_save_retention', function (): void {
    cs_verify_nonce();
    $r = max(1, min(9999, intval($_POST['retention'] ?? 8)));
    update_option('cs_retention', $r);
    wp_send_json_success('Retention set to ' . $r);
});

// ============================================================
// Download handler
// ============================================================

add_action('admin_post_cs_download', function (): void {
    check_admin_referer('cs_download');
    if (!current_user_can('manage_options')) wp_die('Forbidden');

    $file = sanitize_file_name($_GET['file'] ?? '');
    $path = CS_BACKUP_DIR . $file;

    if (!file_exists($path) || strpos(realpath($path), realpath(CS_BACKUP_DIR)) !== 0) {
        wp_die('File not found.');
    }

    header('Content-Type: application/zip');
    header('Content-Disposition: attachment; filename="' . $file . '"');
    header('Content-Length: ' . filesize($path));
    header('Cache-Control: no-cache');
    readfile($path);
    exit;
});

// ============================================================
// Maintenance mode
// ============================================================

function cs_maintenance_on(): void {
    $php = '<?php $upgrading = ' . time() . '; ?>';
    file_put_contents(CS_MAINT_FILE, $php);
}

function cs_maintenance_off(): void {
    if (file_exists(CS_MAINT_FILE)) {
        @unlink(CS_MAINT_FILE);
    }
}

// ============================================================
// Core — Create backup
// ============================================================

function cs_create_backup(
    bool $include_db, bool $include_media,
    bool $include_plugins  = false, bool $include_themes    = false,
    bool $include_mu       = false, bool $include_languages = false,
    bool $include_dropins  = false, bool $include_htaccess  = false,
    bool $include_wpconfig = false
): string {
    $timestamp = date('Y-m-d_H-i-s');

    // Build type slug
    $core_all = $include_db && $include_media && $include_plugins && $include_themes;
    $has_other = $include_mu || $include_languages || $include_dropins || $include_htaccess || $include_wpconfig;
    if ($core_all && !$has_other) {
        $type = 'full';
    } elseif ($core_all && $has_other) {
        $type = 'full-plus';
    } else {
        $parts = [];
        if ($include_db)        $parts[] = 'db';
        if ($include_media)     $parts[] = 'media';
        if ($include_plugins)   $parts[] = 'plugins';
        if ($include_themes)    $parts[] = 'themes';
        if ($include_mu)        $parts[] = 'mu';
        if ($include_languages) $parts[] = 'lang';
        if ($include_dropins)   $parts[] = 'dropins';
        if ($include_htaccess)  $parts[] = 'htaccess';
        if ($include_wpconfig)  $parts[] = 'wpconfig';
        $type = implode('-', $parts);
    }

    $filename = "backup_{$type}_{$timestamp}.zip";
    $zip_path = CS_BACKUP_DIR . $filename;

    $zip = new ZipArchive();
    if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
        throw new Exception('Cannot create zip at: ' . $zip_path);
    }

    if ($include_db) {
        $zip->addFromString('database.sql', cs_dump_database());
    }

    if ($include_media) {
        cs_add_dir_to_zip($zip, wp_upload_dir()['basedir'], 'uploads');
    }

    if ($include_plugins) {
        cs_add_dir_to_zip($zip, WP_PLUGIN_DIR, 'plugins');
    }

    if ($include_themes) {
        cs_add_dir_to_zip($zip, get_theme_root(), 'themes');
    }

    if ($include_mu) {
        $mu_path = defined('WPMU_PLUGIN_DIR') ? WPMU_PLUGIN_DIR : WP_CONTENT_DIR . '/mu-plugins';
        if (is_dir($mu_path)) cs_add_dir_to_zip($zip, $mu_path, 'mu-plugins');
    }

    if ($include_languages) {
        $lang_path = WP_CONTENT_DIR . '/languages';
        if (is_dir($lang_path)) cs_add_dir_to_zip($zip, $lang_path, 'languages');
    }

    if ($include_dropins) {
        foreach (glob(WP_CONTENT_DIR . '/*.php') ?: [] as $f) {
            $zip->addFile($f, 'dropins/' . basename($f));
        }
    }

    if ($include_htaccess) {
        $ht = ABSPATH . '.htaccess';
        if (file_exists($ht)) $zip->addFile($ht, '.htaccess');
    }

    if ($include_wpconfig) {
        $wpc = ABSPATH . 'wp-config.php';
        if (file_exists($wpc)) $zip->addFile($wpc, 'wp-config.php');
    }

    $zip->addFromString('backup-meta.json', json_encode([
        'plugin_version'    => CS_VERSION,
        'created'           => date('c'),
        'wp_version'        => get_bloginfo('version'),
        'site_url'          => get_site_url(),
        'table_prefix'      => $GLOBALS['wpdb']->prefix,
        'include_db'        => $include_db,
        'include_media'     => $include_media,
        'include_plugins'   => $include_plugins,
        'include_themes'    => $include_themes,
        'include_mu'        => $include_mu,
        'include_languages' => $include_languages,
        'include_dropins'   => $include_dropins,
        'include_htaccess'  => $include_htaccess,
        'include_wpconfig'  => $include_wpconfig,
    ], JSON_PRETTY_PRINT));

    $zip->close();
    return $filename;
}

// ============================================================
// Core — Database dump
// ============================================================

function cs_dump_database(): string {
    return cs_mysqldump_available() ? cs_dump_via_mysqldump() : cs_dump_via_php($GLOBALS['wpdb']);
}

function cs_mysqldump_available(): bool {
    exec('which mysqldump 2>/dev/null', $out, $rc);
    return $rc === 0;
}

function cs_dump_via_mysqldump(): string {
    [$host, $port] = cs_parse_db_host(DB_HOST);
    $tmp = tempnam(sys_get_temp_dir(), 'cs_dump_') . '.sql';

    $cmd = sprintf(
        'MYSQL_PWD=%s mysqldump --single-transaction --quick --lock-tables=false -h %s -P %s -u %s %s > %s 2>&1',
        escapeshellarg(DB_PASSWORD),
        escapeshellarg($host),
        escapeshellarg($port),
        escapeshellarg(DB_USER),
        escapeshellarg(DB_NAME),
        escapeshellarg($tmp)
    );

    exec($cmd, $output, $rc);

    if ($rc !== 0 || !file_exists($tmp) || filesize($tmp) < 100) {
        @unlink($tmp);
        return cs_dump_via_php($GLOBALS['wpdb']);
    }

    $sql = file_get_contents($tmp);
    unlink($tmp);
    return $sql;
}

function cs_dump_via_php(\wpdb $wpdb): string {
    $out   = [];
    $out[] = '-- CloudScale Free Backup v' . CS_VERSION;
    $out[] = '-- Generated: ' . date('Y-m-d H:i:s');
    $out[] = '-- Site: ' . get_site_url();
    $out[] = '-- Database: ' . DB_NAME;
    $out[] = '';
    $out[] = 'SET FOREIGN_KEY_CHECKS=0;';
    $out[] = "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO';";
    $out[] = 'SET NAMES utf8mb4;';
    $out[] = '';

    $tables = $wpdb->get_col('SHOW TABLES');

    foreach ($tables as $table) {
        $create  = $wpdb->get_row("SHOW CREATE TABLE `{$table}`", ARRAY_N);
        $out[]   = "DROP TABLE IF EXISTS `{$table}`;";
        $out[]   = $create[1] . ';';
        $out[]   = '';

        $total   = (int) $wpdb->get_var("SELECT COUNT(*) FROM `{$table}`");
        $columns = $wpdb->get_col("DESCRIBE `{$table}`");
        $chunk   = 500;
        $offset  = 0;

        while ($offset < $total) {
            $rows = $wpdb->get_results(
                $wpdb->prepare("SELECT * FROM `{$table}` LIMIT %d OFFSET %d", $chunk, $offset),
                ARRAY_N
            );
            if (empty($rows)) break;

            $col_list = '`' . implode('`, `', $columns) . '`';
            $vals     = [];
            foreach ($rows as $row) {
                $escaped = array_map(fn($v) => $v === null ? 'NULL' : "'" . esc_sql($v) . "'", $row);
                $vals[]  = '(' . implode(', ', $escaped) . ')';
            }

            $out[] = "INSERT INTO `{$table}` ({$col_list}) VALUES";
            $out[] = implode(",\n", $vals) . ';';
            $out[] = '';
            $offset += $chunk;
        }

        $out[] = '';
    }

    $out[] = 'SET FOREIGN_KEY_CHECKS=1;';
    return implode("\n", $out);
}

function cs_add_dir_to_zip(ZipArchive $zip, string $dir, string $prefix): void {
    if (!is_dir($dir)) return;
    $it = new RecursiveIteratorIterator(
        new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
        RecursiveIteratorIterator::SELF_FIRST
    );
    foreach ($it as $file) {
        if ($file->isFile()) {
            $local = $prefix . '/' . ltrim(str_replace($dir, '', $file->getPathname()), '/\\');
            $zip->addFile($file->getPathname(), $local);
        }
    }
}

// ============================================================
// Core — Restore
// ============================================================

function cs_restore_from_zip(string $zip_path): void {
    $zip = new ZipArchive();
    if ($zip->open($zip_path) !== true) {
        throw new Exception('Cannot open zip file.');
    }

    $idx = $zip->locateName('database.sql');
    if ($idx === false) {
        $zip->close();
        throw new Exception('No database.sql in this backup. Is it a DB or Full backup?');
    }

    $sql = $zip->getFromIndex($idx);
    $zip->close();

    if (empty($sql)) {
        throw new Exception('database.sql is empty.');
    }

    cs_execute_sql_string($sql);
}

function cs_restore_sql_file(string $path): void {
    $sql = file_get_contents($path);
    if (empty($sql)) {
        throw new Exception('SQL file is empty.');
    }
    cs_execute_sql_string($sql);
}

function cs_execute_sql_string(string $sql): void {
    global $wpdb;

    if (cs_mysql_cli_available()) {
        cs_restore_via_mysql_cli($sql);
        return;
    }

    $wpdb->query('SET FOREIGN_KEY_CHECKS=0');

    foreach (cs_split_sql($sql) as $stmt) {
        $stmt = trim($stmt);
        if (empty($stmt) || str_starts_with($stmt, '--') || str_starts_with($stmt, '/*')) {
            continue;
        }
        $wpdb->query($stmt);
        if ($wpdb->last_error) {
            error_log('CS Restore statement error: ' . $wpdb->last_error . ' | ' . substr($stmt, 0, 200));
        }
    }

    $wpdb->query('SET FOREIGN_KEY_CHECKS=1');
}

function cs_mysql_cli_available(): bool {
    exec('which mysql 2>/dev/null', $out, $rc);
    return $rc === 0;
}

function cs_restore_via_mysql_cli(string $sql): void {
    $tmp = tempnam(sys_get_temp_dir(), 'cs_restore_') . '.sql';
    file_put_contents($tmp, $sql);

    [$host, $port] = cs_parse_db_host(DB_HOST);

    $cmd = sprintf(
        'MYSQL_PWD=%s mysql -h %s -P %s -u %s %s < %s 2>&1',
        escapeshellarg(DB_PASSWORD),
        escapeshellarg($host),
        escapeshellarg($port),
        escapeshellarg(DB_USER),
        escapeshellarg(DB_NAME),
        escapeshellarg($tmp)
    );

    exec($cmd, $output, $rc);
    unlink($tmp);

    if ($rc !== 0) {
        throw new Exception('mysql CLI restore failed: ' . implode(' | ', $output));
    }
}

function cs_split_sql(string $sql): array {
    $stmts      = [];
    $current    = '';
    $in_string  = false;
    $str_char   = '';
    $len        = strlen($sql);

    for ($i = 0; $i < $len; $i++) {
        $c = $sql[$i];
        if ($in_string) {
            $current .= $c;
            if ($c === '\\') {
                $current .= $sql[++$i] ?? '';
            } elseif ($c === $str_char) {
                $in_string = false;
            }
        } elseif ($c === "'" || $c === '"' || $c === '`') {
            $in_string = true;
            $str_char  = $c;
            $current  .= $c;
        } elseif ($c === ';') {
            $t = trim($current);
            if ($t !== '') $stmts[] = $t;
            $current = '';
        } else {
            $current .= $c;
        }
    }

    $t = trim($current);
    if ($t !== '') $stmts[] = $t;

    return $stmts;
}

// ============================================================
// Utilities
// ============================================================

function cs_parse_db_host(string $host): array {
    $port = '3306';
    if (str_contains($host, ':')) {
        [$host, $port] = explode(':', $host, 2);
    }
    return [$host, $port];
}

function cs_list_backups(): array {
    $backups = [];
    if (!is_dir(CS_BACKUP_DIR)) return $backups;

    $files = glob(CS_BACKUP_DIR . 'backup_*.zip') ?: [];

    foreach ($files as $file) {
        $name = basename($file);
        // Extract type slug between first and second underscore after "backup_"
        // e.g. backup_full_..., backup_db_..., backup_db-media-plugins-themes_...
        if (preg_match('/^backup_([^_]+(?:_[^_]+)?)_\d{4}-\d{2}-\d{2}/', $name, $m)) {
            $slug = $m[1];
            $type = match($slug) {
                'full'                        => 'Full',
                'db'                          => 'DB only',
                'media'                       => 'Media only',
                'plugins'                     => 'Plugins only',
                'themes'                      => 'Themes only',
                'db-media'                    => 'DB + Media',
                'db-plugins'                  => 'DB + Plugins',
                'db-themes'                   => 'DB + Themes',
                'db-media-plugins'            => 'DB + Media + Plugins',
                'db-media-themes'             => 'DB + Media + Themes',
                'db-plugins-themes'           => 'DB + Plugins + Themes',
                'media-plugins'               => 'Media + Plugins',
                'media-themes'               => 'Media + Themes',
                'media-plugins-themes'        => 'Media + Plugins + Themes',
                'plugins-themes'              => 'Plugins + Themes',
                'db-media-plugins-themes'     => 'Full',
                default                       => ucfirst(str_replace('-', ' + ', $slug)),
            };
        } else {
            $type = 'Unknown';
        }
        $backups[] = [
            'name'  => $name,
            'type'  => $type,
            'size'  => filesize($file),
            'date'  => date('Y-m-d H:i', filemtime($file)),
            'mtime' => filemtime($file),
        ];
    }

    usort($backups, fn($a, $b) => $b['mtime'] - $a['mtime']);
    return $backups;
}

function cs_enforce_retention(): void {
    $keep    = intval(get_option('cs_retention', 8));
    $backups = cs_list_backups();
    foreach (array_slice($backups, $keep) as $b) {
        @unlink(CS_BACKUP_DIR . $b['name']);
    }
}

function cs_dir_size(string $dir): int {
    $size = 0;
    if (!is_dir($dir)) return 0;
    foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS)) as $f) {
        $size += $f->getSize();
    }
    return $size;
}

function cs_format_size(int $bytes): string {
    if ($bytes >= 1073741824) return round($bytes / 1073741824, 2) . ' GB';
    if ($bytes >= 1048576)    return round($bytes / 1048576, 2) . ' MB';
    if ($bytes >= 1024)       return round($bytes / 1024, 2) . ' KB';
    return $bytes . ' B';
}

function cs_human_age(int $timestamp): string {
    $diff = time() - $timestamp;
    if ($diff < 3600)   return round($diff / 60) . ' min ago';
    if ($diff < 86400)  return round($diff / 3600) . ' hr ago';
    if ($diff < 604800) return round($diff / 86400) . ' days ago';
    return date('j M Y', $timestamp);
}

function cs_verify_nonce(): void {
    if (!current_user_can('manage_options')) {
        wp_send_json_error('Insufficient permissions.');
    }
    if (!check_ajax_referer('cs_nonce', 'nonce', false)) {
        wp_send_json_error('Security check failed.');
    }
}
