/* CloudScale Cleanup — Admin JS
 * Chunked processing engine: start → loop chunks → finish
 * Each operation sends small AJAX requests until done.
 */
(function ($) {
    'use strict';

    $(document).ready(function () {

    // ── Tab switching ──────────────────────────────────────────────────────────

    $('.csc-tab').on('click', function () {
        var target = $(this).data('tab');
        $('.csc-tab').removeClass('active');
        $('.csc-tab-content').removeClass('active');
        $(this).addClass('active');
        $('#tab-' + target).addClass('active');
    });

    // ── Terminal helpers ───────────────────────────────────────────────────────

    function clearTerminal(id) {
        $('#' + id).empty();
    }

    // Parse "[OPTIMISE] ID 1041 — name.ext (87 KB) — flags"
    // or    "[UNUSED]   ID 1041 — name.ext (87 KB)"
    // into column cells for a cleaner table layout.
    function parseTableLine(text) {
        var m = text.match(/^\s*\[(OPTIMISE|UNUSED)\]\s+ID\s+(\d+)\s+—\s+(.+?)\s+\(([^)]+)\)(?:\s+—\s+(.+))?$/);
        if (!m) { return null; }
        return { tag: m[1], id: m[2], name: m[3], size: m[4], flags: m[5] || '' };
    }

    function appendLine(id, line) {
        var $t  = $('#' + id);
        var cls = 'csc-log-item';
        switch (line.type) {
            case 'section':  cls = 'csc-log-section';  break;
            case 'deleted':  cls = 'csc-log-deleted';  break;
            case 'count':    cls = 'csc-log-count';    break;
            case 'success':  cls = 'csc-log-success';  break;
            case 'error':    cls = 'csc-log-error';    break;
            case 'info':     cls = 'csc-log-info';     break;
        }

        // Attempt columnar render for scan item lines
        if (line.type === 'item') {
            var parsed = parseTableLine(line.text);
            if (parsed) {
                // Ensure a table container exists
                if (!$t.find('.csc-log-table').length) {
                    $t.append('<div class="csc-log-table"></div>');
                }
                var tagColor = parsed.tag === 'OPTIMISE' ? '#ff7b72' : '#79c0ff';
                var html = '<div class="csc-log-row">'
                    + '<span class="csc-log-cell csc-log-cell-id" style="color:#6b7690">ID ' + esc(parsed.id) + '</span>'
                    + '<span class="csc-log-cell csc-log-cell-name" title="' + esc(parsed.name) + '">' + esc(parsed.name) + '</span>'
                    + '<span class="csc-log-cell csc-log-cell-size">' + esc(parsed.size) + '</span>'
                    + (parsed.flags ? '<span class="csc-log-cell csc-log-cell-flags" style="color:' + tagColor + '">' + esc(parsed.flags) + '</span>' : '')
                    + '</div>';
                $t.find('.csc-log-table').append(html);
                $t[0].scrollTop = $t[0].scrollHeight;
                return;
            }
        }

        // For non-table lines, close any open table first then write plain span
        $t.append('<span class="' + cls + '">' + esc(line.text) + '\n</span>');
        $t[0].scrollTop = $t[0].scrollHeight;
    }

    function appendLines(termId, lines) {
        if (!lines || !lines.length) return;
        $.each(lines, function (i, line) { appendLine(termId, line); });
    }

    function renderLines(termId, lines) {
        clearTerminal(termId);
        appendLines(termId, lines);
    }

    function esc(str) {
        return String(str)
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;');
    }

    // ── Progress helpers ───────────────────────────────────────────────────────

    function showProgress(outerSelector, fillId, labelId, labelText) {
        $(outerSelector).show();
        $('#' + fillId).css('width', '0%');
        $('#' + labelId).text(labelText || 'Working…');
    }

    function updateProgress(fillId, labelId, done, total, extraLabel) {
        var pct = total > 0 ? Math.round((done / total) * 100) : 0;
        $('#' + fillId).css('width', pct + '%');
        var label = pct + '% — ' + done + ' / ' + total + ' processed';
        if (extraLabel) { label += ' · ' + extraLabel; }
        $('#' + labelId).text(label);
    }

    function finishProgress(outerSelector, fillId, labelId, finalLabel) {
        $('#' + fillId).css('width', '100%');
        $('#' + labelId).text(finalLabel || 'Complete.');
        setTimeout(function () { $(outerSelector).fadeOut(600); }, 1800);
    }

    // ── Settings collection ────────────────────────────────────────────────────

    function collectSettings() {
        var data = {};
        $('.csc-setting, .csc-small-num').each(function () {
            data[$(this).attr('name')] = $(this).val();
        });
        // Hidden inputs backing the pure-div toggles
        $('input[type="hidden"][data-csc-toggle]').each(function () {
            data[$(this).attr('name')] = $(this).val();
        });
        $('input[type=checkbox][name^=csc_]').not('[name$="[]"]').each(function () {
            data[$(this).attr('name')] = $(this).is(':checked') ? '1' : '0';
        });
        $('input[type=checkbox][name$="[]"]:checked').each(function () {
            var name = $(this).attr('name');
            if (!data[name]) { data[name] = []; }
            data[name].push($(this).val());
        });
        return data;
    }

    // ── Save settings ──────────────────────────────────────────────────────────

    $('.csc-save-btn').on('click', function () {
        var $btn = $(this);
        var origLabel = $btn.text();
        $btn.prop('disabled', true).text('Saving…');
        var payload = collectSettings();
        payload.action = 'csc_save_settings';
        payload.nonce  = CSC.nonce;
        $.post(CSC.ajax_url, payload, function (resp) {
            $btn.prop('disabled', false).text(origLabel);
            notify(resp.success ? 'Settings saved.' : 'Error: ' + (resp.data || 'Unknown'), !resp.success);
        }).fail(function () {
            $btn.prop('disabled', false).text(origLabel);
            notify('Network error.', true);
        });
    });

    function notify(msg, isError) {
        var $n = $('#csc-save-notice');
        $n.text(msg).css('background', isError ? '#e74c3c' : '#27ae60');
        $n.fadeIn(200).delay(2200).fadeOut(400);
    }

    // ═════════════════════════════════════════════════════════════════════════
    // CHUNKED ENGINE
    // ─────────────────────────────────────────────────────────────────────────
    // runChunked({ startAction, chunkAction, finishAction,
    //              termId, progressOuter, progressFill, progressLabel,
    //              confirmMsg, $btn, restoreLabel })
    // ═════════════════════════════════════════════════════════════════════════

    function runChunked(opts) {
        if (opts.confirmMsg && !confirm(opts.confirmMsg)) { return; }

        var total = 0;
        var done  = 0;
        opts.$btn.prop('disabled', true).html('⏳ Starting…');
        clearTerminal(opts.termId);
        appendLine(opts.termId, { type: 'section', text: '=== ' + opts.startLabel + ' ===' });

        showProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Building queue…');

        // Step 1: start
        var startData = { action: opts.startAction, nonce: CSC.nonce };
        if (opts.toggleData) {
            $.extend(startData, opts.toggleData);
        }
        $.post(CSC.ajax_url, startData, function (resp) {
            if (!resp.success) {
                appendLine(opts.termId, { type: 'error', text: 'Error: ' + (resp.data || 'Unknown error') });
                opts.$btn.prop('disabled', false).html(opts.restoreLabel);
                finishProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Failed.');
                return;
            }

            appendLines(opts.termId, resp.data.lines);
            total = resp.data.total;
            done  = total - resp.data.remaining;
            updateProgress(opts.progressFill, opts.progressLabel, done, total);

            if (total === 0) {
                appendLine(opts.termId, { type: 'info', text: '  Nothing to process.' });
                opts.$btn.prop('disabled', false).html(opts.restoreLabel);
                finishProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Nothing to do.');
                return;
            }

            opts.$btn.html('⏳ Processing ' + total + ' items…');
            processNextChunk();
        }).fail(networkError);

        // Step 2: loop
        function processNextChunk() {
            var chunkPayload = { action: opts.chunkAction, nonce: CSC.nonce };
            if (opts.chunkExtra) { $.extend(chunkPayload, opts.chunkExtra); }

            $.post(CSC.ajax_url, chunkPayload, function (resp) {
                if (!resp.success) {
                    appendLine(opts.termId, { type: 'error', text: 'Error: ' + (resp.data || 'Unknown error') });
                    opts.$btn.prop('disabled', false).html(opts.restoreLabel);
                    finishProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Error — see log.');
                    return;
                }

                appendLines(opts.termId, resp.data.lines);
                var remaining = resp.data.remaining;
                done = total - remaining;
                var extra = opts.progressExtra ? opts.progressExtra(resp.data) : null;
                updateProgress(opts.progressFill, opts.progressLabel, done, total, extra);

                if (remaining > 0) {
                    // Small delay to keep the browser responsive and let the
                    // terminal update paint before firing the next request.
                    setTimeout(processNextChunk, 150);
                } else {
                    finish();
                }
            }).fail(networkError);
        }

        // Step 3: finish
        function finish() {
            $.post(CSC.ajax_url, { action: opts.finishAction, nonce: CSC.nonce }, function (resp) {
                if (resp.success) { appendLines(opts.termId, resp.data.lines); }
                opts.$btn.prop('disabled', false).html(opts.restoreLabel);
                finishProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Complete.');
            }).fail(networkError);
        }

        function networkError() {
            appendLine(opts.termId, { type: 'error', text: '  Network error. Check your connection and try again.' });
            opts.$btn.prop('disabled', false).html(opts.restoreLabel);
            finishProgress('#' + opts.progressOuter, opts.progressFill, opts.progressLabel, 'Network error.');
        }
    }

    // ═════════════════════════════════════════════════════════════════════════
    // DATABASE CLEANUP
    // ═════════════════════════════════════════════════════════════════════════

    $('#btn-scan-db').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Scanning…');
        clearTerminal('db-terminal');
        appendLine('db-terminal', { type: 'section', text: '=== DRY RUN — Database Scan ===' });

        // Collect toggle states from track divs
        var postData = { action: 'csc_scan_db', nonce: CSC.nonce };
        var tracked = 0;
        document.querySelectorAll('[data-csc-toggle-track]').forEach(function (track) {
            // Walk up to find the option row, then find the hidden input anywhere inside it
            var row = track.closest('.csc-option-row') || track.parentNode;
            var inputs = row.querySelectorAll('input[type="hidden"]');
            inputs.forEach(function (input) {
                if (input.name && input.name.indexOf('csc_clean_') === 0) {
                    postData[input.name] = track.getAttribute('data-on') === '1' ? '1' : '0';
                    tracked++;
                }
            });
        });

        $.post(CSC.ajax_url, postData, function (resp) {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview');
            if (resp.success) {
                appendLines('db-terminal', resp.data);
                appendLine('db-terminal', { type: 'info', text: '\nDry run complete. No changes have been made.' });
            } else {
                appendLine('db-terminal', { type: 'error', text: 'Server error: ' + (resp.data || 'Unknown — check PHP error log') });
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview');
            appendLine('db-terminal', { type: 'error', text: 'AJAX failed: ' + textStatus + ' — ' + errorThrown + ' (HTTP ' + jqXHR.status + ')' });
            appendLine('db-terminal', { type: 'error', text: 'Response: ' + jqXHR.responseText.substring(0, 200) });
        });
    });

    $('#btn-run-db').on('click', function () {
        var toggleData = {};
        document.querySelectorAll('[data-csc-toggle-track]').forEach(function (track) {
            var row = track.closest('.csc-option-row') || track.parentNode;
            row.querySelectorAll('input[type="hidden"]').forEach(function (input) {
                if (input.name && input.name.indexOf('csc_clean_') === 0) {
                    toggleData[input.name] = track.getAttribute('data-on') === '1' ? '1' : '0';
                }
            });
        });
        runChunked({
            startAction:   'csc_db_start',
            chunkAction:   'csc_db_chunk',
            finishAction:  'csc_db_finish',
            startLabel:    'DATABASE CLEANUP RUNNING',
            termId:        'db-terminal',
            progressOuter: 'db-progress-outer',
            progressFill:  'db-progress-fill',
            progressLabel: 'db-progress-label',
            confirmMsg:    'This will permanently delete the items shown in the dry run. Proceed?',
            $btn:          $(this),
            restoreLabel:  '🗑 Run Cleanup Now',
            toggleData:    toggleData,
        });
    });

    // ═════════════════════════════════════════════════════════════════════════
    // IMAGE CLEANUP
    // ═════════════════════════════════════════════════════════════════════════

    $('#btn-scan-img').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Scanning…');
        clearTerminal('img-terminal');
        appendLine('img-terminal', { type: 'section', text: '=== DRY RUN — Unused Image Scan ===' });

        $.post(CSC.ajax_url, { action: 'csc_scan_images', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview');
            if (resp.success) {
                appendLines('img-terminal', resp.data);
                appendLine('img-terminal', { type: 'info', text: '\nDry run complete. No files deleted.' });
            } else {
                appendLine('img-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function () {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview');
            appendLine('img-terminal', { type: 'error', text: 'Network error.' });
        });
    });

    $('#btn-run-img').on('click', function () {
        runChunked({
            startAction:   'csc_img_start',
            chunkAction:   'csc_img_chunk',
            finishAction:  'csc_img_finish',
            startLabel:    'DELETING UNUSED IMAGES',
            termId:        'img-terminal',
            progressOuter: 'img-progress-outer',
            progressFill:  'img-progress-fill',
            progressLabel: 'img-progress-label',
            confirmMsg:    'This will permanently delete unused media attachments. This cannot be undone. Proceed?',
            $btn:          $(this),
            restoreLabel:  '🗑 Delete Unused Images',
        });
    });

    // ── Orphan files recycle workflow ─────────────────────────────────────────

    function orphanUpdateRecycleBin( count ) {
        $('#orphan-recycle-count').text( count > 0 ? '— ' + count + ' file(s) in recycle bin' : '— recycle bin is empty' );
    }

    // Check recycle bin status on page load
    $.post( CSC.ajax_url, { action: 'csc_recycle_status', nonce: CSC.nonce }, function( resp ) {
        orphanUpdateRecycleBin( resp.success ? resp.data.recycle : 0 );
    });

    // Scan
    $('#btn-scan-orphan').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Scanning…');
        clearTerminal('img-terminal');
        appendLine('img-terminal', { type: 'section', text: '=== ORPHANED FILESYSTEM FILE SCAN ===' });

        $.post(CSC.ajax_url, { action: 'csc_scan_orphan_files', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('🔍 Scan Orphan Files');
            if (resp.success) {
                appendLines('img-terminal', resp.data.lines);
                orphanUpdateRecycleBin( resp.data.recycle );
            } else {
                appendLine('img-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            $btn.prop('disabled', false).html('🔍 Scan Orphan Files');
            appendLine('img-terminal', { type: 'error', text: 'AJAX failed: ' + textStatus + ' — ' + errorThrown + ' (HTTP ' + jqXHR.status + ')' });
        });
    });

    // Move to recycle
    $('#btn-recycle-orphan').on('click', function () {
        if ( !confirm('Move all orphaned files to the recycle bin? You can restore or permanently delete them afterwards.') ) { return; }
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Moving…');
        clearTerminal('img-terminal');

        $.post(CSC.ajax_url, { action: 'csc_recycle_orphan_files', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('♻️ Move to Recycle');
            if (resp.success) {
                appendLines('img-terminal', resp.data.lines);
                orphanUpdateRecycleBin( resp.data.recycle );
            } else {
                appendLine('img-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            $btn.prop('disabled', false).html('♻️ Move to Recycle');
            appendLine('img-terminal', { type: 'error', text: 'AJAX failed: ' + textStatus + ' — ' + errorThrown + ' (HTTP ' + jqXHR.status + ')' });
        });
    });

    // Restore
    $('#btn-restore-orphan').on('click', function () {
        if ( !confirm('Restore all files from the recycle bin to their original locations?') ) { return; }
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Restoring…');
        clearTerminal('img-terminal');

        $.post(CSC.ajax_url, { action: 'csc_restore_orphan_files', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('↩️ Restore All');
            if (resp.success) {
                appendLines('img-terminal', resp.data.lines);
                orphanUpdateRecycleBin( resp.data.recycle );
            } else {
                appendLine('img-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            $btn.prop('disabled', false).html('↩️ Restore All');
            appendLine('img-terminal', { type: 'error', text: 'AJAX failed: ' + textStatus + ' — ' + errorThrown + ' (HTTP ' + jqXHR.status + ')' });
        });
    });

    // Permanently delete
    $('#btn-purge-orphan').on('click', function () {
        if ( !confirm('PERMANENTLY DELETE all files in the recycle bin? This cannot be undone.') ) { return; }
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Deleting…');
        clearTerminal('img-terminal');

        $.post(CSC.ajax_url, { action: 'csc_purge_orphan_files', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('🗑 Permanently Delete');
            if (resp.success) {
                appendLines('img-terminal', resp.data.lines);
                orphanUpdateRecycleBin( resp.data.recycle );
            } else {
                appendLine('img-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function (jqXHR, textStatus, errorThrown) {
            $btn.prop('disabled', false).html('🗑 Permanently Delete');
            appendLine('img-terminal', { type: 'error', text: 'AJAX failed: ' + textStatus + ' — ' + errorThrown + ' (HTTP ' + jqXHR.status + ')' });
        });
    });

    // ═════════════════════════════════════════════════════════════════════════
    // IMAGE OPTIMISATION
    // ═════════════════════════════════════════════════════════════════════════

    $('#btn-scan-optimise').on('click', function () {
        var $btn = $(this);
        $btn.prop('disabled', true).html('⏳ Scanning…');
        clearTerminal('optimise-terminal');
        appendLine('optimise-terminal', { type: 'section', text: '=== DRY RUN — Image Optimisation Preview ===' });

        $.post(CSC.ajax_url, { action: 'csc_scan_optimise', nonce: CSC.nonce }, function (resp) {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview Savings');
            if (resp.success) {
                appendLines('optimise-terminal', resp.data);
                appendLine('optimise-terminal', { type: 'info', text: '\nDry run complete. No files modified.' });
            } else {
                appendLine('optimise-terminal', { type: 'error', text: 'Error: ' + (resp.data || 'Unknown') });
            }
        }).fail(function () {
            $btn.prop('disabled', false).html('🔍 Dry Run — Preview Savings');
            appendLine('optimise-terminal', { type: 'error', text: 'Network error.' });
        });
    });

    $('#btn-run-optimise').on('click', function () {
        runChunked({
            startAction:   'csc_optimise_start',
            chunkAction:   'csc_optimise_chunk',
            finishAction:  'csc_optimise_finish',
            startLabel:    'IMAGE OPTIMISATION RUNNING',
            termId:        'optimise-terminal',
            progressOuter: 'opt-progress-outer',
            progressFill:  'opt-progress-fill',
            progressLabel: 'opt-progress-label',
            confirmMsg:    'This will modify original image files on disk. Take a backup first. Proceed?',
            $btn:          $(this),
            restoreLabel:  '⚡ Optimise Images Now',
            // Show running disk savings in the progress label
            progressExtra: function (data) {
                if (data.total_saved && data.total_saved > 0) {
                    return formatBytes(data.total_saved) + ' saved so far';
                }
                return null;
            },
        });
    });

    // ── Byte formatter (mirrors PHP size_format) ───────────────────────────────

    function formatBytes(bytes) {
        if (bytes < 1024)           { return bytes + ' B'; }
        if (bytes < 1048576)        { return (bytes / 1024).toFixed(1) + ' KB'; }
        if (bytes < 1073741824)     { return (bytes / 1048576).toFixed(1) + ' MB'; }
        return (bytes / 1073741824).toFixed(2) + ' GB';
    }



// ═════════════════════════════════════════════════════════════════════════════
// EXPLAIN MODALS
// ═════════════════════════════════════════════════════════════════════════════

    // Force explain button styles — WordPress admin CSS overrides inline styles
    document.querySelectorAll('[id^="csc-explain-btn-"]').forEach(function(btn) {
        var header = btn.parentElement;
        header.style.setProperty('display', 'flex', 'important');
        header.style.setProperty('align-items', 'center', 'important');
        header.style.setProperty('justify-content', 'space-between', 'important');
        btn.style.setProperty('margin-left', 'auto', 'important');
        btn.style.setProperty('flex-shrink', '0', 'important');
        // Restore color from data attribute set by PHP
        var color = btn.getAttribute('data-color');
        if (color) {
            btn.style.setProperty('background', color, 'important');
        }
    });

    // Force-hide native checkboxes inside slider labels
    // WordPress admin CSS can override our hidden-checkbox styles, so we apply
    // inline styles directly via JS after DOM ready — these always win.
    function initSliders() {
        document.querySelectorAll('.csc-slider-label input[type="checkbox"]').forEach(function (cb) {
            cb.style.cssText = [
                'position:absolute',
                'opacity:0',
                'width:1px',
                'height:1px',
                'margin:-1px',
                'padding:0',
                'border:0',
                'overflow:hidden',
                'clip:rect(0,0,0,0)',
                'white-space:nowrap',
                'pointer-events:none',
                '-webkit-appearance:none',
                'appearance:none',
                'z-index:-1'
            ].join('!important;') + '!important';
        });
    }
    initSliders();

    }); // document.ready

}(jQuery));
