<?php
/**
 * Plugin Name: CloudScale Cleanup
 * Plugin URI:  https://andrewbaker.ninja
 * Description: Database and media library cleanup with dry-run preview, image optimisation, and chunked processing safe on any server. Free, open source, no subscriptions.
 * Version:     1.7.0
 * Author:      Andrew Baker
 * Author URI:  https://andrewbaker.ninja
 * License:     GPL-2.0+
 * Text Domain: cloudscale-cleanup
 * Requires at least: 5.8
 * Requires PHP:      7.4
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'CLOUDSCALE_CLEANUP_VERSION', '1.7.0' );
define( 'CLOUDSCALE_CLEANUP_DIR', plugin_dir_path( __FILE__ ) );
define( 'CLOUDSCALE_CLEANUP_URL', plugin_dir_url( __FILE__ ) );
define( 'CLOUDSCALE_CLEANUP_SLUG', 'cloudscale-cleanup' );

// Clear opcode cache on activation so updated files take effect immediately
register_activation_hook( __FILE__, function() {
    if ( function_exists( 'opcache_reset' ) ) {
        opcache_reset();
    }
} );

// Also clear on every admin page load if version changed
add_action( 'admin_init', function() {
    $cached_version = get_option( 'csc_loaded_version', '' );
    if ( $cached_version !== CLOUDSCALE_CLEANUP_VERSION ) {
        if ( function_exists( 'opcache_reset' ) ) {
            opcache_reset();
        }
        update_option( 'csc_loaded_version', CLOUDSCALE_CLEANUP_VERSION );
    }
} );

/*
 * CHUNKED PROCESSING ARCHITECTURE
 * ─────────────────────────────────────────────────────────────────────────────
 * Every "run" operation works in three AJAX steps:
 *
 *   Step 1  csc_*_start   — Build the full list of IDs to process, store in a
 *                           transient, return the total count to JS.
 *
 *   Step 2  csc_*_chunk   — Pull the transient, process one small batch, update
 *                           the transient with the remaining IDs, return log
 *                           lines + remaining count. JS fires repeatedly until
 *                           remaining === 0.
 *
 *   Step 3  csc_*_finish  — Clean up the transient, write the last-run
 *                           timestamp, return a summary line.
 *
 * Each AJAX request completes in well under 30 seconds on any shared host.
 * Chunk sizes: 50 DB items · 25 image deletions · 5 image optimisations.
 */

define( 'CSC_CHUNK_DB',       50 );
define( 'CSC_CHUNK_IMAGES',   25 );
define( 'CSC_CHUNK_OPTIMISE',  5 );

// ─── Admin menu ──────────────────────────────────────────────────────────────

add_action( 'admin_menu', 'csc_add_menu' );
function csc_add_menu() {
    add_management_page(
        'CloudScale Cleanup',
        'CloudScale Cleanup',
        'manage_options',
        CLOUDSCALE_CLEANUP_SLUG,
        'csc_render_page'
    );
}

// ─── Enqueue assets ──────────────────────────────────────────────────────────

add_action( 'admin_enqueue_scripts', 'csc_enqueue_assets' );
function csc_enqueue_assets( $hook ) {
    if ( $hook !== 'tools_page_cloudscale-cleanup' ) {
        return;
    }
    wp_enqueue_style(
        'cloudscale-cleanup-css',
        CLOUDSCALE_CLEANUP_URL . 'assets/admin-v5.css',
        array(),
        CLOUDSCALE_CLEANUP_VERSION
    );
    wp_enqueue_script(
        'cloudscale-cleanup-js',
        CLOUDSCALE_CLEANUP_URL . 'assets/admin-v5.js',
        array( 'jquery' ),
        CLOUDSCALE_CLEANUP_VERSION,
        true
    );
    wp_localize_script( 'cloudscale-cleanup-js', 'CSC', array(
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'nonce'    => wp_create_nonce( 'csc_nonce' ),
    ) );
}

// ─── Admin dashboard widget ───────────────────────────────────────────────────

add_action( 'wp_dashboard_setup', 'csc_register_dashboard_widget' );
function csc_register_dashboard_widget() {
    wp_add_dashboard_widget(
        'csc_dashboard_widget',
        '🥷 AndrewBaker.Ninja CloudScale Cleanup',
        'csc_render_dashboard_widget'
    );
}

function csc_render_dashboard_widget() {
    $last_db  = get_option( 'csc_last_db_cleanup', null );
    $last_img = get_option( 'csc_last_img_cleanup', null );
    $last_opt = get_option( 'csc_last_img_optimise', null );

    $fmt = function ( $val ) {
        return $val
            ? '<span style="font-size:12px;font-weight:700;color:#fff">' . esc_html( human_time_diff( strtotime( $val ), time() ) . ' ago' ) . '</span>'
            : '<span style="font-size:12px;font-weight:700;color:rgba(255,255,255,0.5)">Never</span>';
    };
    ?>
    <div style="padding:4px 0 8px">
        <p style="margin:0 0 14px;font-size:13px;color:#50575e;line-height:1.5">
            CloudScale Cleanup is keeping your database and media library lean —
            revisions, transients, unused images, and orphaned files all handled.
        </p>

        <div style="display:grid;grid-template-columns:repeat(3,1fr);gap:8px;margin-bottom:16px">
            <div style="background:linear-gradient(135deg,#1565c0 0%,#1976d2 100%);border-radius:8px;padding:10px 8px;text-align:center;box-shadow:0 2px 6px rgba(21,101,192,0.35)">
                <div style="font-size:9px;font-weight:700;text-transform:uppercase;letter-spacing:0.5px;color:rgba(255,255,255,0.7);margin-bottom:5px">⚡ DB Cleanup</div>
                <?php echo $fmt( $last_db ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
            </div>
            <div style="background:linear-gradient(135deg,#4527a0 0%,#5e35b1 100%);border-radius:8px;padding:10px 8px;text-align:center;box-shadow:0 2px 6px rgba(69,39,160,0.35)">
                <div style="font-size:9px;font-weight:700;text-transform:uppercase;letter-spacing:0.5px;color:rgba(255,255,255,0.7);margin-bottom:5px">🖼 Images</div>
                <?php echo $fmt( $last_img ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
            </div>
            <div style="background:linear-gradient(135deg,#00695c 0%,#00897b 100%);border-radius:8px;padding:10px 8px;text-align:center;box-shadow:0 2px 6px rgba(0,105,92,0.35)">
                <div style="font-size:9px;font-weight:700;text-transform:uppercase;letter-spacing:0.5px;color:rgba(255,255,255,0.7);margin-bottom:5px">✨ Optimise</div>
                <?php echo $fmt( $last_opt ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
            </div>
        </div>

        <div style="display:flex;flex-direction:column;gap:10px">
            <a href="https://andrewbaker.ninja" target="_blank" rel="noopener"
               style="display:flex;align-items:center;justify-content:center;gap:8px;background:linear-gradient(135deg,#f953c6 0%,#b91d73 40%,#4f46e5 100%);color:#fff;font-weight:700;font-size:13px;padding:10px 16px;border-radius:8px;text-decoration:none;box-shadow:0 3px 10px rgba(249,83,198,0.4);transition:filter 0.15s,transform 0.15s"
               onmouseover="this.style.filter='brightness(1.15)';this.style.transform='scale(1.02)'"
               onmouseout="this.style.filter='';this.style.transform=''">
                <span style="font-size:15px">🥷</span> Visit AndrewBaker.Ninja
            </a>
            <a href="<?php echo esc_url( admin_url( 'tools.php?page=cloudscale-cleanup' ) ); ?>"
               style="display:flex;align-items:center;justify-content:center;gap:8px;background:linear-gradient(135deg,#0ea5e9 0%,#0369a1 100%);color:#fff;font-weight:700;font-size:13px;padding:10px 16px;border-radius:8px;text-decoration:none;box-shadow:0 3px 10px rgba(14,165,233,0.35);transition:filter 0.15s,transform 0.15s"
               onmouseover="this.style.filter='brightness(1.15)';this.style.transform='scale(1.02)'"
               onmouseout="this.style.filter='';this.style.transform=''">
                <span style="font-size:15px">⚡</span> Open CloudScale Cleanup
            </a>
        </div>
    </div>
    <?php
}


// ─── Front-end sidebar widget ─────────────────────────────────────────────────
/*
 * Registers a widget visible in Appearance -> Widgets (classic widget screen)
 * and the block editor widget screen. Drag it into any sidebar or widget area
 * in your theme to show it on the front end of the site.
 */

add_action( 'widgets_init', function () {
    register_widget( 'CSC_Front_Widget' );
} );

class CSC_Front_Widget extends WP_Widget {

    public function __construct() {
        parent::__construct(
            'csc_front_widget',
            'CloudScale Cleanup',
            array(
                'description' => 'Shows last cleanup run times and links to the CloudScale Cleanup plugin and andrewbaker.ninja.',
                'classname'   => 'widget-csc-cleanup',
            )
        );
    }

    /** Render the widget on the front end */
    public function widget( $args, $instance ) {
        $title    = ! empty( $instance['title'] ) ? $instance['title'] : 'Site Maintenance';
        $last_db  = get_option( 'csc_last_db_cleanup',   null );
        $last_img = get_option( 'csc_last_img_cleanup',  null );
        $last_opt = get_option( 'csc_last_img_optimise', null );

        echo $args['before_widget'];
        echo $args['before_title'] . esc_html( $title ) . $args['after_title'];
        ?>
        <div class="csc-front-widget">
            <ul class="csc-fw-list">
                <li>
                    <span class="csc-fw-label">DB Cleanup</span>
                    <span class="csc-fw-value"><?php echo $last_db  ? esc_html( human_time_diff( strtotime( $last_db  ), time() ) . ' ago' ) : 'Never run'; ?></span>
                </li>
                <li>
                    <span class="csc-fw-label">Image Cleanup</span>
                    <span class="csc-fw-value"><?php echo $last_img ? esc_html( human_time_diff( strtotime( $last_img ), time() ) . ' ago' ) : 'Never run'; ?></span>
                </li>
                <li>
                    <span class="csc-fw-label">Img Optimisation</span>
                    <span class="csc-fw-value"><?php echo $last_opt ? esc_html( human_time_diff( strtotime( $last_opt ), time() ) . ' ago' ) : 'Never run'; ?></span>
                </li>
            </ul>
            <div class="csc-fw-links">
                <a href="https://andrewbaker.ninja" target="_blank" rel="noopener" class="csc-fw-link">andrewbaker.ninja</a>
                <?php if ( current_user_can( 'manage_options' ) ) : ?>
                <a href="<?php echo esc_url( admin_url( 'tools.php?page=cloudscale-cleanup' ) ); ?>" class="csc-fw-link csc-fw-link-admin">Run Cleanup</a>
                <?php endif; ?>
            </div>
            <p class="csc-fw-credit">Powered by <a href="https://andrewbaker.ninja" target="_blank" rel="noopener">CloudScale Cleanup</a></p>
        </div>
        <?php
        echo $args['after_widget'];
    }

    /** Settings form in Appearance -> Widgets */
    public function form( $instance ) {
        $title = ! empty( $instance['title'] ) ? $instance['title'] : 'Site Maintenance';
        ?>
        <p>
            <label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>">Title:</label>
            <input class="widefat"
                   id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"
                   name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>"
                   type="text" value="<?php echo esc_attr( $title ); ?>">
        </p>
        <?php
    }

    /** Save widget settings */
    public function update( $new_instance, $old_instance ) {
        $instance          = $old_instance;
        $instance['title'] = sanitize_text_field( $new_instance['title'] );
        return $instance;
    }
}

// Inline CSS for the front-end widget — only loaded when widget is active
add_action( 'wp_enqueue_scripts', 'csc_enqueue_front_widget_styles' );
function csc_enqueue_front_widget_styles() {
    if ( ! is_active_widget( false, false, 'csc_front_widget', true ) ) {
        return;
    }
    wp_add_inline_style( 'wp-block-library', '
.csc-front-widget{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,sans-serif;font-size:13.5px}
.csc-fw-list{margin:0 0 12px;padding:0;list-style:none}
.csc-fw-list li{display:flex;justify-content:space-between;align-items:center;padding:6px 0;border-bottom:1px solid rgba(0,0,0,.07)}
.csc-fw-list li:last-child{border-bottom:none}
.csc-fw-label{color:#555;font-size:12.5px}
.csc-fw-value{font-weight:600;color:#1a1f2e;font-size:12.5px}
.csc-fw-links{display:flex;gap:8px;flex-wrap:wrap;margin-bottom:10px}
.csc-fw-link{display:inline-block;font-size:12px;font-weight:600;padding:6px 12px;border-radius:5px;text-decoration:none;background:#1a1f2e;color:#fff!important;transition:background .15s}
.csc-fw-link:hover{background:#4a9eff;color:#fff!important}
.csc-fw-link-admin{background:#27ae60}
.csc-fw-link-admin:hover{background:#219150}
.csc-fw-credit{font-size:11px;color:#999;margin:0}
.csc-fw-credit a{color:#4a9eff;text-decoration:none}
    ' );
}

// ─── Settings save ────────────────────────────────────────────────────────────

add_action( 'wp_ajax_csc_save_settings', 'csc_ajax_save_settings' );
function csc_ajax_save_settings() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( 'Insufficient permissions.' );
    }

    $scalars = array(
        'csc_post_revisions_age', 'csc_drafts_age', 'csc_trash_age',
        'csc_autodraft_age', 'csc_spam_comments_age', 'csc_trash_comments_age',
        'csc_img_max_width', 'csc_img_max_height', 'csc_img_quality',
        'csc_schedule_db_hour', 'csc_schedule_img_hour',
        'csc_clean_revisions', 'csc_clean_drafts', 'csc_clean_trashed', 'csc_clean_autodrafts',
        'csc_clean_transients', 'csc_clean_orphan_post', 'csc_clean_orphan_user',
        'csc_clean_spam_comments', 'csc_clean_trash_comments',
    );
    $bools = array(
        'csc_schedule_db_enabled', 'csc_schedule_img_enabled', 'csc_convert_png_to_jpg',
    );
    $arrays = array( 'csc_schedule_db_days', 'csc_schedule_img_days' );

    foreach ( $scalars as $f ) {
        if ( isset( $_POST[ $f ] ) ) {
            $val = sanitize_text_field( wp_unslash( $_POST[ $f ] ) );
            // Toggle fields: only accept '0' or '1'
            if ( in_array( $f, array(
                'csc_clean_revisions', 'csc_clean_drafts', 'csc_clean_trashed', 'csc_clean_autodrafts',
                'csc_clean_transients', 'csc_clean_orphan_post', 'csc_clean_orphan_user',
                'csc_clean_spam_comments', 'csc_clean_trash_comments',
            ), true ) ) {
                $val = $val === '1' ? '1' : '0';
            }
            update_option( $f, $val );
        }
    }
    foreach ( $bools as $f ) {
        update_option( $f, isset( $_POST[ $f ] ) ? '1' : '0' );
    }
    foreach ( $arrays as $f ) {
        update_option( $f, isset( $_POST[ $f ] ) && is_array( $_POST[ $f ] )
            ? array_map( 'sanitize_text_field', $_POST[ $f ] )
            : array()
        );
    }

    csc_schedule_crons();
    wp_send_json_success( 'Settings saved.' );
}

// ─── Cron scheduling ─────────────────────────────────────────────────────────

function csc_schedule_crons() {
    wp_clear_scheduled_hook( 'csc_scheduled_db_cleanup' );
    if ( get_option( 'csc_schedule_db_enabled', '0' ) === '1' ) {
        $ts = csc_next_run_timestamp(
            (array) get_option( 'csc_schedule_db_days', array() ),
            intval( get_option( 'csc_schedule_db_hour', 3 ) )
        );
        if ( $ts ) { wp_schedule_single_event( $ts, 'csc_scheduled_db_cleanup' ); }
    }

    wp_clear_scheduled_hook( 'csc_scheduled_img_cleanup' );
    if ( get_option( 'csc_schedule_img_enabled', '0' ) === '1' ) {
        $ts = csc_next_run_timestamp(
            (array) get_option( 'csc_schedule_img_days', array() ),
            intval( get_option( 'csc_schedule_img_hour', 4 ) )
        );
        if ( $ts ) { wp_schedule_single_event( $ts, 'csc_scheduled_img_cleanup' ); }
    }
}

function csc_next_run_timestamp( $days, $hour ) {
    $map = array(
        'mon' => 'Monday', 'tue' => 'Tuesday', 'wed' => 'Wednesday',
        'thu' => 'Thursday', 'fri' => 'Friday', 'sat' => 'Saturday', 'sun' => 'Sunday',
    );
    $now  = current_time( 'timestamp' );
    $best = null;
    foreach ( $days as $d ) {
        $d = strtolower( trim( $d ) );
        if ( ! isset( $map[ $d ] ) ) { continue; }
        $candidate = strtotime( 'next ' . $map[ $d ], $now );
        $candidate = mktime( $hour, 0, 0, date( 'n', $candidate ), date( 'j', $candidate ), date( 'Y', $candidate ) );
        if ( $candidate <= $now ) { $candidate += WEEK_IN_SECONDS; }
        if ( $best === null || $candidate < $best ) { $best = $candidate; }
    }
    return $best;
}

// Cron handlers — run synchronously (no HTTP chunking needed in a cron context)
add_action( 'csc_scheduled_db_cleanup', 'csc_cron_db_cleanup' );
function csc_cron_db_cleanup() {
    $ids = csc_build_db_id_list();
    foreach ( $ids['revisions']      as $id ) { wp_delete_post( intval( $id ), true ); }
    foreach ( $ids['drafts']         as $id ) { wp_delete_post( intval( $id ), true ); }
    foreach ( $ids['trashed']        as $id ) { wp_delete_post( intval( $id ), true ); }
    foreach ( $ids['autodrafts']     as $id ) { wp_delete_post( intval( $id ), true ); }
    csc_delete_expired_transients();
    csc_delete_orphaned_postmeta();
    csc_delete_orphaned_usermeta();
    foreach ( $ids['spam_comments']  as $id ) { wp_delete_comment( intval( $id ), true ); }
    foreach ( $ids['trash_comments'] as $id ) { wp_delete_comment( intval( $id ), true ); }
    update_option( 'csc_last_db_cleanup', current_time( 'mysql' ) );
    csc_schedule_crons();
}

add_action( 'csc_scheduled_img_cleanup', 'csc_cron_img_cleanup' );
function csc_cron_img_cleanup() {
    $used = csc_get_used_attachment_ids();
    $all  = get_posts( array(
        'post_type' => 'attachment', 'post_status' => 'inherit',
        'posts_per_page' => -1, 'fields' => 'ids',
    ) );
    foreach ( $all as $id ) {
        if ( ! isset( $used[ $id ] ) ) { wp_delete_attachment( $id, true ); }
    }
    update_option( 'csc_last_img_cleanup', current_time( 'mysql' ) );
    csc_schedule_crons();
}

// ═════════════════════════════════════════════════════════════════════════════
// DATABASE CLEANUP
// ═════════════════════════════════════════════════════════════════════════════

function csc_build_db_id_list( $overrides = array() ) {
    global $wpdb;
    $ra  = intval( get_option( 'csc_post_revisions_age', 30 ) );
    $da  = intval( get_option( 'csc_drafts_age', 90 ) );
    $ta  = intval( get_option( 'csc_trash_age', 30 ) );
    $aa  = intval( get_option( 'csc_autodraft_age', 7 ) );
    $sa  = intval( get_option( 'csc_spam_comments_age', 30 ) );
    $tca = intval( get_option( 'csc_trash_comments_age', 30 ) );

    $tog = function( $opt ) use ( $overrides ) {
        if ( ! empty( $overrides ) ) {
            // Full UI submission passed — absent key means toggled off
            return isset( $overrides[ $opt ] ) && $overrides[ $opt ] === '1';
        }
        return get_option( $opt, '1' ) === '1';
    };

    return array(
        'revisions'      => $tog( 'csc_clean_revisions' )      ? $wpdb->get_col( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_type='revision' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $ra ) ) : array(),
        'drafts'         => $tog( 'csc_clean_drafts' )         ? $wpdb->get_col( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_status='draft' AND post_type='post' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $da ) ) : array(),
        'trashed'        => $tog( 'csc_clean_trashed' )        ? $wpdb->get_col( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_status='trash' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $ta ) ) : array(),
        'autodrafts'     => $tog( 'csc_clean_autodrafts' )     ? $wpdb->get_col( $wpdb->prepare( "SELECT ID FROM {$wpdb->posts} WHERE post_status='auto-draft' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $aa ) ) : array(),
        'spam_comments'  => $tog( 'csc_clean_spam_comments' )  ? $wpdb->get_col( $wpdb->prepare( "SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved='spam' AND comment_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $sa ) ) : array(),
        'trash_comments' => $tog( 'csc_clean_trash_comments' ) ? $wpdb->get_col( $wpdb->prepare( "SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved='trash' AND comment_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $tca ) ) : array(),
    );
}

function csc_delete_expired_transients() {
    global $wpdb;
    $keys = $wpdb->get_col( "SELECT option_name FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%' AND option_value < UNIX_TIMESTAMP()" );
    foreach ( $keys as $k ) { delete_transient( str_replace( '_transient_timeout_', '', $k ) ); }
    return count( $keys );
}

function csc_delete_orphaned_postmeta() {
    global $wpdb;
    return (int) $wpdb->query( "DELETE pm FROM {$wpdb->postmeta} pm LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID WHERE p.ID IS NULL" );
}

function csc_delete_orphaned_usermeta() {
    global $wpdb;
    return (int) $wpdb->query( "DELETE um FROM {$wpdb->usermeta} um LEFT JOIN {$wpdb->users} u ON um.user_id = u.ID WHERE u.ID IS NULL" );
}

// Dry run
add_action( 'wp_ajax_csc_scan_db', 'csc_ajax_scan_db' );
function csc_ajax_scan_db() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    // Read toggle state from POST if provided (live UI state), otherwise fall back to DB.
    // If ANY toggle key is present in POST, we treat this as a full UI submission —
    // missing keys default to '0' rather than falling back to DB, preventing stale DB
    // values from overriding the user's current screen state.
    $has_post_toggles = isset( $_POST['csc_clean_revisions'] )
        || isset( $_POST['csc_clean_drafts'] )
        || isset( $_POST['csc_clean_transients'] );



    $toggle = function( $opt ) use ( $has_post_toggles ) {
        if ( $has_post_toggles ) {
            // Full UI submission — use POST value, absent = '0' (toggled off)
            return isset( $_POST[ $opt ] ) && $_POST[ $opt ] === '1';
        }
        // No UI data sent (e.g. scheduled run) — use DB
        return get_option( $opt, '1' ) === '1';
    };

    global $wpdb;
    $ra  = intval( get_option( 'csc_post_revisions_age', 30 ) );
    $da  = intval( get_option( 'csc_drafts_age', 90 ) );
    $ta  = intval( get_option( 'csc_trash_age', 30 ) );
    $aa  = intval( get_option( 'csc_autodraft_age', 7 ) );
    $sa  = intval( get_option( 'csc_spam_comments_age', 30 ) );
    $tca = intval( get_option( 'csc_trash_comments_age', 30 ) );

    $toggle_keys = array(
        'csc_clean_revisions', 'csc_clean_drafts', 'csc_clean_trashed', 'csc_clean_autodrafts',
        'csc_clean_transients', 'csc_clean_orphan_post', 'csc_clean_orphan_user',
        'csc_clean_spam_comments', 'csc_clean_trash_comments',
    );
    $lines = array();

    if ( $toggle( 'csc_clean_revisions' ) ) {
        $revisions = $wpdb->get_results( $wpdb->prepare( "SELECT ID, post_title, post_date FROM {$wpdb->posts} WHERE post_type='revision' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY post_date DESC LIMIT 1000", $ra ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Post Revisions (older than ' . $ra . ' days)' );
        foreach ( $revisions as $r ) { $lines[] = array( 'type' => 'item', 'text' => '  [REVISION] ID ' . $r->ID . ' — ' . esc_html( $r->post_title ) . ' (' . $r->post_date . ')' ); }
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . count( $revisions ) );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Post Revisions — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_drafts' ) ) {
        $drafts = $wpdb->get_results( $wpdb->prepare( "SELECT ID, post_title, post_date FROM {$wpdb->posts} WHERE post_status='draft' AND post_type='post' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY post_date DESC LIMIT 500", $da ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Draft Posts (older than ' . $da . ' days)' );
        foreach ( $drafts as $d ) { $lines[] = array( 'type' => 'item', 'text' => '  [DRAFT] ID ' . $d->ID . ' — ' . esc_html( $d->post_title ) . ' (' . $d->post_date . ')' ); }
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . count( $drafts ) );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Draft Posts — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_trashed' ) ) {
        $trashed = $wpdb->get_results( $wpdb->prepare( "SELECT ID, post_title, post_modified FROM {$wpdb->posts} WHERE post_status='trash' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY) ORDER BY post_modified DESC LIMIT 500", $ta ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Trashed Posts (older than ' . $ta . ' days)' );
        foreach ( $trashed as $t ) { $lines[] = array( 'type' => 'item', 'text' => '  [TRASH] ID ' . $t->ID . ' — ' . esc_html( $t->post_title ) . ' (' . $t->post_modified . ')' ); }
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . count( $trashed ) );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Trashed Posts — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_autodrafts' ) ) {
        $cnt_auto = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_status='auto-draft' AND post_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $aa ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Auto-Drafts (older than ' . $aa . ' days)' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_auto );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Auto-Drafts — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_transients' ) ) {
        $cnt_t = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_%' AND option_value < UNIX_TIMESTAMP()" );
        $lines[] = array( 'type' => 'section', 'text' => 'Expired Transients' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_t );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Expired Transients — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_orphan_post' ) ) {
        $cnt_pm = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->postmeta} pm LEFT JOIN {$wpdb->posts} p ON pm.post_id = p.ID WHERE p.ID IS NULL" );
        $lines[] = array( 'type' => 'section', 'text' => 'Orphaned Post Meta' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_pm . ' rows' );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Orphaned Post Meta — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_orphan_user' ) ) {
        $cnt_um = (int) $wpdb->get_var( "SELECT COUNT(*) FROM {$wpdb->usermeta} um LEFT JOIN {$wpdb->users} u ON um.user_id = u.ID WHERE u.ID IS NULL" );
        $lines[] = array( 'type' => 'section', 'text' => 'Orphaned User Meta' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_um . ' rows' );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Orphaned User Meta — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_spam_comments' ) ) {
        $cnt_spam = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved='spam' AND comment_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $sa ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Spam Comments (older than ' . $sa . ' days)' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_spam );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Spam Comments — SKIPPED (disabled)' );
    }

    if ( $toggle( 'csc_clean_trash_comments' ) ) {
        $cnt_tc = (int) $wpdb->get_var( $wpdb->prepare( "SELECT COUNT(*) FROM {$wpdb->comments} WHERE comment_approved='trash' AND comment_date < DATE_SUB(NOW(), INTERVAL %d DAY)", $tca ) );
        $lines[] = array( 'type' => 'section', 'text' => 'Trashed Comments (older than ' . $tca . ' days)' );
        $lines[] = array( 'type' => 'count', 'text' => '  Found: ' . $cnt_tc );
    } else {
        $lines[] = array( 'type' => 'section', 'text' => 'Trashed Comments — SKIPPED (disabled)' );
    }

    wp_send_json_success( $lines );
}

// Chunked run — Step 1: build queue
add_action( 'wp_ajax_csc_db_start', 'csc_ajax_db_start' );
function csc_ajax_db_start() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    // Collect any toggle overrides sent from the live UI
    $toggle_keys = array(
        'csc_clean_revisions', 'csc_clean_drafts', 'csc_clean_trashed', 'csc_clean_autodrafts',
        'csc_clean_transients', 'csc_clean_orphan_post', 'csc_clean_orphan_user',
        'csc_clean_spam_comments', 'csc_clean_trash_comments',
    );
    $overrides = array();
    foreach ( $toggle_keys as $k ) {
        if ( isset( $_POST[ $k ] ) ) {
            $overrides[ $k ] = $_POST[ $k ] === '1' ? '1' : '0';
        }
    }

    $has_post_toggles = isset( $_POST['csc_clean_revisions'] )
        || isset( $_POST['csc_clean_drafts'] )
        || isset( $_POST['csc_clean_transients'] );

    $tog = function( $opt ) use ( $overrides, $has_post_toggles ) {
        if ( $has_post_toggles ) {
            return isset( $overrides[ $opt ] ) && $overrides[ $opt ] === '1';
        }
        return get_option( $opt, '1' ) === '1';
    };

    $ids   = csc_build_db_id_list( $overrides );
    $queue = array();
    foreach ( $ids['revisions']      as $id ) { $queue[] = array( 'type' => 'post',          'id' => intval( $id ), 'label' => 'revision' ); }
    foreach ( $ids['drafts']         as $id ) { $queue[] = array( 'type' => 'post',          'id' => intval( $id ), 'label' => 'draft' ); }
    foreach ( $ids['trashed']        as $id ) { $queue[] = array( 'type' => 'post',          'id' => intval( $id ), 'label' => 'trashed post' ); }
    foreach ( $ids['autodrafts']     as $id ) { $queue[] = array( 'type' => 'post',          'id' => intval( $id ), 'label' => 'auto-draft' ); }
    foreach ( $ids['spam_comments']  as $id ) { $queue[] = array( 'type' => 'comment',       'id' => intval( $id ), 'label' => 'spam comment' ); }
    foreach ( $ids['trash_comments'] as $id ) { $queue[] = array( 'type' => 'comment',       'id' => intval( $id ), 'label' => 'trashed comment' ); }
    if ( $tog( 'csc_clean_transients' ) )  { $queue[] = array( 'type' => 'transients',  'id' => 0, 'label' => 'expired transients' ); }
    if ( $tog( 'csc_clean_orphan_post' ) ) { $queue[] = array( 'type' => 'orphan_post', 'id' => 0, 'label' => 'orphaned postmeta' ); }
    if ( $tog( 'csc_clean_orphan_user' ) ) { $queue[] = array( 'type' => 'orphan_user', 'id' => 0, 'label' => 'orphaned usermeta' ); }

    set_transient( 'csc_db_queue', $queue, HOUR_IN_SECONDS );

    wp_send_json_success( array(
        'total'     => count( $queue ),
        'remaining' => count( $queue ),
        'lines'     => array( array( 'type' => 'info', 'text' => '  Work queue built: ' . count( $queue ) . ' items.' ) ),
    ) );
}

// Step 2: process a chunk
add_action( 'wp_ajax_csc_db_chunk', 'csc_ajax_db_chunk' );
function csc_ajax_db_chunk() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $queue = get_transient( 'csc_db_queue' );
    if ( ! is_array( $queue ) ) { wp_send_json_error( 'Session expired — please start again.' ); }

    $chunk = array_splice( $queue, 0, CSC_CHUNK_DB );
    set_transient( 'csc_db_queue', $queue, HOUR_IN_SECONDS );

    $lines = array();
    foreach ( $chunk as $item ) {
        switch ( $item['type'] ) {
            case 'post':
                wp_delete_post( $item['id'], true );
                $lines[] = array( 'type' => 'deleted', 'text' => '  Deleted ' . $item['label'] . ' ID ' . $item['id'] );
                break;
            case 'comment':
                wp_delete_comment( $item['id'], true );
                $lines[] = array( 'type' => 'deleted', 'text' => '  Deleted ' . $item['label'] . ' ID ' . $item['id'] );
                break;
            case 'transients':
                $n = csc_delete_expired_transients();
                $lines[] = array( 'type' => 'count', 'text' => '  Deleted ' . $n . ' expired transients.' );
                break;
            case 'orphan_post':
                $n = csc_delete_orphaned_postmeta();
                $lines[] = array( 'type' => 'count', 'text' => '  Deleted ' . $n . ' orphaned postmeta rows.' );
                break;
            case 'orphan_user':
                $n = csc_delete_orphaned_usermeta();
                $lines[] = array( 'type' => 'count', 'text' => '  Deleted ' . $n . ' orphaned usermeta rows.' );
                break;
        }
    }

    wp_send_json_success( array( 'remaining' => count( $queue ), 'lines' => $lines ) );
}

// Step 3: finish
add_action( 'wp_ajax_csc_db_finish', 'csc_ajax_db_finish' );
function csc_ajax_db_finish() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }
    delete_transient( 'csc_db_queue' );
    update_option( 'csc_last_db_cleanup', current_time( 'mysql' ) );
    wp_send_json_success( array( 'lines' => array( array( 'type' => 'success', 'text' => 'Database cleanup complete.' ) ) ) );
}

// ═════════════════════════════════════════════════════════════════════════════
// IMAGE CLEANUP
// ═════════════════════════════════════════════════════════════════════════════

function csc_get_used_attachment_ids() {
    global $wpdb;
    $used = array();

    // Featured images
    foreach ( $wpdb->get_col( "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key='_thumbnail_id'" ) as $id ) {
        $used[ intval( $id ) ] = true;
    }

    // Gutenberg block IDs and legacy class-based image references
    $contents = $wpdb->get_col( "SELECT post_content FROM {$wpdb->posts} WHERE post_status='publish' AND post_type NOT IN ('attachment','revision')" );
    foreach ( $contents as $c ) {
        if ( preg_match_all( '/wp-image-(\d+)/i', $c, $m ) ) {
            foreach ( $m[1] as $id ) { $used[ intval( $id ) ] = true; }
        }
        if ( preg_match_all( '/"id"\s*:\s*(\d+)/i', $c, $m ) ) {
            foreach ( $m[1] as $id ) { $used[ intval( $id ) ] = true; }
        }
    }

    // Widget options and theme mods
    $opts = $wpdb->get_col( "SELECT option_value FROM {$wpdb->options} WHERE option_name LIKE 'widget_%' OR option_name LIKE 'theme_mods_%'" );
    foreach ( $opts as $v ) {
        if ( preg_match_all( '/"id"\s*:\s*(\d+)/i', $v, $m ) ) {
            foreach ( $m[1] as $id ) { $used[ intval( $id ) ] = true; }
        }
    }

    // Site logo and site icon are always protected
    $logo = get_theme_mod( 'custom_logo' );
    if ( $logo ) { $used[ intval( $logo ) ] = true; }
    $icon = get_option( 'site_icon' );
    if ( $icon ) { $used[ intval( $icon ) ] = true; }

    return $used;
}

// Dry run
add_action( 'wp_ajax_csc_scan_images', 'csc_ajax_scan_images' );
function csc_ajax_scan_images() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $used = csc_get_used_attachment_ids();
    $all  = get_posts( array( 'post_type' => 'attachment', 'post_status' => 'inherit', 'posts_per_page' => -1, 'fields' => 'ids' ) );
    $lines = array();
    $lines[] = array( 'type' => 'section', 'text' => 'Unused Media Attachments' );
    $lines[] = array( 'type' => 'info',    'text' => '  Total in library: ' . count( $all ) . '   Confirmed in use: ' . count( $used ) );

    $unused = array();
    foreach ( $all as $id ) {
        if ( ! isset( $used[ $id ] ) ) { $unused[] = $id; }
    }
    $total_unused_size = 0;
    foreach ( $unused as $id ) {
        $file      = get_attached_file( $id );
        $file_size = ( $file && file_exists( $file ) ) ? filesize( $file ) : 0;
        $total_unused_size += $file_size;
        $ext     = $file ? strtoupper( pathinfo( $file, PATHINFO_EXTENSION ) ) : '';
        $size_str = $file_size > 0 ? size_format( $file_size ) : 'file missing';
        $label   = esc_html( get_the_title( $id ) );
        if ( $ext ) { $label .= '.' . strtolower( $ext ); }
        $lines[] = array( 'type' => 'item', 'text' => '  [UNUSED] ID ' . $id . ' — ' . $label . ' (' . $size_str . ')' );
    }
    $lines[] = array( 'type' => 'count', 'text' => '  Total unused: ' . count( $unused ) );
    $lines[] = array( 'type' => 'count', 'text' => '  Total size on disk: ' . size_format( $total_unused_size ) );

    wp_send_json_success( $lines );
}

// Chunked run — Step 1
add_action( 'wp_ajax_csc_img_start', 'csc_ajax_img_start' );
function csc_ajax_img_start() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $used  = csc_get_used_attachment_ids();
    $all   = get_posts( array( 'post_type' => 'attachment', 'post_status' => 'inherit', 'posts_per_page' => -1, 'fields' => 'ids' ) );
    $queue = array();
    foreach ( $all as $id ) {
        if ( ! isset( $used[ $id ] ) ) { $queue[] = intval( $id ); }
    }

    set_transient( 'csc_img_queue', $queue, HOUR_IN_SECONDS );
    wp_send_json_success( array(
        'total'     => count( $queue ),
        'remaining' => count( $queue ),
        'lines'     => array( array( 'type' => 'info', 'text' => '  Found ' . count( $queue ) . ' unused attachments to delete.' ) ),
    ) );
}

// Step 2
add_action( 'wp_ajax_csc_img_chunk', 'csc_ajax_img_chunk' );
function csc_ajax_img_chunk() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $queue = get_transient( 'csc_img_queue' );
    if ( ! is_array( $queue ) ) { wp_send_json_error( 'Session expired — please start again.' ); }

    $chunk = array_splice( $queue, 0, CSC_CHUNK_IMAGES );
    set_transient( 'csc_img_queue', $queue, HOUR_IN_SECONDS );

    $lines = array();
    foreach ( $chunk as $id ) {
        $title = get_the_title( $id );
        wp_delete_attachment( $id, true );
        $lines[] = array( 'type' => 'deleted', 'text' => '  Deleted ID ' . $id . ' — ' . esc_html( $title ) );
    }

    wp_send_json_success( array( 'remaining' => count( $queue ), 'lines' => $lines ) );
}

// Step 3
add_action( 'wp_ajax_csc_img_finish', 'csc_ajax_img_finish' );
function csc_ajax_img_finish() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }
    delete_transient( 'csc_img_queue' );
    update_option( 'csc_last_img_cleanup', current_time( 'mysql' ) );
    wp_send_json_success( array( 'lines' => array( array( 'type' => 'success', 'text' => 'Image cleanup complete.' ) ) ) );
}

// ═════════════════════════════════════════════════════════════════════════════
// ORPHAN FILE SCAN
// ═════════════════════════════════════════════════════════════════════════════

// ── Orphan helpers ────────────────────────────────────────────────────────────

function csc_recycle_dir(): string {
    return trailingslashit( wp_upload_dir()['basedir'] ) . '.csc-recycle/';
}

function csc_recycle_manifest(): string {
    return csc_recycle_dir() . 'manifest.json';
}

function csc_orphan_ext_sets(): array {
    return array(
        'all'       => array(),
        'images'    => array( 'jpg', 'jpeg', 'png', 'gif', 'webp', 'bmp', 'tiff', 'svg' ),
        'documents' => array( 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'ppt', 'pptx', 'txt', 'csv', 'rtf', 'odt', 'ods', 'odp' ),
        'video'     => array( 'mp4', 'mov', 'avi', 'wmv', 'mkv', 'webm', 'flv', 'm4v' ),
        'audio'     => array( 'mp3', 'wav', 'ogg', 'aac', 'flac', 'm4a', 'wma' ),
    );
}

function csc_get_orphan_files( string $type = 'all' ): array {
    global $wpdb;
    $upload_dir = wp_upload_dir();
    $base       = trailingslashit( $upload_dir['basedir'] );
    $recycle    = csc_recycle_dir();
    $ext_sets   = csc_orphan_ext_sets();
    $exts       = isset( $ext_sets[ $type ] ) ? $ext_sets[ $type ] : array();

    $db_files = array();
    foreach ( $wpdb->get_col( "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key='_wp_attached_file'" ) as $f ) {
        $db_files[ $base . $f ] = true;
    }
    foreach ( $wpdb->get_col( "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key='_wp_attachment_metadata'" ) as $raw ) {
        $data = maybe_unserialize( $raw );
        if ( ! is_array( $data ) || ! isset( $data['file'] ) ) { continue; }
        $dir = trailingslashit( $base . dirname( $data['file'] ) );
        if ( isset( $data['sizes'] ) && is_array( $data['sizes'] ) ) {
            foreach ( $data['sizes'] as $sz ) {
                if ( isset( $sz['file'] ) ) { $db_files[ $dir . $sz['file'] ] = true; }
            }
        }
    }

    $orphans = array();

    try {
        $iter = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $base, RecursiveDirectoryIterator::SKIP_DOTS ) );
        foreach ( $iter as $file ) {
            if ( ! $file->isFile() ) { continue; }
            $path = $file->getRealPath();
            // Skip anything inside the recycle bin
            if ( strpos( $path, $recycle ) === 0 ) { continue; }
            $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
            if ( ! empty( $exts ) && ! in_array( $ext, $exts, true ) ) { continue; }
            if ( ! isset( $db_files[ $path ] ) ) {
                $orphans[] = array( 'path' => $path, 'size' => $file->getSize() );
            }
        }
    } catch ( Exception $e ) {
        return array();
    }

    return $orphans;
}

function csc_get_orphan_files_multi( array $types ): array {
    $ext_sets = csc_orphan_ext_sets();
    $exts = array();
    foreach ( $types as $type ) {
        if ( isset( $ext_sets[ $type ] ) ) {
            $exts = array_merge( $exts, $ext_sets[ $type ] );
        }
    }
    $exts = array_unique( $exts );
    // Pass merged ext list via a special 'custom' path
    global $wpdb;
    $upload_dir = wp_upload_dir();
    $base       = trailingslashit( $upload_dir['basedir'] );
    $recycle    = csc_recycle_dir();

    $db_files = array();
    foreach ( $wpdb->get_col( "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key='_wp_attached_file'" ) as $f ) {
        $db_files[ $base . $f ] = true;
    }
    foreach ( $wpdb->get_col( "SELECT meta_value FROM {$wpdb->postmeta} WHERE meta_key='_wp_attachment_metadata'" ) as $raw ) {
        $data = maybe_unserialize( $raw );
        if ( ! is_array( $data ) || ! isset( $data['file'] ) ) { continue; }
        $dir = trailingslashit( $base . dirname( $data['file'] ) );
        if ( isset( $data['sizes'] ) && is_array( $data['sizes'] ) ) {
            foreach ( $data['sizes'] as $sz ) {
                if ( isset( $sz['file'] ) ) { $db_files[ $dir . $sz['file'] ] = true; }
            }
        }
    }

    $orphans = array();
    try {
        $iter = new RecursiveIteratorIterator( new RecursiveDirectoryIterator( $base, RecursiveDirectoryIterator::SKIP_DOTS ) );
        foreach ( $iter as $file ) {
            if ( ! $file->isFile() ) { continue; }
            $path = $file->getRealPath();
            if ( strpos( $path, $recycle ) === 0 ) { continue; }
            $ext = strtolower( pathinfo( $path, PATHINFO_EXTENSION ) );
            if ( ! empty( $exts ) && ! in_array( $ext, $exts, true ) ) { continue; }
            if ( ! isset( $db_files[ $path ] ) ) {
                $orphans[] = array( 'path' => $path, 'size' => $file->getSize() );
            }
        }
    } catch ( Exception $e ) {
        return array();
    }
    return $orphans;
}

function csc_recycle_count(): int {
    $manifest = csc_recycle_manifest();
    if ( ! file_exists( $manifest ) ) { return 0; }
    $data = json_decode( file_get_contents( $manifest ), true );
    return is_array( $data ) ? count( $data ) : 0;
}

// ── Scan ─────────────────────────────────────────────────────────────────────

add_action( 'wp_ajax_csc_scan_orphan_files', 'csc_ajax_scan_orphan_files' );
function csc_ajax_scan_orphan_files() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $raw_types  = sanitize_text_field( $_POST['file_type'] ?? '' );
    $ext_sets   = csc_orphan_ext_sets();
    if ( empty( $raw_types ) ) {
        wp_send_json_error( 'No file type selected. Please select at least one file type.' );
        return;
    }
    $selected_types = array_filter( array_map( 'sanitize_key', explode( ',', $raw_types ) ), fn($t) => array_key_exists( $t, $ext_sets ) );
    if ( empty( $selected_types ) ) {
        wp_send_json_error( 'Invalid file type selection.' );
        return;
    }
    $type_label = implode( ' + ', array_map( fn($t) => ucfirst($t), $selected_types ) );
    $orphans    = csc_get_orphan_files_multi( $selected_types );
    $base      = trailingslashit( wp_upload_dir()['basedir'] );
    $recycle_n = csc_recycle_count();
    $lines     = array();
    $lines[]   = array( 'type' => 'section', 'text' => 'Orphaned Files on Disk — ' . $type_label );

    if ( empty( $orphans ) ) {
        $lines[] = array( 'type' => 'info', 'text' => '  No orphaned files found.' );
    } else {
        // Split into three groups
        $group_trash   = array();
        $group_bak     = array();
        $group_regular = array();

        foreach ( $orphans as $o ) {
            $rel = str_replace( $base, '', $o['path'] );
            if ( strpos( $rel, 'wpmc-trash/' ) === 0 || strpos( $rel, '.trash/' ) === 0 ) {
                $group_trash[] = array_merge( $o, array( 'rel' => $rel ) );
            } elseif ( preg_match( '/\.bak\.[a-z0-9]+$/i', $rel ) ) {
                $group_bak[] = array_merge( $o, array( 'rel' => $rel ) );
            } else {
                $group_regular[] = array_merge( $o, array( 'rel' => $rel ) );
            }
        }

        $render_group = function( array $items, string $tag, string $label ) use ( &$lines ) {
            if ( empty( $items ) ) { return; }
            $size = array_sum( array_column( $items, 'size' ) );
            $lines[] = array( 'type' => 'section', 'text' => '  ── ' . $label . ' (' . count( $items ) . ' files, ' . size_format( $size ) . ')' );
            foreach ( $items as $o ) {
                $lines[] = array( 'type' => 'item', 'text' => '    [' . $tag . '] ' . $o['rel'] . ' (' . size_format( $o['size'] ) . ')' );
            }
        };

        $render_group( $group_trash,   'TRASH',  'Plugin Trash Folder' );
        $render_group( $group_bak,     'BACKUP', 'Backup Files (.bak)' );
        $render_group( $group_regular, 'ORPHAN', 'Unregistered Uploads' );

        $total_size = array_sum( array_column( $orphans, 'size' ) );
        $lines[] = array( 'type' => 'count', 'text' => '  Total: ' . count( $orphans ) . ' files — ' . size_format( $total_size ) . ' recoverable' );
    }

    if ( $recycle_n > 0 ) {
        $lines[] = array( 'type' => 'info', 'text' => '  ♻️ Recycle bin: ' . $recycle_n . ' file(s) awaiting permanent deletion or restore.' );
    }

    wp_send_json_success( array( 'lines' => $lines, 'found' => count( $orphans ), 'recycle' => $recycle_n ) );
}

// ── Move to Recycle ───────────────────────────────────────────────────────────

add_action( 'wp_ajax_csc_recycle_orphan_files', 'csc_ajax_recycle_orphan_files' );
function csc_ajax_recycle_orphan_files() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $raw_types = sanitize_text_field( $_POST['file_type'] ?? '' );
    $ext_sets  = csc_orphan_ext_sets();
    if ( empty( $raw_types ) ) {
        wp_send_json_error( 'No file type selected.' );
        return;
    }
    $selected_types = array_filter( array_map( 'sanitize_key', explode( ',', $raw_types ) ), fn($t) => array_key_exists( $t, $ext_sets ) );
    if ( empty( $selected_types ) ) { wp_send_json_error( 'Invalid file type.' ); return; }
    $orphans = csc_get_orphan_files_multi( $selected_types );
    $recycle = csc_recycle_dir();
    $lines   = array();
    $lines[] = array( 'type' => 'section', 'text' => '=== MOVING ORPHANS TO RECYCLE BIN ===' );

    if ( empty( $orphans ) ) {
        $lines[] = array( 'type' => 'info', 'text' => '  No orphaned files found.' );
        wp_send_json_success( array( 'lines' => $lines, 'moved' => 0 ) );
        return;
    }

    if ( ! wp_mkdir_p( $recycle ) ) {
        wp_send_json_error( 'Could not create recycle directory: ' . $recycle );
        return;
    }

    // Load existing manifest if recycle bin already has files
    $manifest_path = csc_recycle_manifest();
    $manifest = array();
    if ( file_exists( $manifest_path ) ) {
        $manifest = json_decode( file_get_contents( $manifest_path ), true ) ?: array();
    }

    $moved = 0;
    $errors = 0;
    $base = trailingslashit( wp_upload_dir()['basedir'] );

    foreach ( $orphans as $o ) {
        $rel         = str_replace( $base, '', $o['path'] );
        $dest        = $recycle . $rel;
        $dest_dir    = dirname( $dest );

        if ( ! wp_mkdir_p( $dest_dir ) ) {
            $lines[] = array( 'type' => 'error', 'text' => '  Could not create dir for: ' . $rel );
            $errors++;
            continue;
        }

        if ( rename( $o['path'], $dest ) ) {
            $manifest[ $rel ] = $o['path'];
            $lines[] = array( 'type' => 'deleted', 'text' => '  [RECYCLED] ' . $rel . ' (' . size_format( $o['size'] ) . ')' );
            $moved++;
        } else {
            $lines[] = array( 'type' => 'error', 'text' => '  Failed to move: ' . $rel );
            $errors++;
        }
    }

    file_put_contents( $manifest_path, json_encode( $manifest, JSON_PRETTY_PRINT ) );

    $lines[] = array( 'type' => 'success', 'text' => '  ✅ Moved ' . $moved . ' file(s) to recycle bin.' . ( $errors ? ' ' . $errors . ' error(s).' : '' ) );
    $lines[] = array( 'type' => 'info',    'text' => '  Files are in: wp-content/uploads/.csc-recycle/' );
    $lines[] = array( 'type' => 'info',    'text' => '  Use Restore to put them back, or Permanently Delete to wipe them.' );

    wp_send_json_success( array( 'lines' => $lines, 'moved' => $moved, 'recycle' => count( $manifest ) ) );
}

// ── Restore from Recycle ──────────────────────────────────────────────────────

add_action( 'wp_ajax_csc_restore_orphan_files', 'csc_ajax_restore_orphan_files' );
function csc_ajax_restore_orphan_files() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $manifest_path = csc_recycle_manifest();
    $lines         = array();
    $lines[]       = array( 'type' => 'section', 'text' => '=== RESTORING FILES FROM RECYCLE BIN ===' );

    if ( ! file_exists( $manifest_path ) ) {
        $lines[] = array( 'type' => 'info', 'text' => '  Recycle bin is empty — nothing to restore.' );
        wp_send_json_success( array( 'lines' => $lines, 'restored' => 0 ) );
        return;
    }

    $manifest = json_decode( file_get_contents( $manifest_path ), true ) ?: array();
    if ( empty( $manifest ) ) {
        $lines[] = array( 'type' => 'info', 'text' => '  Recycle bin is empty — nothing to restore.' );
        wp_send_json_success( array( 'lines' => $lines, 'restored' => 0 ) );
        return;
    }

    $recycle  = csc_recycle_dir();
    $restored = 0;
    $errors   = 0;

    foreach ( $manifest as $rel => $original_path ) {
        $recycle_path = $recycle . $rel;
        if ( ! file_exists( $recycle_path ) ) {
            $lines[] = array( 'type' => 'error', 'text' => '  Missing from recycle bin: ' . $rel );
            $errors++;
            continue;
        }

        $dest_dir = dirname( $original_path );
        if ( ! wp_mkdir_p( $dest_dir ) ) {
            $lines[] = array( 'type' => 'error', 'text' => '  Could not create dir for: ' . $rel );
            $errors++;
            continue;
        }

        if ( rename( $recycle_path, $original_path ) ) {
            $lines[] = array( 'type' => 'success', 'text' => '  [RESTORED] ' . $rel );
            $restored++;
            unset( $manifest[ $rel ] );
        } else {
            $lines[] = array( 'type' => 'error', 'text' => '  Failed to restore: ' . $rel );
            $errors++;
        }
    }

    // Update or remove manifest
    if ( empty( $manifest ) ) {
        @unlink( $manifest_path );
        // Clean up empty recycle dirs
        csc_rmdir_recursive( $recycle );
    } else {
        file_put_contents( $manifest_path, json_encode( $manifest, JSON_PRETTY_PRINT ) );
    }

    $lines[] = array( 'type' => 'success', 'text' => '  ✅ Restored ' . $restored . ' file(s) to original locations.' . ( $errors ? ' ' . $errors . ' error(s).' : '' ) );

    wp_send_json_success( array( 'lines' => $lines, 'restored' => $restored, 'recycle' => count( $manifest ) ) );
}

// ── Permanently Delete Recycle Bin ────────────────────────────────────────────

add_action( 'wp_ajax_csc_purge_orphan_files', 'csc_ajax_purge_orphan_files' );
function csc_ajax_purge_orphan_files() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $manifest_path = csc_recycle_manifest();
    $lines         = array();
    $lines[]       = array( 'type' => 'section', 'text' => '=== PERMANENTLY DELETING RECYCLE BIN ===' );

    if ( ! file_exists( $manifest_path ) ) {
        $lines[] = array( 'type' => 'info', 'text' => '  Recycle bin is empty — nothing to delete.' );
        wp_send_json_success( array( 'lines' => $lines, 'deleted' => 0 ) );
        return;
    }

    $manifest = json_decode( file_get_contents( $manifest_path ), true ) ?: array();
    $recycle  = csc_recycle_dir();
    $deleted  = 0;
    $errors   = 0;
    $freed    = 0;

    foreach ( $manifest as $rel => $original_path ) {
        $recycle_path = $recycle . $rel;
        if ( file_exists( $recycle_path ) ) {
            $freed += filesize( $recycle_path );
            if ( @unlink( $recycle_path ) ) {
                $lines[] = array( 'type' => 'deleted', 'text' => '  [DELETED] ' . $rel );
                $deleted++;
            } else {
                $lines[] = array( 'type' => 'error', 'text' => '  Failed to delete: ' . $rel );
                $errors++;
            }
        } else {
            $lines[] = array( 'type' => 'info', 'text' => '  Already gone: ' . $rel );
            $deleted++;
        }
    }

    csc_rmdir_recursive( $recycle );

    $lines[] = array( 'type' => 'success', 'text' => '  ✅ Permanently deleted ' . $deleted . ' file(s). Freed ' . size_format( $freed ) . '.' . ( $errors ? ' ' . $errors . ' error(s).' : '' ) );

    wp_send_json_success( array( 'lines' => $lines, 'deleted' => $deleted, 'recycle' => 0 ) );
}

// ── Recycle bin status (for page load) ───────────────────────────────────────

add_action( 'wp_ajax_csc_recycle_status', 'csc_ajax_recycle_status' );
function csc_ajax_recycle_status() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }
    wp_send_json_success( array( 'recycle' => csc_recycle_count() ) );
}

// ── Recursive directory removal ───────────────────────────────────────────────

function csc_rmdir_recursive( string $dir ): void {
    if ( ! is_dir( $dir ) ) { return; }
    try {
        $iter = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator( $dir, RecursiveDirectoryIterator::SKIP_DOTS ),
            RecursiveIteratorIterator::CHILD_FIRST
        );
        foreach ( $iter as $f ) {
            $f->isDir() ? @rmdir( $f->getRealPath() ) : @unlink( $f->getRealPath() );
        }
    } catch ( Exception $e ) {}
    @rmdir( $dir );
}

// ═════════════════════════════════════════════════════════════════════════════
// IMAGE OPTIMISATION
// ═════════════════════════════════════════════════════════════════════════════

// Dry run scan
add_action( 'wp_ajax_csc_scan_optimise', 'csc_ajax_scan_optimise' );
function csc_ajax_scan_optimise() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $max_w       = intval( get_option( 'csc_img_max_width',  1920 ) );
    $max_h       = intval( get_option( 'csc_img_max_height', 1080 ) );
    $quality     = intval( get_option( 'csc_img_quality',    82 ) );
    $convert_png = get_option( 'csc_convert_png_to_jpg', '0' ) === '1';

    $lines = array();
    $lines[] = array( 'type' => 'section', 'text' => 'Image Optimisation Scan — max ' . $max_w . 'x' . $max_h . 'px · JPEG quality ' . $quality );

    $attachments = get_posts( array( 'post_type' => 'attachment', 'post_status' => 'inherit', 'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ), 'posts_per_page' => -1, 'fields' => 'ids' ) );
    $lines[] = array( 'type' => 'info', 'text' => '  Total JPEG/PNG attachments: ' . count( $attachments ) );

    $needs_work  = 0;
    $total_saved = 0;

    foreach ( $attachments as $id ) {
        $file = get_attached_file( $id );
        if ( ! $file || ! file_exists( $file ) ) { continue; }
        $mime     = mime_content_type( $file );
        $size_now = filesize( $file );
        $dims     = @getimagesize( $file );
        if ( ! $dims ) { continue; }
        list( $w, $h ) = $dims;

        $flags  = array();
        $saving = 0;

        if ( $w > $max_w || $h > $max_h ) { $flags[] = 'oversized (' . $w . 'x' . $h . ')'; }

        if ( in_array( $mime, array( 'image/jpeg', 'image/jpg' ), true ) ) {
            $est = (int) ( $size_now * 0.22 );
            if ( $est > 1024 ) { $flags[] = 'recompressible (~' . size_format( $est ) . ' saving)'; $saving += $est; }
        }

        if ( $convert_png && $mime === 'image/png' ) {
            $est = (int) ( $size_now * 0.55 );
            $flags[] = 'PNG→JPEG (~' . size_format( $est ) . ' saving)';
            $saving += $est;
        }

        if ( ! empty( $flags ) ) {
            $needs_work++;
            $total_saved += $saving;
            // Build label: title.ext so the file type is always visible
            $ext   = strtolower( pathinfo( $file, PATHINFO_EXTENSION ) );
            $label = esc_html( get_the_title( $id ) );
            if ( $ext ) { $label .= '.' . $ext; }
            $lines[] = array( 'type' => 'item', 'text' => '  [OPTIMISE] ID ' . $id . ' — ' . $label . ' (' . size_format( $size_now ) . ') — ' . implode( ', ', $flags ) );
        }
    }

    $lines[] = array( 'type' => 'count', 'text' => '  Images to optimise: ' . $needs_work );
    $lines[] = array( 'type' => 'count', 'text' => '  Estimated total saving: ' . size_format( $total_saved ) );
    if ( $convert_png ) {
        $lines[] = array( 'type' => 'info', 'text' => '  Note: PNG→JPEG conversion is ON — all PNGs above will be converted.' );
    }
    wp_send_json_success( $lines );
}

// Chunked run — Step 1: build queue
add_action( 'wp_ajax_csc_optimise_start', 'csc_ajax_optimise_start' );
function csc_ajax_optimise_start() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $max_w       = intval( get_option( 'csc_img_max_width',  1920 ) );
    $max_h       = intval( get_option( 'csc_img_max_height', 1080 ) );
    $convert_png = get_option( 'csc_convert_png_to_jpg', '0' ) === '1';

    $all   = get_posts( array( 'post_type' => 'attachment', 'post_status' => 'inherit', 'post_mime_type' => array( 'image/jpeg', 'image/jpg', 'image/png' ), 'posts_per_page' => -1, 'fields' => 'ids' ) );
    $queue = array();

    foreach ( $all as $id ) {
        $file = get_attached_file( $id );
        if ( ! $file || ! file_exists( $file ) ) { continue; }
        $mime = mime_content_type( $file );
        $dims = @getimagesize( $file );
        if ( ! $dims ) { continue; }
        list( $w, $h ) = $dims;

        $needs = false;
        if ( $w > $max_w || $h > $max_h )                                         { $needs = true; }
        if ( in_array( $mime, array( 'image/jpeg', 'image/jpg' ), true ) )         { $needs = true; }
        if ( $convert_png && $mime === 'image/png' )                               { $needs = true; }

        if ( $needs ) { $queue[] = intval( $id ); }
    }

    set_transient( 'csc_optimise_queue', $queue, 2 * HOUR_IN_SECONDS );
    set_transient( 'csc_optimise_saved', 0,       2 * HOUR_IN_SECONDS );
    set_transient( 'csc_optimise_count', 0,       2 * HOUR_IN_SECONDS );

    wp_send_json_success( array(
        'total'     => count( $queue ),
        'remaining' => count( $queue ),
        'lines'     => array( array( 'type' => 'info', 'text' => '  ' . count( $queue ) . ' images queued. Processing ' . CSC_CHUNK_OPTIMISE . ' per request.' ) ),
    ) );
}

// Step 2: process a chunk
add_action( 'wp_ajax_csc_optimise_chunk', 'csc_ajax_optimise_chunk' );
function csc_ajax_optimise_chunk() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $queue = get_transient( 'csc_optimise_queue' );
    if ( ! is_array( $queue ) ) { wp_send_json_error( 'Session expired — please start again.' ); }

    $max_w       = intval( get_option( 'csc_img_max_width',  1920 ) );
    $max_h       = intval( get_option( 'csc_img_max_height', 1080 ) );
    $quality     = intval( get_option( 'csc_img_quality',    82 ) );
    $convert_png = get_option( 'csc_convert_png_to_jpg', '0' ) === '1';

    $chunk       = array_splice( $queue, 0, CSC_CHUNK_OPTIMISE );
    $total_saved = (int) get_transient( 'csc_optimise_saved' );
    $total_count = (int) get_transient( 'csc_optimise_count' );

    set_transient( 'csc_optimise_queue', $queue, 2 * HOUR_IN_SECONDS );

    $lines = array();

    foreach ( $chunk as $id ) {
        $file = get_attached_file( $id );
        if ( ! $file || ! file_exists( $file ) ) {
            $lines[] = array( 'type' => 'info', 'text' => '  Skipped ID ' . $id . ' — file not found.' );
            continue;
        }

        $mime     = mime_content_type( $file );
        $size_old = filesize( $file );
        $title    = get_the_title( $id );
        $dims     = @getimagesize( $file );
        if ( ! $dims ) {
            $lines[] = array( 'type' => 'info', 'text' => '  Skipped ID ' . $id . ' — cannot read image dimensions.' );
            continue;
        }
        list( $w, $h ) = $dims;

        $editor = wp_get_image_editor( $file );
        if ( is_wp_error( $editor ) ) {
            $lines[] = array( 'type' => 'error', 'text' => '  [ERROR] ID ' . $id . ': ' . $editor->get_error_message() );
            continue;
        }

        $editor->set_quality( $quality );

        if ( $w > $max_w || $h > $max_h ) {
            $editor->resize( $max_w, $max_h, false );
        }

        // PNG to JPEG conversion path
        if ( $convert_png && $mime === 'image/png' ) {
            $new_file = preg_replace( '/\.png$/i', '.jpg', $file );
            $result   = $editor->save( $new_file, 'image/jpeg' );
            if ( is_wp_error( $result ) ) {
                $lines[] = array( 'type' => 'error', 'text' => '  [ERROR] ID ' . $id . ' PNG→JPEG: ' . $result->get_error_message() );
                continue;
            }
            @unlink( $file );
            update_attached_file( $id, $new_file );
            $meta = wp_generate_attachment_metadata( $id, $new_file );
            wp_update_attachment_metadata( $id, $meta );
            wp_update_post( array( 'ID' => $id, 'post_mime_type' => 'image/jpeg' ) );
            csc_update_image_references( $id, $file, $new_file );
            $size_new = file_exists( $new_file ) ? filesize( $new_file ) : 0;
        } else {
            // Recompress / resize in place
            $result = $editor->save( $file );
            if ( is_wp_error( $result ) ) {
                $lines[] = array( 'type' => 'error', 'text' => '  [ERROR] ID ' . $id . ': ' . $result->get_error_message() );
                continue;
            }
            $meta = wp_generate_attachment_metadata( $id, $file );
            wp_update_attachment_metadata( $id, $meta );
            $size_new = file_exists( $file ) ? filesize( $file ) : $size_old;
        }

        $saved        = max( 0, $size_old - $size_new );
        $total_saved += $saved;
        $total_count++;

        $lines[] = array( 'type' => 'deleted', 'text' => '  [OPTIMISED] ID ' . $id . ' — ' . esc_html( $title ) . ' ' . size_format( $size_old ) . ' → ' . size_format( $size_new ) . ' (saved ' . size_format( $saved ) . ')' );
    }

    set_transient( 'csc_optimise_saved', $total_saved, 2 * HOUR_IN_SECONDS );
    set_transient( 'csc_optimise_count', $total_count, 2 * HOUR_IN_SECONDS );

    wp_send_json_success( array( 'remaining' => count( $queue ), 'total_saved' => $total_saved, 'lines' => $lines ) );
}

// Step 3: finish
add_action( 'wp_ajax_csc_optimise_finish', 'csc_ajax_optimise_finish' );
function csc_ajax_optimise_finish() {
    check_ajax_referer( 'csc_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error( 'Insufficient permissions.' ); }

    $total_saved = (int) get_transient( 'csc_optimise_saved' );
    $total_count = (int) get_transient( 'csc_optimise_count' );

    delete_transient( 'csc_optimise_queue' );
    delete_transient( 'csc_optimise_saved' );
    delete_transient( 'csc_optimise_count' );

    update_option( 'csc_last_img_optimise', current_time( 'mysql' ) );
    wp_send_json_success( array( 'lines' => array(
        array( 'type' => 'count',   'text' => '  Images processed: ' . $total_count ),
        array( 'type' => 'count',   'text' => '  Total disk space saved: ' . size_format( $total_saved ) ),
        array( 'type' => 'success', 'text' => 'Image optimisation complete.' ),
    ) ) );
}

// Helper: update image URL references after PNG → JPEG conversion
function csc_update_image_references( $attachment_id, $old_file, $new_file ) {
    global $wpdb;
    $upload_dir = wp_upload_dir();
    $base       = trailingslashit( $upload_dir['basedir'] );
    $old_url    = $upload_dir['baseurl'] . '/' . ltrim( str_replace( $base, '', $old_file ), '/' );
    $new_url    = $upload_dir['baseurl'] . '/' . ltrim( str_replace( $base, '', $new_file ), '/' );

    $wpdb->query( $wpdb->prepare(
        "UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s) WHERE post_content LIKE %s",
        $old_url, $new_url, '%' . $wpdb->esc_like( basename( $old_file ) ) . '%'
    ) );
    $wpdb->query( $wpdb->prepare( "UPDATE {$wpdb->posts} SET guid = %s WHERE ID = %d", $new_url, $attachment_id ) );
}

// ═════════════════════════════════════════════════════════════════════════════
// ADMIN PAGE
// ═════════════════════════════════════════════════════════════════════════════


// ─── Explain modal helper ────────────────────────────────────────────────────

function csc_explain_btn( string $id, string $title, array $items, string $color = 'rgba(255,255,255,0.2)' ): void {
    $btn_id   = 'csc-explain-btn-' . $id;
    $modal_id = 'csc-explain-modal-' . $id;
    ?>
    <button type="button" id="<?php echo esc_attr( $btn_id ); ?>"
        data-color="<?php echo esc_attr( $color ); ?>"
        onclick="document.getElementById('<?php echo esc_attr( $modal_id ); ?>').style.display='flex'"
        style="background:<?php echo esc_attr( $color ); ?>!important;border:1px solid rgba(255,255,255,0.5)!important;border-radius:5px!important;color:#fff!important;font-size:12px!important;font-weight:600!important;padding:5px 14px!important;cursor:pointer!important;margin-left:auto!important;flex-shrink:0!important;display:block!important;box-shadow:none!important;text-shadow:none!important;text-transform:none!important;letter-spacing:normal!important;line-height:1.4!important">
        Explain...
    </button>
    <div id="<?php echo esc_attr( $modal_id ); ?>" style="display:none;position:fixed;inset:0;z-index:99999;background:rgba(0,0,0,0.6);align-items:center;justify-content:center;padding:16px">
        <div style="background:#fff;border-radius:10px;max-width:640px;width:100%;max-height:88vh;overflow-y:auto;box-shadow:0 20px 60px rgba(0,0,0,0.4)">
            <div style="background:#1a2a3a;border-radius:10px 10px 0 0;padding:16px 20px;display:flex;justify-content:space-between;align-items:center">
                <strong style="color:#fff;font-size:15px"><?php echo esc_html( $title ); ?></strong>
                <button type="button" onclick="document.getElementById('<?php echo esc_attr( $modal_id ); ?>').style.display='none'"
                    style="background:rgba(255,255,255,0.2);border:1px solid rgba(255,255,255,0.4);border-radius:5px;color:#fff;font-size:16px;font-weight:700;padding:2px 10px;cursor:pointer;line-height:1">&#x2715;</button>
            </div>
            <div style="padding:20px 24px;font-size:13px;line-height:1.6;color:#1d2327">
                <?php foreach ( $items as $item ) :
                    $rec    = $item['rec'];
                    $is_on  = strpos( $rec, 'Recommended' ) !== false;
                    $is_opt = strpos( $rec, 'Optional' ) !== false;
                    $bg     = $is_on ? '#edfaef' : ( $is_opt ? '#f6f7f7' : '#f0f6fc' );
                    $col    = $is_on ? '#1a7a34' : ( $is_opt ? '#50575e' : '#1a4a7a' );
                    $bdr    = $is_on ? '#1a7a34' : ( $is_opt ? '#c3c4c7' : '#2271b1' );
                ?>
                <div style="border:1px solid #e0e0e0;border-radius:6px;padding:12px 14px;margin-bottom:10px">
                    <div style="display:flex;align-items:center;gap:10px;margin-bottom:5px;flex-wrap:wrap">
                        <strong style="font-size:13px"><?php echo esc_html( $item['name'] ); ?></strong>
                        <span style="background:<?php echo esc_attr( $bg ); ?>;color:<?php echo esc_attr( $col ); ?>;border:1px solid <?php echo esc_attr( $bdr ); ?>;border-radius:4px;font-size:11px;font-weight:600;padding:1px 8px;white-space:nowrap"><?php echo esc_html( $rec ); ?></span>
                    </div>
                    <p style="margin:0;color:#50575e;font-size:12px;line-height:1.5"><?php echo esc_html( $item['desc'] ); ?></p>
                </div>
                <?php endforeach; ?>
            </div>
            <div style="padding:12px 24px 20px;text-align:right">
                <button type="button" onclick="document.getElementById('<?php echo esc_attr( $modal_id ); ?>').style.display='none'"
                    style="background:#1a2a3a;border:none;border-radius:6px;color:#fff;font-size:13px;font-weight:600;padding:8px 24px;cursor:pointer">
                    Got it
                </button>
            </div>
        </div>
    </div>
    <?php
}

function csc_render_page() {
    $dow            = array( 'mon' => 'Mon', 'tue' => 'Tue', 'wed' => 'Wed', 'thu' => 'Thu', 'fri' => 'Fri', 'sat' => 'Sat', 'sun' => 'Sun' );
    $db_sched_days  = (array) get_option( 'csc_schedule_db_days',  array( 'mon', 'wed', 'fri' ) );
    $img_sched_days = (array) get_option( 'csc_schedule_img_days', array( 'mon', 'wed', 'fri' ) );
    ?>
    <div class="csc-wrap">

        <div class="csc-header">
            <div class="csc-header-inner">
                <div class="csc-header-title">
                    <span class="csc-logo">⚡</span>
                    <div>
                        <h1>CloudScale Cleanup</h1>
                        <p>Database and Media Library Cleanup &middot; Free and Open Source &middot; <a href="https://andrewbaker.ninja" target="_blank">andrewbaker.ninja</a></p>
                    </div>
                </div>
                <div class="csc-header-version">v<?php echo esc_html( CLOUDSCALE_CLEANUP_VERSION ); ?></div>
            </div>
        </div>

        <div class="csc-tabs">
            <button class="csc-tab active" data-tab="db-cleanup">Database Cleanup</button>
            <button class="csc-tab" data-tab="img-cleanup">Image Cleanup</button>
            <button class="csc-tab" data-tab="img-optimise">Image Optimisation</button>
            <button class="csc-tab" data-tab="settings">Settings</button>
        </div>

        <!-- ═══ Database Cleanup ═══ -->
        <div class="csc-tab-content active" id="tab-db-cleanup">
            <div class="csc-cards-row">
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-blue"><span>Database Cleanup</span> <?php csc_explain_btn(
            'db-cleanup',
            'Database Cleanup — What it does',
            [
            [ 'rec' => '✅ Recommended', 'name' => 'Post Revisions', 'desc' => 'Every time you save or update a post, WordPress stores a complete copy. On an active blog this can mean hundreds of revision rows per post. They consume significant database space over time.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Draft Posts', 'desc' => 'Posts saved as drafts but never published. The threshold controls how old a draft must be before deletion — fresh drafts are never touched.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Trashed Posts', 'desc' => 'Posts moved to the trash. WordPress keeps them indefinitely by default. This removes them permanently after the configured number of days.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Auto-Drafts', 'desc' => 'WordPress creates an auto-draft record when you open Add New Post. If you navigate away without saving, the empty record remains. These accumulate silently.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Expired Transients', 'desc' => 'Temporary cached values stored in your options table by plugins and themes. After expiry WordPress should delete them, but many accumulate. Completely safe to delete.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Orphaned Post Meta', 'desc' => 'Post meta rows referencing a post ID that no longer exists. Left behind when posts are deleted without their metadata being cleaned up.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'Orphaned User Meta', 'desc' => 'Metadata rows referencing deleted user accounts. Accumulates when users are removed from the system.' ],
            ],
            '#00e676'
        ); ?></div>
                    <div class="csc-card-body">
                        <p class="csc-options-intro">Select which items to include in every cleanup run. Toggle off anything you want to preserve.</p>
                        <div class="csc-options-grid" id="csc-db-toggles">
                            <?php
                            $db_toggles = array(
                                'csc_clean_revisions'      => array( 'Post Revisions',    'Old revision copies saved every time a post is edited. On active blogs these accumulate into thousands of rows.' ),
                                'csc_clean_drafts'         => array( 'Draft Posts',        'Unpublished drafts older than the configured threshold. Fresh drafts are never touched.' ),
                                'csc_clean_trashed'        => array( 'Trashed Posts',      'Posts in the WordPress trash older than the threshold. WordPress keeps them indefinitely by default.' ),
                                'csc_clean_autodrafts'     => array( 'Auto-Drafts',        'Empty placeholder records left when the editor is abandoned without saving.' ),
                                'csc_clean_transients'     => array( 'Expired Transients', 'Stale cached values stored in wp_options past their expiry date. Completely safe to delete.' ),
                                'csc_clean_orphan_post'    => array( 'Orphaned Post Meta', 'Post meta rows whose parent post has been deleted. Left behind when posts are removed without proper cleanup.' ),
                                'csc_clean_orphan_user'    => array( 'Orphaned User Meta', 'Meta rows referencing deleted user accounts. Accumulates when users are removed from the system.' ),
                                'csc_clean_spam_comments'  => array( 'Spam Comments',      'Comments flagged as spam older than the configured threshold. Safe to remove after the review window.' ),
                                'csc_clean_trash_comments' => array( 'Trashed Comments',   'Comments moved to the WordPress comment trash older than the threshold.' ),
                            );
                            foreach ( $db_toggles as $opt => $info ) :
                                $is_on = get_option( $opt, '1' ) === '1';
                            ?>
                            <div class="csc-option-row" style="display:flex;align-items:center;justify-content:space-between;gap:16px;padding:10px 14px;border-radius:6px;background:#fff;border:1px solid transparent;transition:background 0.12s;"
                                 onmouseover="this.style.background='#f0f6fc';this.style.borderColor='#c5d9f0'"
                                 onmouseout="this.style.background='#fff';this.style.borderColor='transparent'">
                                <div style="display:flex;flex-direction:column;gap:2px;flex:1;min-width:0;">
                                    <span style="font-size:13px;font-weight:700;color:#1d2327;"><?php echo esc_html( $info[0] ); ?></span>
                                    <span style="font-size:12px;color:#787c82;line-height:1.5;"><?php echo esc_html( $info[1] ); ?></span>
                                </div>
                                <!-- Hidden input holds the real value for form submission -->
                                <input type="hidden" name="<?php echo esc_attr( $opt ); ?>" value="<?php echo $is_on ? '1' : '0'; ?>" data-csc-toggle="1">
                                <!-- Pure-div toggle — zero CSS class dependencies, all inline -->
                                <div data-csc-toggle-track="1"
                                     data-on="<?php echo $is_on ? '1' : '0'; ?>"
                                     onclick="cscToggle(this)"
                                     style="position:relative;display:inline-block;width:44px;height:24px;min-width:44px;border-radius:24px;background:<?php echo $is_on ? '#00a32a' : '#c3c4c7'; ?>;cursor:pointer;transition:background 0.2s;flex-shrink:0;">
                                    <span style="position:absolute;top:3px;left:<?php echo $is_on ? '23px' : '3px'; ?>;width:18px;height:18px;background:#fff;border-radius:50%;box-shadow:0 1px 4px rgba(0,0,0,0.3);transition:left 0.2s;"></span>
                                </div>
                            </div>
                            <?php endforeach; ?>
                        </div>
                        <script>
                        function cscToggle(track) {
                            var isOn = track.getAttribute('data-on') === '1';
                            var newOn = !isOn;
                            track.setAttribute('data-on', newOn ? '1' : '0');
                            track.style.background = newOn ? '#00a32a' : '#c3c4c7';
                            track.querySelector('span').style.left = newOn ? '23px' : '3px';
                            // Update the hidden input
                            var row = track.parentNode;
                            var hidden = row.querySelector('input[type="hidden"][data-csc-toggle]');
                            if (hidden) hidden.value = newOn ? '1' : '0';
                        }
                        </script>
                        <div class="csc-button-row" style="margin-top:18px">
                            <button class="csc-btn csc-btn-primary csc-save-btn" data-group="db-types">Save Selection</button>
                            <button class="csc-btn csc-btn-secondary" id="btn-scan-db">🔍 Dry Run — Preview</button>
                            <button class="csc-btn csc-btn-danger"    id="btn-run-db">🗑 Run Cleanup Now</button>
                        </div>
                        <div class="csc-progress-outer" id="db-progress-outer" style="display:none">
                            <div class="csc-progress-bar"><div class="csc-progress-fill" id="db-progress-fill"></div></div>
                            <div class="csc-progress-label" id="db-progress-label">Preparing…</div>
                        </div>
                    </div>
                </div>
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-teal"><span>Cleanup Thresholds</span> <?php csc_explain_btn(
            'thresholds',
            'Cleanup Thresholds — What each setting means',
            [
                [ 'rec' => 'ℹ️ Info', 'name' => 'Why thresholds exist', 'desc' => 'Every threshold prevents the cleanup from touching items that are too recent to be considered safe. This protects you from accidentally deleting a draft you were actively working on or a comment that arrived in spam yesterday.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Post revisions older than N days', 'desc' => 'Only revisions created more than N days ago are deleted. Very recent revisions are kept so you can still roll back recent edits. Default: 30 days.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Draft posts older than N days', 'desc' => 'Only posts in Draft status whose last-modified date is older than N days are deleted. A draft you edited yesterday will never be touched. Default: 90 days.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Trashed posts older than N days', 'desc' => 'Posts moved to the WordPress trash older than N days are permanently deleted. Default: 30 days.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Auto-drafts older than N days', 'desc' => 'The empty placeholder records WordPress creates when you open the editor. These are nearly always safe to delete immediately — a threshold of 7 days gives you a buffer. Default: 7 days.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Spam comments older than N days', 'desc' => 'Comments flagged as spam by Akismet or manually. Keeping them for 30 days lets you review false positives before permanent deletion. Default: 30 days.' ],
                [ 'rec' => '✅ Recommended', 'name' => 'Trashed comments older than N days', 'desc' => 'Comments you have manually moved to the WordPress comment trash. The threshold gives you a safety window to change your mind. Default: 30 days.' ],
            ],
            '#ff6d00'
        ); ?></div>
                    <div class="csc-card-body csc-settings-inline">
                        <label>Post revisions older than   <input type="number" class="csc-setting" name="csc_post_revisions_age" value="<?php echo esc_attr( get_option( 'csc_post_revisions_age', 30 ) ); ?>" min="1"> days</label>
                        <label>Draft posts older than       <input type="number" class="csc-setting" name="csc_drafts_age"         value="<?php echo esc_attr( get_option( 'csc_drafts_age',         90 ) ); ?>" min="1"> days</label>
                        <label>Trashed posts older than     <input type="number" class="csc-setting" name="csc_trash_age"          value="<?php echo esc_attr( get_option( 'csc_trash_age',          30 ) ); ?>" min="1"> days</label>
                        <label>Auto-drafts older than       <input type="number" class="csc-setting" name="csc_autodraft_age"      value="<?php echo esc_attr( get_option( 'csc_autodraft_age',       7 ) ); ?>" min="1"> days</label>
                        <label>Spam comments older than     <input type="number" class="csc-setting" name="csc_spam_comments_age"  value="<?php echo esc_attr( get_option( 'csc_spam_comments_age',  30 ) ); ?>" min="1"> days</label>
                        <label>Trashed comments older than  <input type="number" class="csc-setting" name="csc_trash_comments_age" value="<?php echo esc_attr( get_option( 'csc_trash_comments_age', 30 ) ); ?>" min="1"> days</label>
                        <button class="csc-btn csc-btn-primary csc-save-btn" data-group="db">Save Thresholds</button>
                    </div>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-slate-db"><span>Scheduled Database Cleanup</span> <?php csc_explain_btn(
            'db-schedule',
            'Scheduled Database Cleanup — How it works',
            [
            [ 'rec' => 'ℹ️ Info', 'name' => 'What scheduling does', 'desc' => 'When enabled, the plugin registers a WordPress Cron job that automatically runs the full database cleanup on the selected days at the configured hour — without you having to log in manually.' ],
            [ 'rec' => '⬜ Optional', 'name' => 'Day and hour selection', 'desc' => 'You can select multiple days per week. The hour is in your server local time, not your browser timezone. Most VPS hosts default to UTC.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'How WordPress Cron works', 'desc' => 'WordPress Cron is triggered by page visits, not a real system clock. On low-traffic sites a job scheduled for 3AM may not run until the first visitor arrives. For precise scheduling, disable WP-Cron in wp-config.php and add a real server cron.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'After each scheduled run', 'desc' => 'The plugin automatically schedules the next run after each execution, so the schedule remains active indefinitely without any manual intervention.' ],
            ],
            '#f48fb1'
        ); ?></div>
                <div class="csc-card-body">
                    <label class="csc-toggle-label">
                        <input type="checkbox" name="csc_schedule_db_enabled" value="1" <?php checked( get_option( 'csc_schedule_db_enabled', '0' ), '1' ); ?>>
                        Enable automatic scheduled cleanup
                    </label>
                    <div class="csc-schedule-row">
                        <?php foreach ( $dow as $val => $label ) : ?>
                        <label class="csc-day-label">
                            <input type="checkbox" name="csc_schedule_db_days[]" value="<?php echo esc_attr( $val ); ?>" <?php checked( in_array( $val, $db_sched_days, true ), true ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </label>
                        <?php endforeach; ?>
                        <label class="csc-hour-label">at hour <input type="number" name="csc_schedule_db_hour" class="csc-small-num" value="<?php echo esc_attr( get_option( 'csc_schedule_db_hour', 3 ) ); ?>" min="0" max="23"> (server time)</label>
                    </div>
                    <button class="csc-btn csc-btn-primary csc-save-btn" data-group="db-schedule">Save Schedule</button>
                    <?php $next = wp_next_scheduled( 'csc_scheduled_db_cleanup' ); if ( $next ) { echo '<p class="csc-next-run">Next run: ' . esc_html( date_i18n( 'D j M Y H:i', $next ) ) . '</p>'; } ?>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-dark">Output Log</div>
                <div class="csc-card-body csc-terminal-wrap">
                    <div style="display:flex;align-items:center;gap:6px;padding:4px 12px;background:#0d1b2a;border-bottom:2px solid #00e5ff;border-radius:6px 6px 0 0"><span style="width:7px;height:7px;border-radius:50%;background:#00e5ff;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#00e5ff;opacity:.5;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#00e5ff;opacity:.25;display:inline-block;flex-shrink:0"></span><span style="margin-left:8px;background:#00e5ff;color:#0d1b2a;font-family:monospace;font-size:10px;font-weight:800;letter-spacing:.1em;padding:2px 10px;border-radius:20px;text-transform:uppercase">⚙ Database Console</span></div>
                    <pre class="csc-terminal" id="db-terminal">Ready. Press Dry Run to preview or Run Cleanup Now to execute.</pre>
                </div>
            </div>
        </div>

        <!-- ═══ Image Cleanup ═══ -->
        <div class="csc-tab-content" id="tab-img-cleanup">
            <div class="csc-cards-row">
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-purple"><span>Unused Image Cleanup</span> <?php csc_explain_btn(
            'unused-images',
            'Unused Image Cleanup — How images are detected',
            [
            [ 'rec' => 'ℹ️ Info', 'name' => 'What unused means', 'desc' => 'An attachment is considered unused if it cannot be found in post content, featured images, widget settings, theme mods, the site logo, or the site icon. The site logo and icon are always protected.' ],
            [ 'rec' => '⬜ Optional', 'name' => 'What is always protected', 'desc' => 'The site logo and site icon set in Appearance Customize are never flagged as unused, regardless of whether they appear in post content.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'What gets deleted', 'desc' => 'wp_delete_attachment() is called for each unused ID. This removes the database record and all associated files on disk — the original upload plus every generated thumbnail size.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'Chunked processing', 'desc' => 'Deletions are processed in batches of 25 per request so the operation never risks hitting PHP timeout limits, even on shared hosting with libraries of thousands of images.' ],
            ],
            '#00e5ff'
        ); ?></div>
                    <div class="csc-card-body">
                        <p>Identifies media library attachments not referenced in any post content, featured image, widget, or theme option. The site logo and site icon are always preserved regardless of reference status. Deletions are chunked at <?php echo CSC_CHUNK_IMAGES; ?> per request with a live progress bar.</p>
                        <div class="csc-button-row">
                            <button class="csc-btn csc-btn-secondary" id="btn-scan-img">🔍 Dry Run — Preview</button>
                            <button class="csc-btn csc-btn-danger"    id="btn-run-img">🗑 Delete Unused Images</button>
                        </div>
                        <div class="csc-progress-outer" id="img-progress-outer" style="display:none">
                            <div class="csc-progress-bar"><div class="csc-progress-fill" id="img-progress-fill"></div></div>
                            <div class="csc-progress-label" id="img-progress-label">Preparing…</div>
                        </div>
                    </div>
                </div>
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-amber"><span>Orphaned Filesystem Files</span> <?php csc_explain_btn(
            'orphan-files',
            'Orphaned Filesystem Files — What they are',
            [
            [ 'rec' => 'ℹ️ Info', 'name' => 'What an orphaned file is', 'desc' => 'An orphaned file exists physically on disk inside wp-content/uploads but has no corresponding WordPress attachment record in the database.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'How they accumulate', 'desc' => 'Files uploaded via FTP without being registered in WordPress, partially completed uploads, images imported without the media importer, or files left behind by deleted plugins.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'The recycle workflow', 'desc' => 'Scan finds orphans. Move to Recycle places them in wp-content/uploads/.csc-recycle/ with a manifest recording their original paths. Restore moves them back exactly. Permanently Delete wipes the recycle bin.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'What the scan checks', 'desc' => 'Every image file found recursively under the uploads directory is checked against files registered in _wp_attached_file and _wp_attachment_metadata. Files not in either set are reported as orphans.' ],
            ],
            '#ff4081'
        ); ?></div>
                    <div class="csc-card-body">
                        <p>Scans the uploads directory for files that exist on disk but have no corresponding WordPress attachment record. Use the recycle workflow to safely remove them.</p>
                        <script>
                        var cscPillOff = 'display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #c3c4c7;font-size:12px;font-weight:700;cursor:pointer;background:#fff;color:#50575e;margin:0 4px 4px 0';
                        var cscPillOn  = 'display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #00a32a;font-size:12px;font-weight:700;cursor:pointer;background:#00a32a;color:#fff;margin:0 4px 4px 0';
                        window.cscOrphanTypes = [];
                        function cscOrphanToggle(el, type) {
                            var idx = window.cscOrphanTypes.indexOf(type);
                            if (idx === -1) {
                                window.cscOrphanTypes.push(type);
                                el.style.cssText = cscPillOn;
                            } else {
                                window.cscOrphanTypes.splice(idx, 1);
                                el.style.cssText = cscPillOff;
                            }
                            var joined = window.cscOrphanTypes.join(',');
                            document.getElementById('btn-scan-orphan').setAttribute('data-ftype', joined);
                            document.getElementById('btn-recycle-orphan').setAttribute('data-ftype', joined);
                        }
                        </script>
                        <div style="display:flex;gap:0;flex-wrap:wrap;margin-bottom:14px">
                            <span class="csc-orphan-pill" onclick="cscOrphanToggle(this,'images')"    style="display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #c3c4c7;font-size:12px;font-weight:700;cursor:pointer;background:#fff;color:#50575e;margin:0 4px 4px 0">🖼 Images</span>
                            <span class="csc-orphan-pill" onclick="cscOrphanToggle(this,'documents')" style="display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #c3c4c7;font-size:12px;font-weight:700;cursor:pointer;background:#fff;color:#50575e;margin:0 4px 4px 0">📄 Documents</span>
                            <span class="csc-orphan-pill" onclick="cscOrphanToggle(this,'video')"     style="display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #c3c4c7;font-size:12px;font-weight:700;cursor:pointer;background:#fff;color:#50575e;margin:0 4px 4px 0">🎬 Video</span>
                            <span class="csc-orphan-pill" onclick="cscOrphanToggle(this,'audio')"     style="display:inline-block;padding:6px 14px;border-radius:20px;border:2px solid #c3c4c7;font-size:12px;font-weight:700;cursor:pointer;background:#fff;color:#50575e;margin:0 4px 4px 0">🎵 Audio</span>
                        </div>
                        <div class="csc-button-row" style="flex-wrap:wrap;gap:10px">
                            <button class="csc-btn csc-btn-secondary" id="btn-scan-orphan">🔍 Scan Orphan Files</button>
                            <button class="csc-btn csc-btn-danger"    id="btn-recycle-orphan">♻️ Move to Recycle</button>
                        </div>
                        <div id="orphan-recycle-actions" style="margin-top:12px;padding:12px 14px;background:#fff8e1;border:1px solid #ffe082;border-radius:6px">
                            <strong style="font-size:12px;color:#5d4037">♻️ Recycle Bin</strong>
                            <span id="orphan-recycle-count" style="font-size:12px;color:#5d4037;margin-left:6px">— checking…</span>
                            <div style="margin-top:10px;display:flex;gap:10px;flex-wrap:wrap">
                                <button class="csc-btn" id="btn-restore-orphan"
                                    style="background:#43a047;color:#fff;border:none;border-radius:6px;padding:7px 18px;font-size:13px;font-weight:600;cursor:pointer">
                                    ↩️ Restore All
                                </button>
                                <button class="csc-btn" id="btn-purge-orphan"
                                    style="background:#b71c1c;color:#fff;border:none;border-radius:6px;padding:7px 18px;font-size:13px;font-weight:600;cursor:pointer">
                                    🗑 Permanently Delete
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-slate-img"><span>Scheduled Image Cleanup</span> <?php csc_explain_btn(
            'img-schedule',
            'Scheduled Image Cleanup — How it works',
            [
            [ 'rec' => 'ℹ️ Info', 'name' => 'What it does', 'desc' => 'Runs the unused image cleanup automatically on the selected days and hour. The same detection logic as the manual cleanup is used — the site logo and site icon are always protected.' ],
            [ 'rec' => '⬜ Optional', 'name' => 'Use with caution on active sites', 'desc' => 'Automated image cleanup is most appropriate for sites where images are always attached to posts via the standard WordPress editor. Review a manual dry run before enabling the schedule.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'After each run', 'desc' => 'The next scheduled run is automatically registered, so the schedule stays active indefinitely.' ],
            ],
            '#69f0ae'
        ); ?></div>
                <div class="csc-card-body">
                    <label class="csc-toggle-label">
                        <input type="checkbox" name="csc_schedule_img_enabled" value="1" <?php checked( get_option( 'csc_schedule_img_enabled', '0' ), '1' ); ?>>
                        Enable automatic scheduled cleanup
                    </label>
                    <div class="csc-schedule-row">
                        <?php foreach ( $dow as $val => $label ) : ?>
                        <label class="csc-day-label">
                            <input type="checkbox" name="csc_schedule_img_days[]" value="<?php echo esc_attr( $val ); ?>" <?php checked( in_array( $val, $img_sched_days, true ), true ); ?>>
                            <?php echo esc_html( $label ); ?>
                        </label>
                        <?php endforeach; ?>
                        <label class="csc-hour-label">at hour <input type="number" name="csc_schedule_img_hour" class="csc-small-num" value="<?php echo esc_attr( get_option( 'csc_schedule_img_hour', 4 ) ); ?>" min="0" max="23"> (server time)</label>
                    </div>
                    <button class="csc-btn csc-btn-primary csc-save-btn" data-group="img-schedule">Save Schedule</button>
                    <?php $next = wp_next_scheduled( 'csc_scheduled_img_cleanup' ); if ( $next ) { echo '<p class="csc-next-run">Next run: ' . esc_html( date_i18n( 'D j M Y H:i', $next ) ) . '</p>'; } ?>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-dark">Output Log</div>
                <div class="csc-card-body csc-terminal-wrap">
                    <div style="display:flex;align-items:center;gap:6px;padding:4px 12px;background:#1a0533;border-bottom:2px solid #e040fb;border-radius:6px 6px 0 0"><span style="width:7px;height:7px;border-radius:50%;background:#e040fb;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#e040fb;opacity:.5;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#e040fb;opacity:.25;display:inline-block;flex-shrink:0"></span><span style="margin-left:8px;background:#e040fb;color:#fff;font-family:monospace;font-size:10px;font-weight:800;letter-spacing:.1em;padding:2px 10px;border-radius:20px;text-transform:uppercase">🖼 Image Console</span></div>
                    <pre class="csc-terminal" id="img-terminal">Ready. Press Dry Run to preview or Delete Unused Images to execute.</pre>
                </div>
            </div>
        </div>

        <!-- ═══ Image Optimisation ═══ -->
        <div class="csc-tab-content" id="tab-img-optimise">
            <div class="csc-cards-row">
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-red"><span>Image Optimisation</span> <?php csc_explain_btn(
            'img-optimise',
            'Image Optimisation — How it works',
            [
            [ 'rec' => 'ℹ️ Info', 'name' => 'What it does', 'desc' => 'Processes your original uploaded images in two ways: Resize (scales down images exceeding your configured maximum, preserving aspect ratio) and Recompress (re-saves JPEG files at the configured quality level).' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'Thumbnail regeneration', 'desc' => 'After each image is processed, all registered WordPress thumbnail sizes are regenerated from the new optimised original to ensure consistency.' ],
            [ 'rec' => '⬜ Optional', 'name' => 'PNG to JPEG conversion', 'desc' => 'When enabled, PNG files without transparency are converted to JPEG. Photographic PNGs typically shrink by 40-70% when converted. PNG files with transparency are never converted.' ],
            [ 'rec' => 'ℹ️ Info', 'name' => 'Chunked processing', 'desc' => 'Images are processed 5 at a time per request to keep each request well under 30 seconds. Always take a full site backup before running on a production site.' ],
            ],
            '#ff1744'
        ); ?></div>
                    <div class="csc-card-body">
                        <p>Resizes oversized originals and recompresses JPEGs to the configured quality target. Processes <?php echo CSC_CHUNK_OPTIMISE; ?> images per request — chunked to stay well within any server's PHP timeout limit regardless of media library size. All WordPress thumbnail sizes are regenerated after each image is processed.</p>
                        <div class="csc-button-row">
                            <button class="csc-btn csc-btn-secondary" id="btn-scan-optimise">🔍 Dry Run — Preview Savings</button>
                            <button class="csc-btn csc-btn-danger"    id="btn-run-optimise">⚡ Optimise Images Now</button>
                        </div>
                        <div class="csc-progress-outer" id="opt-progress-outer" style="display:none">
                            <div class="csc-progress-bar"><div class="csc-progress-fill" id="opt-progress-fill"></div></div>
                            <div class="csc-progress-label" id="opt-progress-label">Preparing…</div>
                        </div>
                        <p class="csc-note">This modifies original image files on disk. Take a full backup first. WordPress 5.3+ preserves a scaled backup original automatically.</p>
                    </div>
                </div>
                <div class="csc-card">
                    <div class="csc-card-header csc-card-header-green"><span>Optimisation Settings</span> <?php csc_explain_btn(
            'opt-settings',
            'Optimisation Settings — What each option does',
            [
            [ 'rec' => '✅ Recommended', 'name' => 'Maximum width and height (px)', 'desc' => 'Any image whose width or height exceeds the maximum will be scaled down proportionally. Default: 1920x1080. If your theme never displays images wider than 1200px, setting the max to 1200 will produce better storage savings.' ],
            [ 'rec' => '✅ Recommended', 'name' => 'JPEG quality (1-100)', 'desc' => 'Controls compression when saving JPEG files. 80-85 is the sweet spot — excellent quality with significant size reduction. Default: 82.' ],
            [ 'rec' => '⬜ Optional', 'name' => 'Convert non-transparent PNGs to JPEG', 'desc' => 'Converts eligible PNGs (those with no transparency) to JPEG. A PNG screenshot at 200KB will typically become a 40-80KB JPEG with no visible difference at screen resolution.' ],
            ],
            '#ffd600'
        ); ?></div>
                    <div class="csc-card-body csc-settings-inline">
                        <label>Maximum width (px)   <input type="number" class="csc-setting" name="csc_img_max_width"  value="<?php echo esc_attr( get_option( 'csc_img_max_width',  1920 ) ); ?>" min="200"></label>
                        <label>Maximum height (px)  <input type="number" class="csc-setting" name="csc_img_max_height" value="<?php echo esc_attr( get_option( 'csc_img_max_height', 1080 ) ); ?>" min="200"></label>
                        <label>JPEG quality (1–100) <input type="number" class="csc-setting" name="csc_img_quality"    value="<?php echo esc_attr( get_option( 'csc_img_quality',    82   ) ); ?>" min="1" max="100"></label>
                        <label class="csc-toggle-label">
                            <input type="checkbox" name="csc_convert_png_to_jpg" value="1" <?php checked( get_option( 'csc_convert_png_to_jpg', '1' ), '1' ); ?>>
                            Convert non-transparent PNGs to JPEG
                        </label>
                        <p class="csc-note">Recommended defaults: 1920&times;1080 · quality 82. PNG conversion yields 40–70% size reduction on photographic images.</p>
                        <button class="csc-btn csc-btn-primary csc-save-btn" data-group="optimise">Save Settings</button>
                    </div>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-dark">Output Log</div>
                <div class="csc-card-body csc-terminal-wrap">
                    <div style="display:flex;align-items:center;gap:6px;padding:4px 12px;background:#0a1f00;border-bottom:2px solid #76ff03;border-radius:6px 6px 0 0"><span style="width:7px;height:7px;border-radius:50%;background:#76ff03;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#76ff03;opacity:.5;display:inline-block;flex-shrink:0"></span><span style="width:7px;height:7px;border-radius:50%;background:#76ff03;opacity:.25;display:inline-block;flex-shrink:0"></span><span style="margin-left:8px;background:#76ff03;color:#0a1f00;font-family:monospace;font-size:10px;font-weight:800;letter-spacing:.1em;padding:2px 10px;border-radius:20px;text-transform:uppercase">⚡ Optimisation Console</span></div>
                    <pre class="csc-terminal" id="optimise-terminal">Ready. Press Dry Run to preview savings or Optimise Images Now to execute.</pre>
                </div>
            </div>
        </div>

        <!-- ═══ Settings ═══ -->
        <div class="csc-tab-content" id="tab-settings">
            <div class="csc-card">
                <div class="csc-card-header csc-card-header-blue">About CloudScale Cleanup</div>
                <div class="csc-card-body csc-about">
                    <p><strong>CloudScale Cleanup</strong> is a free, open source WordPress plugin by <a href="https://andrewbaker.ninja" target="_blank">Andrew Baker</a> — Chief Information Officer at Capitec Bank and author of a popular technology blog covering cloud architecture, banking technology, and enterprise software.</p>
                    <p>No accounts. No API keys. No subscriptions. No data leaves your server. All processing uses standard WordPress APIs.</p>
                    <div class="csc-about-links">
                        <a href="https://andrewbaker.ninja" target="_blank" class="csc-btn csc-btn-primary">andrewbaker.ninja</a>
                        <a href="https://andrewbaker.ninja/2026/02/24/cloudscale-free-backup-and-restore-a-wordpress-backup-plugin-that-does-exactly-what-it-says/" target="_blank" class="csc-btn csc-btn-secondary">CloudScale Backup Plugin</a>
                        <a href="https://andrewbaker.ninja/2026/02/24/cloudscale-seo-ai-optimiser-enterprise-grade-wordpress-seo-completely-free/" target="_blank" class="csc-btn csc-btn-secondary">CloudScale SEO Plugin</a>
                    </div>
                    <div class="csc-stats-grid">
                        <div class="csc-stat-box">
                            <div class="csc-stat-label">Last DB Cleanup</div>
                            <div class="csc-stat-value"><?php echo esc_html( get_option( 'csc_last_db_cleanup', 'Never' ) ); ?></div>
                        </div>
                        <div class="csc-stat-box">
                            <div class="csc-stat-label">Last Image Cleanup</div>
                            <div class="csc-stat-value"><?php echo esc_html( get_option( 'csc_last_img_cleanup', 'Never' ) ); ?></div>
                        </div>
                        <div class="csc-stat-box">
                            <div class="csc-stat-label">Last Optimisation</div>
                            <div class="csc-stat-value"><?php echo esc_html( get_option( 'csc_last_img_optimise', 'Never' ) ); ?></div>
                        </div>
                        <div class="csc-stat-box">
                            <div class="csc-stat-label">Version</div>
                            <div class="csc-stat-value"><?php echo esc_html( CLOUDSCALE_CLEANUP_VERSION ); ?></div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="csc-card">
                <div class="csc-card-header csc-card-header-teal">WordPress Cron Status</div>
                <div class="csc-card-body">
                    <?php
                    $next_db  = wp_next_scheduled( 'csc_scheduled_db_cleanup' );
                    $next_img = wp_next_scheduled( 'csc_scheduled_img_cleanup' );
                    $db_en    = get_option( 'csc_schedule_db_enabled',  '0' ) === '1';
                    $img_en   = get_option( 'csc_schedule_img_enabled', '0' ) === '1';
                    ?>
                    <table class="csc-status-table">
                        <tr>
                            <td>DB Cleanup</td>
                            <td><span class="csc-badge <?php echo $db_en  ? 'csc-badge-green' : 'csc-badge-grey'; ?>"><?php echo $db_en  ? 'Enabled' : 'Disabled'; ?></span></td>
                            <td><?php echo $next_db  ? esc_html( 'Next: ' . date_i18n( 'D j M Y H:i', $next_db  ) ) : '—'; ?></td>
                        </tr>
                        <tr>
                            <td>Image Cleanup</td>
                            <td><span class="csc-badge <?php echo $img_en ? 'csc-badge-green' : 'csc-badge-grey'; ?>"><?php echo $img_en ? 'Enabled' : 'Disabled'; ?></span></td>
                            <td><?php echo $next_img ? esc_html( 'Next: ' . date_i18n( 'D j M Y H:i', $next_img ) ) : '—'; ?></td>
                        </tr>
                    </table>
                    <p class="csc-note">WordPress Cron fires when someone visits your site. On low-traffic sites scheduled jobs may run a few minutes after the configured time. For exact timing, add a real server cron job calling <code>wp-cron.php</code> directly.</p>
                </div>
            </div>
        </div>

        <div id="csc-save-notice" class="csc-save-notice" style="display:none">Settings saved.</div>

    </div>
    <?php
}

