<?php
/**
 * CloudScale Page Views - Dashboard Widget  v2.0.1
 *
 * WordPress admin dashboard widget showing:
 *   - Today's view count + delta vs yesterday
 *   - Last 7 days total
 *   - Time-period chart: 7 Hours / 7 Days / 1 Month / 6 Months
 *   - Top 5 posts for selected period
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

add_action( 'wp_dashboard_setup', 'cspv_register_dashboard_widget' );

function cspv_register_dashboard_widget() {
    wp_add_dashboard_widget(
        'cspv_dashboard_widget',
        '☁ CloudScale Page Views',
        'cspv_render_dashboard_widget',
        null,
        null,
        'normal',
        'high'
    );
}

function cspv_render_dashboard_widget() {
    global $wpdb;
    $table = $wpdb->prefix . 'cspv_views';

    $table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table ) );

    $today   = current_time( 'Y-m-d' );
    $today_s = $today . ' 00:00:00';
    $today_e = $today . ' 23:59:59';
    $yest    = date( 'Y-m-d', strtotime( '-1 day', strtotime( $today ) ) );
    $yest_s  = $yest . ' 00:00:00';
    $yest_e  = $yest . ' 23:59:59';
    $week_s  = date( 'Y-m-d', strtotime( '-6 days', strtotime( $today ) ) ) . ' 00:00:00';

    $today_views = 0;
    $yest_views  = 0;
    $week_views  = 0;
    $top_today   = array();

    if ( $table_exists ) {
        $today_views = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table}` WHERE viewed_at BETWEEN %s AND %s",
            $today_s, $today_e ) );

        $yest_views = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table}` WHERE viewed_at BETWEEN %s AND %s",
            $yest_s, $yest_e ) );

        $week_views = (int) $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM `{$table}` WHERE viewed_at >= %s", $week_s ) );

        $top_today = $wpdb->get_results( $wpdb->prepare(
            "SELECT post_id, COUNT(*) AS views FROM `{$table}`
             WHERE viewed_at BETWEEN %s AND %s
             GROUP BY post_id ORDER BY views DESC LIMIT 5",
            $today_s, $today_e ) );
    }

    // Delta badge
    $delta_html = '';
    if ( $yest_views > 0 ) {
        $delta = $today_views - $yest_views;
        $pct   = round( ( $delta / $yest_views ) * 100 );
        $arrow = $delta >= 0 ? '↑' : '↓';
        $color = $delta >= 0 ? '#1db954' : '#e53e3e';
        $delta_html = '<span style="font-size:11px;color:' . $color . ';font-weight:700;margin-left:6px;white-space:nowrap;">'
                    . $arrow . ' ' . abs( $pct ) . '% vs yesterday</span>';
    } elseif ( $today_views > 0 ) {
        $delta_html = '<span style="font-size:11px;color:#1db954;font-weight:700;margin-left:6px;">New views today</span>';
    }

    // Build all four period datasets in PHP so no AJAX needed
    // 7 Hours: last 7 complete hours + current hour
    $hour_labels = array();
    $hour_values = array();
    $now_hour    = (int) current_time( 'G' ); // 0-23
    for ( $h = 6; $h >= 0; $h-- ) {
        $hr       = ( $now_hour - $h + 24 ) % 24;
        $label    = sprintf( '%02d:00', $hr );
        $hour_labels[] = $label;
        if ( $table_exists ) {
            $hr_s = current_time( 'Y-m-d' ) . ' ' . sprintf( '%02d:00:00', $hr );
            $hr_e = current_time( 'Y-m-d' ) . ' ' . sprintf( '%02d:59:59', $hr );
            // Handle midnight wrap: if hour rolled back to yesterday
            if ( $h > $now_hour ) {
                $hr_s = $yest . ' ' . sprintf( '%02d:00:00', $hr );
                $hr_e = $yest . ' ' . sprintf( '%02d:59:59', $hr );
            }
            $hour_values[] = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table}` WHERE viewed_at BETWEEN %s AND %s",
                $hr_s, $hr_e ) );
        } else {
            $hour_values[] = 0;
        }
    }

    // 7 Days
    $day7_labels = array();
    $day7_values = array();
    for ( $i = 6; $i >= 0; $i-- ) {
        $d = date( 'Y-m-d', strtotime( "-{$i} days", strtotime( $today ) ) );
        $day7_labels[] = date( 'j M', strtotime( $d ) );
        if ( $table_exists ) {
            $day7_values[] = (int) $wpdb->get_var( $wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table}` WHERE DATE(viewed_at) = %s", $d ) );
        } else {
            $day7_values[] = 0;
        }
    }

    // 1 Month (28 days) — query once, fill array
    $month_labels = array();
    $month_values = array();
    $m28_s        = date( 'Y-m-d', strtotime( '-27 days', strtotime( $today ) ) ) . ' 00:00:00';
    $raw_month    = array();
    if ( $table_exists ) {
        $rows = $wpdb->get_results( $wpdb->prepare(
            "SELECT DATE(viewed_at) AS day, COUNT(*) AS views
             FROM `{$table}` WHERE viewed_at >= %s
             GROUP BY day", $m28_s ) );
        foreach ( $rows as $r ) { $raw_month[ $r->day ] = (int) $r->views; }
    }
    for ( $i = 27; $i >= 0; $i-- ) {
        $d              = date( 'Y-m-d', strtotime( "-{$i} days", strtotime( $today ) ) );
        $dow            = (int) date( 'N', strtotime( $d ) );
        $month_labels[] = $dow === 1 ? date( 'j M', strtotime( $d ) ) : date( 'j', strtotime( $d ) );
        $month_values[] = $raw_month[ $d ] ?? 0;
    }

    // 6 Months — group by week (26 weeks)
    $m6_labels = array();
    $m6_values = array();
    $m6_s      = date( 'Y-m-d', strtotime( '-181 days', strtotime( $today ) ) ) . ' 00:00:00';
    $raw_6m    = array();
    if ( $table_exists ) {
        $rows = $wpdb->get_results( $wpdb->prepare(
            "SELECT DATE_FORMAT(viewed_at, '%%Y-%%u') AS week_key,
                    MIN(DATE(viewed_at)) AS week_start,
                    COUNT(*) AS views
             FROM `{$table}` WHERE viewed_at >= %s
             GROUP BY week_key ORDER BY week_key ASC", $m6_s ) );
        foreach ( $rows as $r ) { $raw_6m[ $r->week_key ] = array( 'views' => (int) $r->views, 'start' => $r->week_start ); }
    }
    // Build 26 week slots
    for ( $i = 25; $i >= 0; $i-- ) {
        $week_start     = date( 'Y-m-d', strtotime( '-' . ( $i * 7 ) . ' days', strtotime( $today ) ) );
        $wk             = date( 'Y-W', strtotime( $week_start ) );
        $m6_labels[]    = date( 'j M', strtotime( $week_start ) );
        $m6_values[]    = isset( $raw_6m[ str_replace( '-', '-', $wk ) ] )
                            ? $raw_6m[ str_replace( '-', '-', $wk ) ]['views'] : 0;
    }
    // Fill 6m from raw (simpler — just use ordered results directly)
    if ( ! empty( $raw_6m ) ) {
        $m6_labels = array();
        $m6_values = array();
        foreach ( $raw_6m as $wk => $data ) {
            $m6_labels[] = date( 'j M', strtotime( $data['start'] ) );
            $m6_values[] = $data['views'];
        }
        // Ensure exactly 26 entries, pad front with zeros if needed
        while ( count( $m6_labels ) < 26 ) {
            array_unshift( $m6_labels, '' );
            array_unshift( $m6_values, 0 );
        }
    } else {
        // No data — still generate 26 labeled slots
        $m6_labels = array();
        $m6_values = array();
        for ( $i = 25; $i >= 0; $i-- ) {
            $d           = date( 'Y-m-d', strtotime( '-' . ( $i * 7 ) . ' days', strtotime( $today ) ) );
            $m6_labels[] = date( 'j M', strtotime( $d ) );
            $m6_values[] = 0;
        }
    }

    $stats_url   = admin_url( 'tools.php?page=cloudscale-page-views' );
    $throttle_on = cspv_throttle_enabled();
    $blocked     = count( cspv_get_blocklist() );
    $widget_id   = 'cspv-dw-' . substr( md5( uniqid() ), 0, 6 );

    $periods = array(
        'hours'  => array( 'label' => '7 Hours',  'labels' => $hour_labels,  'values' => $hour_values  ),
        'days'   => array( 'label' => '7 Days',   'labels' => $day7_labels,  'values' => $day7_values  ),
        'month'  => array( 'label' => '1 Month',  'labels' => $month_labels, 'values' => $month_values ),
        'months' => array( 'label' => '6 Months', 'labels' => $m6_labels,    'values' => $m6_values    ),
    );
    ?>

<style>
#cspv_dashboard_widget .inside { padding: 0; margin: 0; }

.cspv-dw-banner {
    background: linear-gradient(135deg, #2d1b69 0%, #5b21b6 50%, #7c3aed 100%);
    padding: 14px 16px 12px;
    display: flex; align-items: flex-start; justify-content: space-between; gap: 10px;
}
.cspv-dw-today-count { font-size: 38px; font-weight: 800; color: #1db954; line-height: 1; }
.cspv-dw-today-label {
    font-size: 10px; color: rgba(255,255,255,.7);
    text-transform: uppercase; letter-spacing: .06em;
    margin-top: 3px; display: flex; align-items: center; flex-wrap: wrap; gap: 4px;
}
.cspv-dw-week-block { text-align: right; }
.cspv-dw-week-num   { font-size: 20px; font-weight: 700; color: #0d9488; line-height: 1; }
.cspv-dw-week-label { font-size: 10px; color: rgba(255,255,255,.65); text-transform: uppercase; letter-spacing: .04em; margin-top: 2px; }

/* Period buttons */
.cspv-dw-periods {
    display: flex; gap: 0; border-bottom: 1px solid #eee;
    background: #fafafa;
}
.cspv-dw-period-btn {
    flex: 1; padding: 7px 4px; font-size: 11px; font-weight: 600;
    text-align: center; cursor: pointer; border: none; background: transparent;
    color: #999; border-bottom: 2px solid transparent;
    transition: color .15s, border-color .15s; white-space: nowrap;
    font-family: inherit;
}
.cspv-dw-period-btn:hover  { color: #059669; }
.cspv-dw-period-btn.active { color: #059669; border-bottom-color: #10b981; background: #fff; }

/* Chart */
.cspv-dw-chart-wrap {
    padding: 8px 14px 0; background: #fff;
    border-bottom: 1px solid #f0f0f0;
}
.cspv-dw-canvas { width: 100% !important; height: 110px !important; display: block; }

/* Top posts */
.cspv-dw-list-header {
    font-size: 10px; font-weight: 700; text-transform: uppercase;
    letter-spacing: .05em; color: #aaa;
    padding: 7px 16px 3px; display: flex; justify-content: space-between;
}
.cspv-dw-row {
    display: flex; align-items: center;
    padding: 5px 16px; border-top: 1px solid #f5f5f5;
    font-size: 12px; gap: 8px;
}
.cspv-dw-row-title {
    flex: 1; white-space: nowrap; overflow: hidden;
    text-overflow: ellipsis; color: #1a2332; text-decoration: none; font-weight: 600;
}
.cspv-dw-row-title:hover { text-decoration: underline; color: #059669; }
.cspv-dw-row-bar  { height: 3px; background: #d1fae5; border-radius: 2px; flex-shrink: 0; width: 48px; overflow: hidden; }
.cspv-dw-row-fill { height: 100%; background: linear-gradient(90deg, #059669, #34d399); border-radius: 2px; }
.cspv-dw-row-num  { font-weight: 700; color: #059669; min-width: 24px; text-align: right; flex-shrink: 0; }
.cspv-dw-empty    { padding: 10px 16px; color: #bbb; font-size: 12px; font-style: italic; }

.cspv-dw-footer {
    padding: 8px 16px; border-top: 1px solid #eee;
    display: flex; justify-content: space-between; align-items: center;
}
.cspv-dw-link {
    display: inline-block;
    padding: 5px 12px;
    background: linear-gradient(135deg, #7c3aed, #a855f7);
    color: #fff;
    font-size: 11px;
    font-weight: 700;
    text-decoration: none;
    border-radius: 20px;
    letter-spacing: .03em;
    transition: opacity .15s;
}
.cspv-dw-link:hover { opacity: .85; color: #fff; text-decoration: none; }
.cspv-dw-shield     { font-size: 11px; }
.cspv-dw-shield.on  { color: #1db954; font-weight: 600; }
.cspv-dw-shield.off { color: #e53e3e; }
</style>

<!-- Banner -->
<div class="cspv-dw-banner">
    <div>
        <div class="cspv-dw-today-count"><?php echo number_format( $today_views ); ?></div>
        <div class="cspv-dw-today-label">
            Views today
            <?php echo $delta_html; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?>
        </div>
    </div>
    <div class="cspv-dw-week-block">
        <div class="cspv-dw-week-num"><?php echo number_format( $week_views ); ?></div>
        <div class="cspv-dw-week-label">Last 7 days</div>
    </div>
</div>

<!-- Period selector -->
<div class="cspv-dw-periods" id="<?php echo esc_attr( $widget_id ); ?>-periods">
    <?php foreach ( $periods as $key => $p ) : ?>
    <button class="cspv-dw-period-btn<?php echo $key === 'hours' ? ' active' : ''; ?>"
            data-period="<?php echo esc_attr( $key ); ?>">
        <?php echo esc_html( $p['label'] ); ?>
    </button>
    <?php endforeach; ?>
</div>

<!-- Chart -->
<div class="cspv-dw-chart-wrap">
    <canvas id="<?php echo esc_attr( $widget_id ); ?>" class="cspv-dw-canvas"></canvas>
</div>

<!-- Top posts today -->
<?php if ( empty( $top_today ) ) : ?>
    <div class="cspv-dw-empty">No views recorded today yet.</div>
<?php else : ?>
    <div class="cspv-dw-list-header"><span>Top today</span><span>Views</span></div>
    <?php
    $max = (int) $top_today[0]->views;
    foreach ( $top_today as $row ) :
        $pid   = absint( $row->post_id );
        $post  = get_post( $pid );
        $title = $post ? $post->post_title : 'Post #' . $pid;
        $url   = ( $post && 'publish' === $post->post_status ) ? get_permalink( $post ) : '';
        $pct   = $max > 0 ? round( ( (int) $row->views / $max ) * 100 ) : 0;
    ?>
    <div class="cspv-dw-row">
        <?php if ( $url ) : ?>
            <a href="<?php echo esc_url( $url ); ?>" target="_blank" class="cspv-dw-row-title"><?php echo esc_html( $title ); ?></a>
        <?php else : ?>
            <span class="cspv-dw-row-title"><?php echo esc_html( $title ); ?></span>
        <?php endif; ?>
        <div class="cspv-dw-row-bar"><div class="cspv-dw-row-fill" style="width:<?php echo (int) $pct; ?>%"></div></div>
        <span class="cspv-dw-row-num"><?php echo number_format( (int) $row->views ); ?></span>
    </div>
    <?php endforeach; ?>
<?php endif; ?>

<!-- Footer -->
<div class="cspv-dw-footer">
    <a href="<?php echo esc_url( $stats_url ); ?>" class="cspv-dw-link">View Full Statistics</a>
    <span class="cspv-dw-shield <?php echo $throttle_on ? 'on' : 'off'; ?>">
        <?php echo $throttle_on
            ? '🛡 ' . ( $blocked > 0 ? number_format( $blocked ) . ' blocked' : 'Protection on' )
            : '⚠ Protection off'; ?>
    </span>
</div>

<script>
(function() {
    var canvasId  = <?php echo wp_json_encode( $widget_id ); ?>;
    var periodsId = <?php echo wp_json_encode( $widget_id . '-periods' ); ?>;

    var datasets = <?php echo wp_json_encode( $periods ); ?>;
    var chartInst = null;
    var activePeriod = 'hours';

    function makeColors(values, period) {
        return values.map(function(v, i) {
            var isLast = i === values.length - 1;
            if (period === 'days' || period === 'hours') {
                return isLast ? '#059669' : 'rgba(5,150,105,0.25)';
            }
            return 'rgba(5,150,105,0.35)';
        });
    }

    function makeHoverColors(values, period) {
        return values.map(function(v, i) {
            var isLast = i === values.length - 1;
            if (period === 'days' || period === 'hours') {
                return isLast ? '#10b981' : 'rgba(5,150,105,0.55)';
            }
            return '#10b981';
        });
    }

    var currentPeriod = 'hours';
    function drawChart(period) {
        currentPeriod = period;
        var canvas = document.getElementById(canvasId);
        if (!canvas || !window.Chart) { return; }

        var data   = datasets[period];
        var labels = data.labels;
        var values = data.values;

        // Determine which x-axis labels to show based on dataset size
        // Always show at least a few — never blank axis even if all values are 0
        var maxTicks = labels.length <= 7 ? labels.length : (labels.length <= 28 ? 7 : 8);

        if (chartInst) { chartInst.destroy(); }

        chartInst = new Chart(canvas.getContext('2d'), {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [{
                    data: values,
                    backgroundColor: makeColors(values, period),
                    hoverBackgroundColor: makeHoverColors(values, period),
                    borderRadius: 2,
                    borderSkipped: false,
                }]
            },
            options: {
                responsive: false,
                animation: false,
                plugins: {
                    legend: { display: false },
                    tooltip: {
                        backgroundColor: '#2d1b69',
                        titleColor: 'rgba(255,255,255,.65)',
                        bodyColor: '#fff',
                        bodyFont: { size: 12, weight: '700' },
                        padding: 8,
                        displayColors: false,
                        callbacks: {
                            title: function(items) {
                                var lbl = items[0].label || '';
                                if (currentPeriod === 'hours' && lbl.indexOf(':') !== -1) {
                                    var h = parseInt(lbl.split(':')[0], 10);
                                    var suffix = h >= 12 ? ' PM' : ' AM';
                                    return lbl + suffix;
                                }
                                return lbl;
                            },
                            label: function(c) { return c.parsed.y.toLocaleString() + ' views'; }
                        }
                    }
                },
                scales: {
                    x: {
                        grid: { display: false },
                        ticks: {
                            color: '#999',
                            font: { size: 10 },
                            maxRotation: 0,
                            autoSkip: true,
                            maxTicksLimit: maxTicks,
                            padding: 4,
                        }
                    },
                    y: {
                        display: false,
                        beginAtZero: true,
                        // Ensure chart has minimum height when all values are 0
                        suggestedMax: Math.max(1, Math.max.apply(null, values)),
                    }
                },
                layout: { padding: { top: 2, left: 0, right: 2, bottom: 0 } }
            }
        });
    }

    function init() {
        var canvas = document.getElementById(canvasId);
        if (!canvas) { return; }

        // Wire period buttons
        var btns = document.getElementById(periodsId);
        if (btns) {
            btns.querySelectorAll('.cspv-dw-period-btn').forEach(function(btn) {
                btn.addEventListener('click', function() {
                    btns.querySelectorAll('.cspv-dw-period-btn').forEach(function(b) {
                        b.classList.remove('active');
                    });
                    btn.classList.add('active');
                    activePeriod = btn.dataset.period;
                    drawChart(activePeriod);
                });
            });
        }

        drawChart(activePeriod);
    }

    if (window.Chart) {
        init();
    } else {
        var s    = document.createElement('script');
        s.src    = 'https://cdnjs.cloudflare.com/ajax/libs/Chart.js/4.4.1/chart.umd.min.js';
        s.onload = init;
        document.head.appendChild(s);
    }
})();
</script>
    <?php
}
