/**
 * CloudScale Page Views - beacon.js  v2.0.0
 *
 * MODE: 'record' (singular post/page)
 *   Single POST to record the view. The response contains the updated
 *   count which is stored in window.cspvViews and used to update any
 *   .cspv-views-count elements on the page.
 *   NO second API call is made — the record response IS the count.
 *
 * MODE: 'fetch' (home / archive / search listing pages)
 *   Does NOT increment anything. Reads [data-cspv-id] attributes from
 *   the DOM, fetches all counts in one GET, injects them back.
 *
 * THEME INTEGRATION
 * -----------------
 * Single post template — just call the PHP helper, nothing else needed:
 *   <?php cspv_the_views(); ?>
 *
 * Archive/listing template — add data-cspv-id to your count element:
 *   <span class="cspv-views-count" data-cspv-id="<?php the_ID(); ?>">
 *       <?php echo cspv_get_view_count(); ?>
 *   </span>
 *
 * The beacon handles both automatically. You never need both on the
 * same page — on singular pages data-cspv-id is ignored in favour of
 * the record response.
 */

( function () {
    'use strict';

    if ( typeof cspvData === 'undefined' ) { return; }

    var debug   = cspvData.debug === true;
    var postId  = cspvData.postId ? String( cspvData.postId ) : null;

    function log() {
        if ( debug && typeof console !== 'undefined' ) {
            console.log.apply( console, ['[CloudScale PV]'].concat(
                Array.prototype.slice.call( arguments ) ) );
        }
    }

    // ------------------------------------------------------------------
    // Update DOM elements with a counts map { "postId": count, ... }
    // ------------------------------------------------------------------
    function updateDOM( counts ) {
        Object.keys( counts ).forEach( function( id ) {
            var val = Number( counts[ id ] ).toLocaleString();

            // Archive-style elements with explicit data-cspv-id
            document.querySelectorAll( '[data-cspv-id="' + id + '"]' )
                .forEach( function( el ) { el.textContent = val; } );
        } );

        // On singular pages also update plain .cspv-views-count elements
        // (the ones output by cspv_the_views() with no data-cspv-id)
        if ( cspvData.mode === 'record' && postId && counts[ postId ] !== undefined ) {
            var val = Number( counts[ postId ] ).toLocaleString();
            document.querySelectorAll( '.cspv-views-count:not([data-cspv-id])' )
                .forEach( function( el ) { el.textContent = val; } );

            // Also update auto-display counters
            document.querySelectorAll( '.cspv-ad-num' )
                .forEach( function( el ) { el.textContent = val; } );

            // Expose as a global so advanced templates can read it
            window.cspvViews = counts[ postId ];
            log( 'window.cspvViews set to', window.cspvViews );
        }
    }

    // ------------------------------------------------------------------
    // MODE: record
    // ------------------------------------------------------------------
    function recordView() {
        log( 'record mode — post', postId );

        // Deduplicate: only record once per session per post
        var sessionKey = 'cspv_seen_' + postId;
        try {
            if ( sessionStorage.getItem( sessionKey ) ) {
                log( 'already recorded this session — skipping beacon, fetching count' );
                // Still fetch the current count so the display stays fresh
                fetch( cspvData.apiUrl.replace( '/record/', '/counts?ids=' ) + postId, {
                    method: 'GET', credentials: 'same-origin'
                } )
                .then( function( r ) { return r.ok ? r.json() : null; } )
                .then( function( counts ) { if ( counts ) updateDOM( counts ); } )
                .catch( function() {} );
                return;
            }
        } catch(e) { /* sessionStorage blocked — allow recording */ }

        fetch( cspvData.apiUrl, {
            method:      'POST',
            headers:     { 'Content-Type': 'application/json', 'X-WP-Nonce': cspvData.nonce },
            body:        JSON.stringify( { referrer: document.referrer || '' } ),
            credentials: 'same-origin',
            keepalive:   true,
        } )
        .then( function( r ) {
            if ( ! r.ok ) { throw new Error( 'HTTP ' + r.status ); }
            return r.json();
        } )
        .then( function( data ) {
            if ( ! data || data.views === undefined ) { return; }
            log( 'recorded. count now:', data.views, '| logged:', data.logged );
            var counts = {};
            counts[ postId ] = data.views;
            updateDOM( counts );
            // Mark as seen for this session
            try { sessionStorage.setItem( sessionKey, '1' ); } catch(e) {}
        } )
        .catch( function( err ) { log( 'beacon error:', err ); } );
    }

    // ------------------------------------------------------------------
    // MODE: fetch (archive / listing)
    // ------------------------------------------------------------------
    function fetchCounts() {
        var els  = document.querySelectorAll( '[data-cspv-id]' );
        var ids  = [];
        var seen = {};

        els.forEach( function( el ) {
            var id = el.getAttribute( 'data-cspv-id' );
            if ( id && ! seen[ id ] ) { ids.push( id ); seen[ id ] = true; }
        } );

        if ( ids.length === 0 ) {
            log( 'fetch mode — no [data-cspv-id] elements found.' );
            return;
        }

        log( 'fetch mode — fetching counts for', ids.length, 'posts' );

        fetch( cspvData.countsUrl + '?ids=' + ids.join( ',' ), {
            method: 'GET', credentials: 'same-origin',
        } )
        .then( function( r ) {
            if ( ! r.ok ) { throw new Error( 'HTTP ' + r.status ); }
            return r.json();
        } )
        .then( function( counts ) {
            if ( ! counts ) { return; }
            log( 'received counts:', counts );
            updateDOM( counts );
        } )
        .catch( function( err ) { log( 'fetch error:', err ); } );
    }

    // ------------------------------------------------------------------
    // Boot
    // ------------------------------------------------------------------
    function boot() {
        if ( cspvData.mode === 'record' ) {
            recordView();
        } else {
            fetchCounts();
        }
    }

    if ( document.readyState === 'complete' ) {
        boot();
    } else {
        window.addEventListener( 'load', boot );
    }

} )();
