<?php
/**
 * CloudScale Page Views - IP Throttling  v2.0.0
 *
 * Blocks IPs that exceed the request threshold within a rolling window.
 * Blocks are stored as transients and automatically expire after 1 hour
 * — no permanent blocklist, no manual cleanup needed.
 *
 * The permanent blocklist (cspv_ip_blocklist option) is kept only for the
 * admin UI display. The authoritative block check always uses transients.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

define( 'CSPV_BLOCK_DURATION', 3600 ); // blocks auto-expire after 1 hour

// -------------------------------------------------------------------------
// Config helpers — enabled by default
// -------------------------------------------------------------------------

function cspv_throttle_enabled() {
    // Default true — must be explicitly disabled
    $val = get_option( 'cspv_throttle_enabled', null );
    return $val === null ? true : (bool) $val;
}

function cspv_throttle_limit() {
    return max( 1, (int) get_option( 'cspv_throttle_limit', 50 ) );
}

function cspv_throttle_window_seconds() {
    return (int) get_option( 'cspv_throttle_window', 3600 );
}

// -------------------------------------------------------------------------
// Core throttle check
// -------------------------------------------------------------------------

function cspv_is_throttled( $ip_hash ) {
    if ( ! cspv_throttle_enabled() || empty( $ip_hash ) ) {
        return false;
    }

    // Never throttle logged-in admins (useful during testing)
    if ( current_user_can( 'manage_options' ) ) {
        return false;
    }

    // Check block transient first — already blocked, still within 1 hr window
    if ( false !== get_transient( 'cspv_block_' . substr( $ip_hash, 0, 32 ) ) ) {
        return true;
    }

    $limit  = cspv_throttle_limit();
    $window = cspv_throttle_window_seconds();
    $key    = 'cspv_ip_' . substr( $ip_hash, 0, 32 );

    $count = (int) get_transient( $key );
    $count++;
    set_transient( $key, $count, $window );

    if ( $count >= $limit ) {
        cspv_block_ip( $ip_hash );
        return true;
    }

    return false;
}

// -------------------------------------------------------------------------
// Block / unblock — transient-based, auto-expires in 1 hour
// -------------------------------------------------------------------------

function cspv_block_ip( $ip_hash ) {
    $block_key = 'cspv_block_' . substr( $ip_hash, 0, 32 );

    // Only log and update persistent list on first block (not on repeated checks)
    if ( false === get_transient( $block_key ) ) {
        set_transient( $block_key, 1, CSPV_BLOCK_DURATION );

        // Persist to options for admin display — with expiry timestamp
        $list = cspv_get_blocklist();
        $expires = time() + CSPV_BLOCK_DURATION;
        $list[ $ip_hash ] = array(
            'blocked_at' => current_time( 'mysql' ),
            'expires'    => $expires,
        );
        update_option( 'cspv_ip_blocklist', $list, false );
        cspv_log_block_event( $ip_hash );
    }
}

function cspv_unblock_ip( $ip_hash ) {
    // Remove transient so block is immediately lifted
    delete_transient( 'cspv_block_' . substr( $ip_hash, 0, 32 ) );
    delete_transient( 'cspv_ip_'    . substr( $ip_hash, 0, 32 ) ); // reset counter too

    $list = cspv_get_blocklist();
    unset( $list[ $ip_hash ] );
    update_option( 'cspv_ip_blocklist', $list, false );
}

function cspv_clear_blocklist() {
    $list = cspv_get_blocklist();
    foreach ( array_keys( $list ) as $ip_hash ) {
        delete_transient( 'cspv_block_' . substr( $ip_hash, 0, 32 ) );
        delete_transient( 'cspv_ip_'    . substr( $ip_hash, 0, 32 ) );
    }
    update_option( 'cspv_ip_blocklist', array(), false );
    update_option( 'cspv_block_log',    array(), false );
}

function cspv_ip_is_blocked( $ip_hash ) {
    return false !== get_transient( 'cspv_block_' . substr( $ip_hash, 0, 32 ) );
}

// -------------------------------------------------------------------------
// Blocklist — now keyed by hash (assoc array) with expiry data
// Prunes expired entries on read so the admin list stays tidy
// -------------------------------------------------------------------------

function cspv_get_blocklist() {
    $raw = get_option( 'cspv_ip_blocklist', array() );

    // Legacy: if stored as a flat indexed array (old format), convert
    if ( isset( $raw[0] ) && is_string( $raw[0] ) ) {
        $converted = array();
        foreach ( $raw as $hash ) {
            $converted[ $hash ] = array( 'blocked_at' => '—', 'expires' => 0 );
        }
        $raw = $converted;
    }

    // Prune entries whose transient has expired
    $now    = time();
    $pruned = false;
    foreach ( $raw as $hash => $data ) {
        $expires = isset( $data['expires'] ) ? (int) $data['expires'] : 0;
        if ( $expires > 0 && $expires < $now ) {
            unset( $raw[ $hash ] );
            $pruned = true;
        }
    }
    if ( $pruned ) {
        update_option( 'cspv_ip_blocklist', $raw, false );
    }

    return is_array( $raw ) ? $raw : array();
}

// -------------------------------------------------------------------------
// Block event log
// -------------------------------------------------------------------------

function cspv_log_block_event( $ip_hash ) {
    $log = (array) get_option( 'cspv_block_log', array() );
    array_unshift( $log, array(
        'ip_hash'    => $ip_hash,
        'blocked_at' => current_time( 'mysql' ),
        'expires_at' => date( 'Y-m-d H:i:s', time() + CSPV_BLOCK_DURATION ),
    ) );
    update_option( 'cspv_block_log', array_slice( $log, 0, 100 ), false );
}

function cspv_get_block_log() {
    return (array) get_option( 'cspv_block_log', array() );
}

// -------------------------------------------------------------------------
// AJAX handlers
// -------------------------------------------------------------------------

add_action( 'wp_ajax_cspv_save_throttle_settings', 'cspv_ajax_save_throttle_settings' );
add_action( 'wp_ajax_cspv_unblock_ip',             'cspv_ajax_unblock_ip' );
add_action( 'wp_ajax_cspv_clear_blocklist',         'cspv_ajax_clear_blocklist' );

function cspv_ajax_save_throttle_settings() {
    if ( ! check_ajax_referer( 'cspv_throttle', 'nonce', false ) ) {
        wp_send_json_error( array( 'message' => 'Security check failed.' ), 403 );
        return;
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => 'Insufficient permissions.' ), 403 );
        return;
    }

    $enabled = ! empty( $_POST['enabled'] );
    $limit   = isset( $_POST['limit'] ) ? max( 1, min( 10000, (int) $_POST['limit'] ) ) : 50;
    $raw_win = isset( $_POST['window'] ) ? (int) $_POST['window'] : 3600;
    $window  = in_array( $raw_win, array( 600, 1800, 3600, 7200, 86400 ), true ) ? $raw_win : 3600;

    update_option( 'cspv_throttle_enabled', $enabled, false );
    update_option( 'cspv_throttle_limit',   $limit,   false );
    update_option( 'cspv_throttle_window',  $window,  false );

    wp_send_json_success( array( 'enabled' => $enabled, 'limit' => $limit, 'window' => $window ) );
}

function cspv_ajax_unblock_ip() {
    if ( ! check_ajax_referer( 'cspv_throttle', 'nonce', false ) ) {
        wp_send_json_error( array( 'message' => 'Security check failed.' ), 403 );
        return;
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => 'Insufficient permissions.' ), 403 );
        return;
    }

    $ip_hash = isset( $_POST['ip_hash'] ) ? sanitize_text_field( wp_unslash( $_POST['ip_hash'] ) ) : '';
    if ( empty( $ip_hash ) || ! preg_match( '/^[a-f0-9]{64}$/i', $ip_hash ) ) {
        wp_send_json_error( array( 'message' => 'Invalid IP hash.' ), 400 );
        return;
    }

    cspv_unblock_ip( $ip_hash );
    wp_send_json_success();
}

function cspv_ajax_clear_blocklist() {
    if ( ! check_ajax_referer( 'cspv_throttle', 'nonce', false ) ) {
        wp_send_json_error( array( 'message' => 'Security check failed.' ), 403 );
        return;
    }
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( array( 'message' => 'Insufficient permissions.' ), 403 );
        return;
    }

    cspv_clear_blocklist();
    wp_send_json_success();
}
