<?php
/**
 * Plugin Name: CloudScale SEO AI Optimizer
 * Plugin URI: https://wordpress.org/plugins/cloudscale-seo-ai-optimizer/
 * Description: Lightweight, code-first SEO: titles, meta descriptions, canonicals, OpenGraph/Twitter Cards, JSON-LD schema, and AI-powered meta description generation via the Anthropic Claude API.
 * Requires at least: 6.0
 * Requires PHP: 8.0
 *              JSON-LD schemas, robots, optional sitemap, and AI-powered meta description generation
 *              via the Anthropic Claude API.
 * Version: 4.8.1
 * Author: Andrew Baker
 * License: GPLv2 or later
 */

if (!defined('ABSPATH')) exit;

// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedClassFound
final class CloudScale_SEO_AI_Optimizer {

    const OPT        = 'cs_seo_options';
    const META_TITLE = '_cs_seo_title';
    const META_DESC  = '_cs_seo_desc';
    const META_OGIMG = '_cs_seo_ogimg';

    // Separate option key for AI config — keeps sensitive data isolated.
    const AI_OPT     = 'cs_seo_ai_options';

    private array $opts;
    private array $ai_opts;

    public function __construct() {
        $this->opts    = $this->get_opts();
        $this->ai_opts = $this->get_ai_opts();

        add_action('admin_menu',     [$this, 'admin_menu']);
        add_action('admin_notices',  [$this, 'admin_notices']);
        add_action('admin_head',     [$this, 'admin_head_css']);
        add_filter('admin_footer_text', [$this, 'admin_footer_text']);
        add_filter('update_footer',     [$this, 'admin_footer_version'], 11);
        add_action('admin_init',     [$this, 'register_settings']);
        add_action('add_meta_boxes', [$this, 'add_metabox']);
        add_action('save_post',      [$this, 'save_metabox'], 10, 2);

        add_filter('pre_get_document_title', [$this, 'filter_title'], 20);
        add_action('wp_head', [$this, 'render_head'], 1);

        // Suppress WordPress core canonical (prevents duplicate canonical error).
        remove_action('wp_head', 'rel_canonical');
        add_filter('wpseo_canonical',             '__return_false', 99);
        add_filter('rank_math/frontend/canonical', '__return_false', 99);

        add_action('init', [$this, 'maybe_register_sitemap']);
        add_filter('robots_txt', [$this, 'filter_robots_txt'], 99, 2);
        add_action('init', [$this, 'register_rest_meta']);

        // WP Cron batch job for scheduled generation.
        add_action('cs_seo_daily_batch', [$this, 'run_scheduled_batch']);

        // AJAX handlers for AI meta writer — both logged-in admin calls.
        add_action('wp_ajax_cs_seo_ai_generate_one',  [$this, 'ajax_generate_one']);
        add_action('wp_ajax_cs_seo_ai_generate_all',  [$this, 'ajax_generate_all']);
        add_action('wp_ajax_cs_seo_ai_fix_desc',      [$this, 'ajax_fix_desc']);
        add_action('wp_ajax_cs_seo_ai_get_posts',     [$this, 'ajax_get_posts']);
        add_action('wp_ajax_cs_seo_ai_test_key',      [$this, 'ajax_test_key']);
        add_action('wp_ajax_cs_seo_ai_get_batch_log', [$this, 'ajax_get_batch_log']);
        add_action('wp_ajax_cs_seo_sitemap_preview',  [$this, 'ajax_sitemap_preview']);
        add_action('wp_ajax_cs_seo_rename_robots',    [$this, 'ajax_rename_robots']);
        add_action('wp_ajax_cs_seo_fetch_robots',     [$this, 'ajax_fetch_robots']);
    }

    // =========================================================================
    // Options
    // =========================================================================

    public static function defaults(): array {
        $site = get_bloginfo('name');
        return [
            'site_name'               => $site,
            'site_lang'               => 'en-US',
            'title_suffix'            => ' | ' . $site,
            'home_title'              => $site,
            'home_desc'               => '',
            'default_desc'            => '',
            'default_og_image'        => '',
            'twitter_handle'          => '',
            'enable_og'               => 1,
            'enable_schema_person'    => 1,
            'enable_schema_website'   => 1,
            'enable_schema_article'   => 1,
            'enable_schema_breadcrumbs' => 1,
            'strip_tracking_params'   => 1,
            'enable_sitemap'          => 0,
            'noindex_search'          => 1,
            'noindex_404'             => 1,
            'noindex_attachment'      => 1,
            'noindex_author_archives' => 0,
            'noindex_tag_archives'    => 0,
            'person_name'             => '',
            'person_job_title'        => '',
            'person_url'              => home_url('/'),
            'person_image'            => '',
            'sameas'                  => '',
            'robots_txt'              => self::default_robots_txt(),
            'block_ai_bots'           => 1,
            'sitemap_post_types'      => ['post', 'page'],
            'sitemap_taxonomies'      => 0,
            'sitemap_exclude'         => '',
        ];
    }

    public static function ai_defaults(): array {
        return [
            'anthropic_key'    => '',
            'model'            => 'claude-sonnet-4-20250514',
            'overwrite'        => 0,
            'min_chars'        => 140,
            'max_chars'        => 155,
            'prompt'           => self::default_prompt(),
            'schedule_enabled' => 0,
            'schedule_days'    => [],
        ];
    }

    public static function default_robots_txt(): string {
        return "User-agent: Googlebot\nAllow: /\nDisallow: /wp-admin/\nDisallow: /wp-login.php\nDisallow: /xmlrpc.php\nDisallow: /?s=\nDisallow: /search/\nDisallow: /*?prp_page_paginated_recent_posts\n\nUser-agent: *\nAllow: /\nDisallow: /wp-admin/\nDisallow: /wp-login.php\nDisallow: /xmlrpc.php\nDisallow: /?s=\nDisallow: /search/\nDisallow: /*?prp_page_paginated_recent_posts";
    }

    private static function default_prompt(): string {
        return 'You are an expert SEO copywriter. Site context will be injected automatically from the site settings below.

Write a single meta description for the article provided. Rules:
- HARD LIMIT: The character range is specified separately — count carefully before outputting. If your draft exceeds the maximum, shorten it. If it is under the minimum, expand it.
- Include the primary keyword or topic naturally in the first half
- Must be a complete, compelling sentence that makes a reader want to click
- No marketing fluff. No "In this post..." or "This article covers..." openers
- Write as a factual, punchy statement about what the article delivers
- Output ONLY the meta description text — no quotes, no labels, nothing else';
    }

    private function get_opts(): array {
        $saved = get_option(self::OPT, []);
        return array_merge(self::defaults(), is_array($saved) ? $saved : []);
    }

    private function get_ai_opts(): array {
        $saved = get_option(self::AI_OPT, []);
        return array_merge(self::ai_defaults(), is_array($saved) ? $saved : []);
    }

    // =========================================================================
    // REST meta registration
    // =========================================================================

    public function register_rest_meta(): void {
        foreach (['post', 'page'] as $post_type) {
            register_post_meta($post_type, self::META_TITLE, [
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'auth_callback'     => fn() => current_user_can('edit_posts'),
                'sanitize_callback' => 'sanitize_text_field',
            ]);
            register_post_meta($post_type, self::META_DESC, [
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'auth_callback'     => fn() => current_user_can('edit_posts'),
                'sanitize_callback' => 'sanitize_textarea_field',
            ]);
            register_post_meta($post_type, self::META_OGIMG, [
                'show_in_rest'      => true,
                'single'            => true,
                'type'              => 'string',
                'auth_callback'     => fn() => current_user_can('edit_posts'),
                'sanitize_callback' => 'esc_url_raw',
            ]);
        }
    }

    // =========================================================================
    // Title filter
    // =========================================================================

    public function filter_title(string $default): string {
        if (is_admin()) return $default;

        if (is_front_page() || is_home()) {
            $t = trim((string) $this->opts['home_title']);
            return $t ?: $default;
        }

        if (is_singular()) {
            $pid    = (int) get_queried_object_id();
            $custom = trim((string) get_post_meta($pid, self::META_TITLE, true));
            if ($custom !== '') return $custom;
            return $default;
        }

        $suffix = (string) $this->opts['title_suffix'];
        if ($suffix && substr($default, -strlen($suffix)) !== $suffix) {
            return $default . $suffix;
        }
        return $default;
    }

    // =========================================================================
    // Head output
    // =========================================================================

    public function render_head(): void {
        if (is_admin()) return;
        echo $this->build_seo_block(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- build_seo_block() returns pre-escaped HTML
    }

    private function build_seo_block(): string {
        $out = "\n<!-- CloudScale SEO AI Optimizer 4.2.0 -->\n";

        $canonical = $this->canonical_url();
        if ($canonical) $out .= '<link rel="canonical" href="' . esc_url($canonical) . '">' . "\n";

        $desc = $this->meta_desc();
        if ($desc) $out .= '<meta name="description" content="' . esc_attr($desc) . '">' . "\n";

        $pname = trim((string) $this->opts['person_name']);
        if ($pname) $out .= '<meta name="author" content="' . esc_attr($pname) . '">' . "\n";

        $robots = $this->robots();
        if ($robots) $out .= '<meta name="robots" content="' . esc_attr($robots) . '">' . "\n";

        if ((int) $this->opts['enable_og']) {
            $out .= $this->render_og_tags();
        }

        $out .= $this->render_schemas();
        $out .= "<!-- /CloudScale SEO AI Optimizer -->\n";
        return $out;
    }

    // =========================================================================
    // Canonical / URL helpers
    // =========================================================================

    private function canonical_url(): string {
        if (is_singular()) return $this->clean_url((string) get_permalink((int) get_queried_object_id()));
        if (is_front_page() || is_home()) return $this->clean_url(home_url('/'));
        if (is_archive()) return $this->clean_url((string) get_pagenum_link(max(1, (int) get_query_var('paged'))));
        return '';
    }

    private function clean_url(string $url): string {
        if (!(int) $this->opts['strip_tracking_params']) return $url;
        $p = wp_parse_url($url);
        if (!$p) return $url;
        $scheme = $p['scheme'] ?? 'https';
        $host   = $p['host']   ?? '';
        $path   = $p['path']   ?? '/';
        $port   = isset($p['port']) ? ':' . $p['port'] : '';
        $qs     = '';
        if (!empty($p['query'])) {
            parse_str($p['query'], $q);
            foreach (array_keys($q) as $k) {
                $kl = strtolower((string) $k);
                if (
                    strpos($kl, 'utm_') === 0 ||
                    strpos($kl, 'prp_page_') === 0 ||
                    in_array($kl, ['fbclid','gclid','msclkid'], true)
                ) unset($q[$k]);
            }
            if ($q) $qs = '?' . http_build_query($q);
        }
        return $scheme . '://' . $host . $port . $path . $qs;
    }

    // =========================================================================
    // Meta description
    // =========================================================================

    private function meta_desc(): string {
        if (is_front_page() || is_home()) {
            $h = trim((string) $this->opts['home_desc']);
            if ($h) return $this->clip($h, 160);
        }
        if (is_singular()) {
            $pid    = (int) get_queried_object_id();
            $custom = trim((string) get_post_meta($pid, self::META_DESC, true));
            if ($custom) return $this->clip($custom, 160);
            $post = get_post($pid);
            if ($post) {
                if (!empty($post->post_excerpt)) {
                    return $this->clip($this->text_from_html((string) $post->post_excerpt), 160);
                }
                return $this->clip($this->text_from_html((string) $post->post_content), 160);
            }
        }
        $d = trim((string) $this->opts['default_desc']);
        return $d ? $this->clip($d, 160) : '';
    }

    private function text_from_html(string $raw): string {
        $raw = strip_shortcodes($raw);
        $raw = wp_strip_all_tags($raw);
        return (string) preg_replace('/\s+/', ' ', $raw);
    }

    // =========================================================================
    // Robots
    // =========================================================================

    private function robots(): string {
        if ((int) $this->opts['noindex_search']          && is_search())     return 'noindex,follow';
        if ((int) $this->opts['noindex_404']             && is_404())        return 'noindex,follow';
        if ((int) $this->opts['noindex_attachment']      && is_attachment()) return 'noindex,follow';
        if ((int) $this->opts['noindex_author_archives'] && is_author())     return 'noindex,follow';
        if ((int) $this->opts['noindex_tag_archives']    && is_tag())        return 'noindex,follow';
        return '';
    }

    private function is_noindexed(): bool {
        return $this->robots() !== '';
    }

    // =========================================================================
    // OG image
    // =========================================================================

    private function og_image_data(): array {
        $url = ''; $width = 0; $height = 0; $type = ''; $alt = '';

        if (is_singular()) {
            $pid    = (int) get_queried_object_id();
            $custom = trim((string) get_post_meta($pid, self::META_OGIMG, true));
            if ($custom) {
                $url = $custom;
                $att_id = attachment_url_to_postid($custom);
                if ($att_id) {
                    $meta = wp_get_attachment_metadata($att_id);
                    if (!empty($meta['width']))  $width  = (int) $meta['width'];
                    if (!empty($meta['height'])) $height = (int) $meta['height'];
                    $type = get_post_mime_type($att_id) ?: '';
                    $alt  = trim((string) get_post_meta($att_id, '_wp_attachment_image_alt', true));
                }
            } elseif (has_post_thumbnail($pid)) {
                $thumb_id = (int) get_post_thumbnail_id($pid);
                $src = wp_get_attachment_image_src($thumb_id, 'full');
                if (!empty($src[0])) {
                    $url    = (string) $src[0];
                    $width  = isset($src[1]) ? (int) $src[1] : 0;
                    $height = isset($src[2]) ? (int) $src[2] : 0;
                    $type   = get_post_mime_type($thumb_id) ?: '';
                    $alt    = trim((string) get_post_meta($thumb_id, '_wp_attachment_image_alt', true));
                }
            }
        }

        if (!$url) {
            $url = trim((string) $this->opts['default_og_image']);
            if ($url) {
                $att_id = attachment_url_to_postid($url);
                if ($att_id) {
                    $meta = wp_get_attachment_metadata($att_id);
                    if (!empty($meta['width']))  $width  = (int) $meta['width'];
                    if (!empty($meta['height'])) $height = (int) $meta['height'];
                    $type = get_post_mime_type($att_id) ?: '';
                    $alt  = trim((string) get_post_meta($att_id, '_wp_attachment_image_alt', true));
                }
            }
        }

        return compact('url', 'width', 'height', 'type', 'alt');
    }

    // =========================================================================
    // OG / Twitter
    // =========================================================================

    private function render_og_tags(): string {
        $title   = $this->page_title();
        $desc    = $this->meta_desc();
        $url     = $this->canonical_url() ?: home_url('/');
        $site    = (string) $this->opts['site_name'];
        $locale  = str_replace('-', '_', (string) $this->opts['site_lang']);
        $twitter = (string) $this->opts['twitter_handle'];
        $type    = is_singular('post') ? 'article' : 'website';
        $img     = $this->og_image_data();
        $out     = '';

        $og = [
            'og:locale'      => $locale,
            'og:type'        => $type,
            'og:title'       => $title,
            'og:description' => $desc,
            'og:url'         => $url,
            'og:site_name'   => $site,
        ];

        if ($type === 'article' && is_singular()) {
            $pid = (int) get_queried_object_id();
            $published = get_post_time('c', true, $pid);
            $modified  = get_post_modified_time('c', true, $pid);
            if ($published) $og['article:published_time'] = $published;
            if ($modified)  $og['article:modified_time']  = $modified;
            $og['article:author'] = (string) $this->opts['person_url'];
            $cats = get_the_category($pid);
            if (!empty($cats[0])) $og['article:section'] = $cats[0]->name;
        }

        foreach ($og as $k => $v) {
            if ((string)$v === '') continue;
            $out .= '<meta property="' . esc_attr($k) . '" content="' . esc_attr((string)$v) . '">' . "\n";
        }

        if ($img['url']) {
            $out .= '<meta property="og:image" content="'        . esc_attr($img['url'])            . '">' . "\n";
            if ($img['width'])  $out .= '<meta property="og:image:width" content="'  . esc_attr((string)$img['width'])  . '">' . "\n";
            if ($img['height']) $out .= '<meta property="og:image:height" content="' . esc_attr((string)$img['height']) . '">' . "\n";
            if ($img['type'])   $out .= '<meta property="og:image:type" content="'   . esc_attr($img['type'])           . '">' . "\n";
            if ($img['alt'])    $out .= '<meta property="og:image:alt" content="'    . esc_attr($img['alt'])            . '">' . "\n";
        }

        $out .= '<meta name="twitter:card" content="'        . esc_attr($img['url'] ? 'summary_large_image' : 'summary') . '">' . "\n";
        $out .= '<meta name="twitter:title" content="'       . esc_attr($title)      . '">' . "\n";
        if ($desc)       $out .= '<meta name="twitter:description" content="' . esc_attr($desc)      . '">' . "\n";
        if ($img['url']) $out .= '<meta name="twitter:image" content="'       . esc_attr($img['url']) . '">' . "\n";
        if ($img['alt']) $out .= '<meta name="twitter:image:alt" content="'   . esc_attr($img['alt']) . '">' . "\n";
        if ($twitter)    $out .= '<meta name="twitter:site" content="'        . esc_attr($twitter)    . '">' . "\n";
        if ($twitter)    $out .= '<meta name="twitter:creator" content="'     . esc_attr($twitter)    . '">' . "\n";

        return $out;
    }

    // =========================================================================
    // Schemas
    // =========================================================================

    private function render_schemas(): string {
        $out     = '';
        $noindex = $this->is_noindexed();

        if ((int) $this->opts['enable_schema_website'] && (is_front_page() || is_home())) {
            $out .= $this->schema_tag($this->schema_website());
        }
        if ((int) $this->opts['enable_schema_person'] && !$noindex) {
            $out .= $this->schema_tag($this->schema_person());
        }
        if ((int) $this->opts['enable_schema_breadcrumbs'] && !$noindex) {
            $bc = $this->schema_breadcrumbs();
            if ($bc) $out .= $this->schema_tag($bc);
        }
        if ((int) $this->opts['enable_schema_article'] && is_singular('post') && !$noindex) {
            $art = $this->schema_article();
            if ($art) $out .= $this->schema_tag($art);
        }
        return $out;
    }

    private function schema_tag(array $schema): string {
        return '<script type="application/ld+json">'
            . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE)
            . "</script>\n";
    }

    private function schema_website(): array {
        return [
            '@context'        => 'https://schema.org',
            '@type'           => 'WebSite',
            'name'            => (string) $this->opts['site_name'],
            'url'             => home_url('/'),
            'potentialAction' => [
                '@type'       => 'SearchAction',
                'target'      => ['@type' => 'EntryPoint', 'urlTemplate' => home_url('/?s={search_term_string}')],
                'query-input' => 'required name=search_term_string',
            ],
        ];
    }

    private function schema_person(): array {
        $sameAs = array_values(array_filter(
            array_map('trim', (array) preg_split('/\r\n|\r|\n/', (string) $this->opts['sameas']))
        ));
        $s = [
            '@context' => 'https://schema.org',
            '@type'    => 'Person',
            'name'     => (string) $this->opts['person_name'],
            'jobTitle' => (string) $this->opts['person_job_title'],
            'url'      => (string) $this->opts['person_url'],
        ];
        if ($sameAs) $s['sameAs'] = $sameAs;
        $img = trim((string) $this->opts['person_image']);
        if ($img) $s['image'] = $img;
        return $s;
    }

    private function schema_article(): ?array {
        if (!is_singular()) return null;
        $pid  = (int) get_queried_object_id();
        $post = get_post($pid);
        if (!$post) return null;

        $img        = $this->og_image_data();
        $cats       = get_the_category($pid);
        $tags       = wp_get_post_tags($pid, ['fields' => 'names']);
        $word_count = str_word_count(wp_strip_all_tags((string) $post->post_content));
        $mins       = max(1, (int) ceil($word_count / 200));
        $published  = get_post_time('c', true, $pid);
        $modified   = get_post_modified_time('c', true, $pid);

        $s = [
            '@context'         => 'https://schema.org',
            '@type'            => 'BlogPosting',
            'mainEntityOfPage' => ['@type' => 'WebPage', '@id' => $this->canonical_url()],
            'headline'         => get_the_title($pid),
            'description'      => $this->meta_desc(),
            'author'           => [
                '@type' => 'Person',
                'name'  => get_the_author_meta('display_name', (int) $post->post_author) ?: (string) $this->opts['person_name'],
                'url'   => (string) $this->opts['person_url'],
            ],
            'publisher' => [
                '@type' => 'Person',
                'name'  => (string) $this->opts['person_name'],
                'url'   => (string) $this->opts['person_url'],
            ],
            'wordCount'    => $word_count,
            'timeRequired' => 'PT' . $mins . 'M',
        ];

        if ($published) $s['datePublished'] = $published;
        if ($modified)  $s['dateModified']  = $modified;

        if ($img['url']) {
            $image = ['@type' => 'ImageObject', 'url' => $img['url']];
            if ($img['width'])  $image['width']  = $img['width'];
            if ($img['height']) $image['height'] = $img['height'];
            $s['image'] = [$image];
        }

        $pimg = trim((string) $this->opts['person_image']);
        if ($pimg) $s['publisher']['logo'] = ['@type' => 'ImageObject', 'url' => $pimg];

        if (!empty($cats[0])) $s['articleSection'] = $cats[0]->name;
        if (!empty($tags))    $s['keywords']       = implode(', ', $tags);

        return $s;
    }

    private function schema_breadcrumbs(): ?array {
        $items = [];
        $pos   = 1;
        $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => 'Home', 'item' => home_url('/')];

        if (is_singular('post')) {
            $pid  = (int) get_queried_object_id();
            $cats = get_the_category($pid);
            if (!empty($cats[0])) {
                $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => $cats[0]->name, 'item' => get_category_link($cats[0]->term_id)];
            }
            $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => get_the_title($pid), 'item' => get_permalink($pid)];
        } elseif (is_page()) {
            $pid = (int) get_queried_object_id();
            foreach (array_reverse(get_post_ancestors($pid)) as $anc) {
                $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => get_the_title($anc), 'item' => get_permalink($anc)];
            }
            $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => get_the_title($pid), 'item' => get_permalink($pid)];
        } elseif (is_category() || is_tag() || is_author()) {
            $items[] = ['@type' => 'ListItem', 'position' => $pos++, 'name' => $this->page_title(), 'item' => $this->canonical_url()];
        } else {
            return null;
        }

        if (count($items) <= 1) return null;
        return ['@context' => 'https://schema.org', '@type' => 'BreadcrumbList', 'itemListElement' => $items];
    }

    // =========================================================================
    // Helpers
    // =========================================================================

    private function page_title(): string {
        if (is_singular()) {
            $pid    = (int) get_queried_object_id();
            $custom = trim((string) get_post_meta($pid, self::META_TITLE, true));
            if ($custom !== '') return $custom;
            return get_the_title($pid);
        }
        if (is_front_page() || is_home()) {
            $t = trim((string) $this->opts['home_title']);
            return $t ?: (string) $this->opts['site_name'];
        }
        return wp_get_document_title();
    }

    private function clip(string $s, int $max): string {
        $s = trim((string) preg_replace('/\s+/', ' ', $s));
        if ($s === '' || mb_strlen($s) <= $max) return $s;
        return rtrim(mb_substr($s, 0, $max - 1)) . '…';
    }

    // =========================================================================
    // Metabox
    // =========================================================================

    public function add_metabox(): void {
        foreach (['post', 'page'] as $pt) {
            add_meta_box('cs_seo_adv', 'CloudScale SEO', [$this, 'render_metabox'], $pt, 'normal', 'high');
        }
    }

    public function render_metabox(WP_Post $post): void {
        wp_nonce_field('cs_seo_save', 'cs_seo_nonce');
        $title = (string) get_post_meta($post->ID, self::META_TITLE, true);
        $desc  = (string) get_post_meta($post->ID, self::META_DESC,  true);
        $ogimg = (string) get_post_meta($post->ID, self::META_OGIMG, true);
        $has_key = !empty($this->ai_opts['anthropic_key']);
        ?>
        <p><strong>Custom SEO title</strong> — leave blank to auto-generate<br>
            <input class="widefat" name="cs_seo_title" value="<?php echo esc_attr($title); ?>"></p>
        <p>
            <strong>Meta description</strong> — leave blank to use excerpt / post content<br>
            <textarea class="widefat" rows="3" name="cs_seo_desc" id="cs_seo_desc_<?php echo (int) $post->ID; ?>"><?php echo esc_textarea($desc); ?></textarea>
            <span id="cs_seo_char_<?php echo (int) $post->ID; ?>" style="font-size:11px;color:#888;">
                <?php echo $desc ? esc_html( (string) mb_strlen($desc) ) . ' chars' : 'No description set'; ?>
            </span>
        </p>
        <?php if ($has_key): ?>
        <p>
            <button type="button" class="button" id="cs_seo_gen_<?php echo (int) $post->ID; ?>"
                onclick="csSeoGenOne(<?php echo (int) $post->ID; ?>)">
                ✦ Generate with Claude
            </button>
            <span id="cs_seo_gen_status_<?php echo (int) $post->ID; ?>" style="margin-left:8px;font-size:12px;color:#888;"></span>
        </p>
        <script>
        function csSeoGenOne(postId) {
            const btn    = document.getElementById('cs_seo_gen_' + postId);
            const status = document.getElementById('cs_seo_gen_status_' + postId);
            const field  = document.getElementById('cs_seo_desc_' + postId);
            const chars  = document.getElementById('cs_seo_char_' + postId);
            btn.disabled = true;
            status.textContent = '⟳ Generating...';
            status.style.color = '#888';
            fetch(ajaxurl, {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: new URLSearchParams({
                    action: 'cs_seo_ai_generate_one',
                    post_id: postId,
                    nonce: '<?php echo esc_js( wp_create_nonce('cs_seo_nonce') ); ?>'
                })
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    field.value = data.data.description;
                    chars.textContent = data.data.chars + ' chars';
                    chars.style.color = data.data.chars >= 140 && data.data.chars <= 160 ? '#46b450' : '#dc3232';
                    status.textContent = '✓ Done — save post to keep';
                    status.style.color = '#46b450';
                } else {
                    status.textContent = '✗ ' + (data.data || 'Error');
                    status.style.color = '#dc3232';
                }
            })
            .catch(e => {
                status.textContent = '✗ ' + e.message;
                status.style.color = '#dc3232';
            })
            .finally(() => { btn.disabled = false; });
        }
        </script>
        <?php else: ?>
        <p style="color:#888;font-size:12px;"><em>Add an Anthropic API key in <a href="<?php echo esc_url( admin_url('options-general.php?page=cs-seo-optimizer#ai') ); ?>">SEO Settings → AI Meta Writer</a> to enable per-post generation.</em></p>
        <?php endif; ?>
        <p><strong>OG image URL</strong> — leave blank to use featured image<br>
            <input class="widefat" name="cs_seo_ogimg" value="<?php echo esc_attr($ogimg); ?>"></p>
        <?php
    }

    public function save_metabox(int $post_id, WP_Post $post): void {
        if (!isset($_POST['cs_seo_nonce'])) return;
        if (!wp_verify_nonce( sanitize_key( wp_unslash( $_POST['cs_seo_nonce'] ) ), 'cs_seo_save')) return;
        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
        if (!current_user_can('edit_post', $post_id)) return;
        $this->set_meta($post_id, self::META_TITLE, sanitize_text_field( wp_unslash( (string) ($_POST['cs_seo_title'] ?? '') ) ));
        $this->set_meta($post_id, self::META_DESC,  sanitize_textarea_field( wp_unslash( (string) ($_POST['cs_seo_desc'] ?? '') ) ));
        $this->set_meta($post_id, self::META_OGIMG, esc_url_raw( wp_unslash( (string) ($_POST['cs_seo_ogimg'] ?? '') ) ));
    }

    private function set_meta(int $id, string $key, string $val): void {
        $val === '' ? delete_post_meta($id, $key) : update_post_meta($id, $key, $val);
    }

    // =========================================================================
    // AJAX handlers
    // =========================================================================
    // Scheduled batch
    // =========================================================================

    /**
     * WP Cron callback — fires daily. Checks if today is a scheduled day,
     * then generates descriptions for all posts that don't have one yet.
     * Logs results to a transient so the admin UI can show last run status.
     */
    public function run_scheduled_batch(): void {
        $ai = $this->get_ai_opts();
        if (!(int) $ai['schedule_enabled']) return;

        $days = (array) $ai['schedule_days'];
        if (empty($days)) return;

        // Check if today (server time) is a scheduled day.
        $today = strtolower(gmdate('D')); // 'mon','tue' etc.
        if (!in_array($today, $days, true)) return;

        $log     = [];
        $done    = 0;
        $errors  = 0;
        $skipped = 0;
        $start   = time();

        $q = new WP_Query([
            'post_type'           => ['post', 'page'],
            'post_status'         => 'publish',
            'posts_per_page'      => 500,
            'no_found_rows'       => true,
            'ignore_sticky_posts' => true,
            'orderby'             => 'date',
            'order'               => 'DESC',
        ]);

        foreach ($q->posts as $p) {
            $existing = trim((string) get_post_meta($p->ID, self::META_DESC, true));
            if ($existing) {
                $skipped++;
                continue; // Scheduled batch only processes unprocessed posts.
            }
            try {
                $desc = $this->call_anthropic($p->ID);
                update_post_meta($p->ID, self::META_DESC, sanitize_textarea_field($desc));
                $log[] = ['status' => 'ok', 'title' => get_the_title($p->ID), 'chars' => mb_strlen($desc)];
                $done++;
            } catch (\Throwable $e) {
                $log[] = ['status' => 'err', 'title' => get_the_title($p->ID), 'message' => $e->getMessage()];
                $errors++;
                sleep(2); // Back off on error.
            }
            sleep(1); // Pace requests — T4g Micro friendly.
        }

        // Store last run summary as a transient (24h).
        set_transient('cs_seo_last_batch', [
            'date'    => gmdate('Y-m-d H:i:s'),
            'day'     => ucfirst($today),
            'done'    => $done,
            'skipped' => $skipped,
            'errors'  => $errors,
            'elapsed' => round((time() - $start) / 60, 1),
            'log'     => array_slice($log, 0, 100), // Keep last 100 entries.
        ], DAY_IN_SECONDS);
    }

    /**
     * AJAX: return the last scheduled batch result for display in the UI.
     */
    public function ajax_get_batch_log(): void {
        $this->ajax_check();
        $data = get_transient('cs_seo_last_batch');
        if ($data) {
            wp_send_json_success($data);
        } else {
            wp_send_json_success(null);
        }
    }

    // =========================================================================

    private function ajax_check(): void {
        if (!current_user_can('manage_options')) wp_send_json_error('Forbidden', 403);
        if (!check_ajax_referer('cs_seo_nonce', 'nonce', false)) wp_send_json_error('Bad nonce', 403);
    }

    /**
     * Call Anthropic API and return a generated description for a single post.
     */
    private function call_anthropic(int $post_id): string {
        $post = get_post($post_id);
        if (!$post) throw new \RuntimeException( esc_html( "Post {$post_id} not found" ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped

        $key    = trim((string) $this->ai_opts['anthropic_key']);
        $model  = trim((string) $this->ai_opts['model']) ?: 'claude-sonnet-4-20250514';
        $prompt = trim((string) $this->ai_opts['prompt']) ?: self::default_prompt();
        $min    = max(100, (int) $this->ai_opts['min_chars']);
        $max    = min(200, (int) $this->ai_opts['max_chars']);

        if (!$key) throw new \RuntimeException('No Anthropic API key configured');

        // Build clean content for Claude.
        $content = $this->text_from_html((string) $post->post_content);
        $content = mb_substr($content, 0, 6000); // generous context, well under token limits

        $user_msg = "Article title: \"{$post->post_title}\"\n\nArticle content:\n{$content}";

        // Build site context from SEO settings — keeps the prompt generic
        // and automatically reflects whatever the site owner has configured.
        $site_context_parts = [];
        $site_name  = trim((string) $this->opts['site_name']);
        $person     = trim((string) $this->opts['person_name']);
        $job_title  = trim((string) $this->opts['person_job_title']);
        $home_desc  = trim((string) $this->opts['home_desc']);
        $def_desc   = trim((string) $this->opts['default_desc']);

        if ($site_name)  $site_context_parts[] = "Site name: {$site_name}";
        if ($person)     $site_context_parts[] = "Author: {$person}" . ($job_title ? ", {$job_title}" : '');
        // Use home description or default description as the site topic summary.
        $topic = $home_desc ?: $def_desc;
        if ($topic)      $site_context_parts[] = "Site description: {$topic}";

        $site_context = $site_context_parts
            ? "\n\nSITE CONTEXT:\n" . implode("\n", $site_context_parts)
            : '';

        // Add dynamic char range — single source of truth, prompt never hardcodes range.
        $system = $prompt . $site_context
            . "\n\nCHARACTER REQUIREMENT: The description MUST be between {$min} and {$max} characters including spaces. Count every character carefully. Do not produce output outside this range.";

        $payload = [
            'model'      => $model,
            'max_tokens' => 300,
            'system'     => $system,
            'messages'   => [['role' => 'user', 'content' => $user_msg]],
        ];

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'timeout' => 45,
            'headers' => [
                'Content-Type'      => 'application/json',
                'x-api-key'         => $key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode($payload),
        ]);

        if (is_wp_error($response)) {
            throw new \RuntimeException( esc_html( 'HTTP error: ' . $response->get_error_message() ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        $code = wp_remote_retrieve_response_code($response);

        // On rate limit (429), wait 10 seconds and retry once.
        if ($code === 429) {
            sleep(10);
            $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
                'timeout' => 45,
                'headers' => [
                    'Content-Type'      => 'application/json',
                    'x-api-key'         => $key,
                    'anthropic-version' => '2023-06-01',
                ],
                'body' => wp_json_encode($payload),
            ]);
            if (is_wp_error($response)) {
                throw new \RuntimeException( esc_html( 'HTTP error after retry: ' . $response->get_error_message() ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
            }
            $code = wp_remote_retrieve_response_code($response);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            $msg = $body['error']['message'] ?? "API returned HTTP {$code}";
            throw new \RuntimeException( esc_html( $msg ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        $text = trim($body['content'][0]['text'] ?? '');
        $text = trim($text, '"\'');
        if (!$text) throw new \RuntimeException('Empty response from Claude');

        // If within range, done.
        $len = mb_strlen($text);
        if ($len >= $min && $len <= $max) return $text;

        // Out of range — give Claude one correction attempt with the exact count.
        $direction      = $len > $max ? 'too long' : 'too short';
        $correction_msg = "That description is {$direction} at {$len} characters. "
            . "Rewrite it so it is between {$min} and {$max} characters. "
            . "Output ONLY the new description text, nothing else.";

        $payload['messages'][] = ['role' => 'assistant', 'content' => $text];
        $payload['messages'][] = ['role' => 'user',      'content' => $correction_msg];

        $response2 = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'timeout' => 30,
            'headers' => [
                'Content-Type'      => 'application/json',
                'x-api-key'         => $key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode($payload),
        ]);

        if (!is_wp_error($response2) && wp_remote_retrieve_response_code($response2) === 200) {
            $body2 = json_decode(wp_remote_retrieve_body($response2), true);
            $text2 = trim(trim($body2['content'][0]['text'] ?? ''), '"\'');
            if ($text2 && mb_strlen($text2) >= $min && mb_strlen($text2) <= $max) {
                return $text2;
            }
            if ($text2) return $text2;
        }

        // Fallback: return original even if slightly off.
        return $text;
    }

    public function ajax_generate_one(): void {
        $this->ajax_check();
        $post_id = (int) sanitize_key( wp_unslash( $_POST['post_id'] ?? 0 ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verified in ajax_check()
        if (!$post_id) wp_send_json_error('Missing post_id');

        try {
            $desc = $this->call_anthropic($post_id);
            update_post_meta($post_id, self::META_DESC, sanitize_textarea_field($desc));
            wp_send_json_success([
                'post_id'     => $post_id,
                'description' => $desc,
                'chars'       => mb_strlen($desc),
            ]);
        } catch (\Throwable $e) {
            wp_send_json_error($e->getMessage());
        }
    }

    /**
     * Fix an existing description that is too short or too long.
     * Sends the current description back to Claude with post context and explicit correction instruction.
     */
    private function call_anthropic_fix(int $post_id, string $existing_desc): string {
        $post = get_post($post_id);
        if (!$post) throw new \RuntimeException( esc_html( "Post {$post_id} not found" ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped

        $key    = trim((string) $this->ai_opts['anthropic_key']);
        $model  = trim((string) $this->ai_opts['model']) ?: 'claude-sonnet-4-20250514';
        $prompt = trim((string) $this->ai_opts['prompt']) ?: self::default_prompt();
        $min    = max(100, (int) $this->ai_opts['min_chars']);
        $max    = min(200, (int) $this->ai_opts['max_chars']);

        if (!$key) throw new \RuntimeException('No Anthropic API key configured');

        $len       = mb_strlen($existing_desc);
        $direction = $len > $max ? 'too long' : 'too short';

        $content  = $this->text_from_html((string) $post->post_content);
        $content  = mb_substr($content, 0, 6000);

        // Build site context same as call_anthropic.
        $site_context_parts = [];
        $site_name = trim((string) $this->opts['site_name']);
        $person    = trim((string) $this->opts['person_name']);
        $job_title = trim((string) $this->opts['person_job_title']);
        $home_desc = trim((string) $this->opts['home_desc']);
        $def_desc  = trim((string) $this->opts['default_desc']);

        if ($site_name) $site_context_parts[] = "Site name: {$site_name}";
        if ($person)    $site_context_parts[] = "Author: {$person}" . ($job_title ? ", {$job_title}" : '');
        $topic = $home_desc ?: $def_desc;
        if ($topic) $site_context_parts[] = "Site description: {$topic}";

        $site_context = $site_context_parts
            ? "\n\nSITE CONTEXT:\n" . implode("\n", $site_context_parts)
            : '';

        $system = $prompt . $site_context
            . "\n\nCHARACTER REQUIREMENT: The description MUST be between {$min} and {$max} characters including spaces. Count every character carefully. Do not produce output outside this range.";

        // First message gives Claude the article context.
        // Second simulates Claude having written the bad description.
        // Third is the correction instruction — Claude knows exactly what it wrote and why it failed.
        $user_msg = "Article title: \"{$post->post_title}\"\n\nArticle content:\n{$content}";

        $correction = "The existing meta description for this article is {$direction} at {$len} characters:\n\n"
            . "\"{$existing_desc}\"\n\n"
            . "Rewrite it so it is between {$min} and {$max} characters. Keep the meaning and keyword focus. "
            . "Output ONLY the rewritten description, nothing else.";

        $payload = [
            'model'      => $model,
            'max_tokens' => 300,
            'system'     => $system,
            'messages'   => [
                ['role' => 'user',      'content' => $user_msg],
                ['role' => 'assistant', 'content' => $existing_desc],
                ['role' => 'user',      'content' => $correction],
            ],
        ];

        $do_request = function() use ($key, $payload) {
            return wp_remote_post('https://api.anthropic.com/v1/messages', [
                'timeout' => 45,
                'headers' => [
                    'Content-Type'      => 'application/json',
                    'x-api-key'         => $key,
                    'anthropic-version' => '2023-06-01',
                ],
                'body' => wp_json_encode($payload),
            ]);
        };

        $response = $do_request();
        if (is_wp_error($response)) throw new \RuntimeException( esc_html( 'HTTP error: ' . $response->get_error_message() ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped

        if (wp_remote_retrieve_response_code($response) === 429) {
            sleep(10);
            $response = $do_request();
            if (is_wp_error($response)) throw new \RuntimeException( esc_html( 'HTTP error after retry: ' . $response->get_error_message() ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200) {
            $msg = $body['error']['message'] ?? "API returned HTTP {$code}";
            throw new \RuntimeException( esc_html( $msg ) ); // phpcs:ignore WordPress.Security.EscapeOutput.ExceptionNotEscaped
        }

        $text = trim(trim($body['content'][0]['text'] ?? ''), '"\'');
        if (!$text) throw new \RuntimeException('Empty response from Claude');

        // Retry up to 3 times if still out of range, each time telling Claude the exact count.
        $attempt = 0;
        while (mb_strlen($text) < $min || mb_strlen($text) > $max) {
            if (++$attempt > 3) break;

            $current_len = mb_strlen($text);
            $direction   = $current_len > $max ? 'too long' : 'too short';

            $payload['messages'][] = ['role' => 'assistant', 'content' => $text];
            $payload['messages'][] = ['role' => 'user', 'content'
                => "FAILED. Your previous response was {$current_len} characters which is {$direction}. "
                . "You did not follow the instructions. "
                . "The description MUST be between {$min} and {$max} characters — this is a hard requirement. "
                . "Before you write anything, count out {$min} to {$max} characters in your head, then write a description that fits exactly within that count. "
                . "Check your character count before outputting. "
                . "Output ONLY the description text, no explanation, no quotes, nothing else."];

            $retry = $do_request();
            if (is_wp_error($retry)) break;
            if (wp_remote_retrieve_response_code($retry) === 429) {
                sleep(10);
                $retry = $do_request();
            }
            if (is_wp_error($retry) || wp_remote_retrieve_response_code($retry) !== 200) break;

            $retry_body = json_decode(wp_remote_retrieve_body($retry), true);
            $retry_text = trim(trim($retry_body['content'][0]['text'] ?? ''), '"\'');
            if (!$retry_text) break;
            $text = $retry_text;
        }

        return $text;
    }

    public function ajax_fix_desc(): void {
        $this->ajax_check();
        $post_id = (int) sanitize_key( wp_unslash( $_POST['post_id'] ?? 0 ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verified in ajax_check()
        if (!$post_id) wp_send_json_error('Missing post_id');

        $min = max(100, (int) $this->ai_opts['min_chars']);
        $max = min(200, (int) $this->ai_opts['max_chars']);

        $existing = trim((string) get_post_meta($post_id, self::META_DESC, true));
        $len      = mb_strlen($existing);

        // Only fix if actually out of range.
        if (!$existing) {
            wp_send_json_success(['post_id' => $post_id, 'status' => 'skipped', 'message' => 'No description to fix']);
            return;
        }
        if ($len >= $min && $len <= $max) {
            wp_send_json_success(['post_id' => $post_id, 'status' => 'skipped', 'message' => 'Already in range (' . $len . ' chars)']);
            return;
        }

        try {
            $desc      = $this->call_anthropic_fix($post_id, $existing);
            $new_len   = mb_strlen($desc);
            $in_range  = ($new_len >= $min && $new_len <= $max);
            update_post_meta($post_id, self::META_DESC, sanitize_textarea_field($desc));
            wp_send_json_success([
                'post_id'       => $post_id,
                'status'        => $in_range ? 'fixed' : 'fixed_imperfect',
                'description'   => $desc,
                'chars'         => $new_len,
                'was_chars'     => $len,
                'in_range'      => $in_range,
                'message'       => ($in_range
                    ? 'Fixed: was ' . $len . ' chars, now ' . $new_len . ' chars'
                    : 'Saved but still out of range: was ' . $len . ' chars, now ' . $new_len . ' chars'),
            ]);
        } catch (\Throwable $e) {
            wp_send_json_error(['post_id' => $post_id, 'message' => $e->getMessage()]);
        }
    }

    /**
     * Generate all — called once per post by the JS polling loop.
     * Returns result for a single post_id; JS calls this repeatedly.
     */
    public function ajax_generate_all(): void {
        $this->ajax_check();
        $post_id   = (int) sanitize_key( wp_unslash( $_POST['post_id'] ?? 0 ) );   // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verified in ajax_check()
        $overwrite = (int) sanitize_key( wp_unslash( $_POST['overwrite'] ?? 0 ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing

        if (!$post_id) wp_send_json_error('Missing post_id');

        // Skip if already has a description and overwrite is off.
        $existing = trim((string) get_post_meta($post_id, self::META_DESC, true));
        if ($existing && !$overwrite) {
            wp_send_json_success([
                'post_id'     => $post_id,
                'status'      => 'skipped',
                'description' => $existing,
                'chars'       => mb_strlen($existing),
                'message'     => 'Skipped — description already exists',
            ]);
            return;
        }

        try {
            $desc = $this->call_anthropic($post_id);
            update_post_meta($post_id, self::META_DESC, sanitize_textarea_field($desc));
            wp_send_json_success([
                'post_id'     => $post_id,
                'status'      => 'generated',
                'description' => $desc,
                'chars'       => mb_strlen($desc),
                'message'     => 'Generated: ' . mb_strlen($desc) . ' chars',
            ]);
        } catch (\Throwable $e) {
            wp_send_json_error([
                'post_id' => $post_id,
                'message' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Return paginated list of posts with their current SEO desc status.
     */
    public function ajax_get_posts(): void {
        global $wpdb;
        $this->ajax_check();
        $page = max(1, (int) sanitize_key( wp_unslash( $_POST['page'] ?? 1 ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verified in ajax_check()
        $per_page = 50;

        $q = new WP_Query([
            'post_type'           => ['post', 'page'],
            'post_status'         => 'publish',
            'posts_per_page'      => $per_page,
            'paged'               => $page,
            'orderby'             => 'date',
            'order'               => 'DESC',
            'ignore_sticky_posts' => true,
        ]);

        $items = [];
        foreach ($q->posts as $p) {
            $desc = trim((string) get_post_meta($p->ID, self::META_DESC, true));
            $items[] = [
                'id'          => $p->ID,
                'title'       => get_the_title($p->ID),
                'type'        => $p->post_type,
                'date'        => get_the_date('Y-m-d', $p->ID),
                'has_desc'    => $desc !== '',
                'desc'        => $desc,
                'desc_chars'  => mb_strlen($desc),
            ];
        }

        wp_send_json_success([
            'posts'          => $items,
            'total'          => (int) $q->found_posts,
            'total_pages'    => (int) $q->max_num_pages,
            'page'            => $page,
            'total_with_desc' => (int) $wpdb->get_var( // phpcs:ignore WordPress.DB.DirectDatabaseQuery
                $wpdb->prepare(
                    "SELECT COUNT(DISTINCT p.ID)
                     FROM {$wpdb->posts} p
                     INNER JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID
                     WHERE p.post_type IN ('post','page')
                     AND p.post_status = 'publish'
                     AND pm.meta_key = %s
                     AND pm.meta_value != ''",
                    self::META_DESC
                )
            ),
        ]);
    }

    /**
     * Test the stored Anthropic API key with a minimal API call.
     */
    public function ajax_test_key(): void {
        $this->ajax_check();
        // Prefer the key passed directly from the field (pre-save test).
        // Fall back to the saved key if nothing passed.
        $key = sanitize_text_field( wp_unslash( $_POST['live_key'] ?? $this->ai_opts['anthropic_key'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Missing -- verified in ajax_check()
        if (!$key) wp_send_json_error('No API key entered');

        $response = wp_remote_post('https://api.anthropic.com/v1/messages', [
            'timeout' => 15,
            'headers' => [
                'Content-Type'      => 'application/json',
                'x-api-key'         => $key,
                'anthropic-version' => '2023-06-01',
            ],
            'body' => wp_json_encode([
                'model'      => $this->ai_opts['model'] ?: 'claude-sonnet-4-20250514',
                'max_tokens' => 10,
                'messages'   => [['role' => 'user', 'content' => 'Reply with: OK']],
            ]),
        ]);

        if (is_wp_error($response)) {
            wp_send_json_error('Connection failed: ' . $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code === 200) {
            wp_send_json_success('API key valid. Model: ' . ($body['model'] ?? 'unknown'));
        } else {
            $msg = $body['error']['message'] ?? "HTTP {$code}";
            wp_send_json_error("API error: {$msg}");
        }
    }

    // =========================================================================
    // Admin menu & settings
    // =========================================================================

    public function admin_notices(): void {
        // Show post-rename confirmation if the backup flag is set
        $bak = get_option('cs_seo_robots_bak');
        if ($bak !== false) {
            // Dismiss handler
            if (isset($_GET['_cs_dismiss_robotsbak']) && check_admin_referer('cs_dismiss_robotsbak')) {
                delete_option('cs_seo_robots_bak');
            } else {
                echo '<div class="notice notice-success is-dismissible">';
                echo '<p><strong>CloudScale SEO:</strong> The physical <code>robots.txt</code> file has been renamed to <code>robots.txt.bak</code>. ';
                echo 'The plugin is now managing your robots.txt. Your original rules have been preserved — review the Robots.txt card and merge anything you want to keep.</p>';
                echo '<p><a href="' . esc_url(wp_nonce_url(admin_url('tools.php?page=cs-seo-optimizer&_cs_dismiss_robotsbak=1'), 'cs_dismiss_robotsbak')) . '">Dismiss</a></p>';
                echo '</div>';
            }
        }
    }

    public function admin_head_css(): void {
        if (!$this->is_our_page()) return;
        echo '<style>#wpfooter { display:none !important; } #wpcontent, #wpbody-content { padding-bottom:0 !important; }</style>';
    }

    /**
     * Render an "? Explain" button and its modal.
     * $id      — unique ID suffix, e.g. 'identity'
     * $title   — modal heading
     * $items   — array of ['rec' => '✅ Recommended', 'name' => '...', 'desc' => '...']
     *            rec values: '✅ Recommended' | '⬜ Optional' | 'ℹ️ Info'
     */
    private function explain_btn(string $id, string $title, array $items): void {
        $btn_id   = 'ab-explain-btn-' . $id;
        $modal_id = 'ab-explain-modal-' . $id;
        ?>
        <button type="button" id="<?php echo esc_attr($btn_id); ?>"
            onclick="document.getElementById('<?php echo esc_attr($modal_id); ?>').style.display='flex'"
            style="background:rgba(255,255,255,0.2);border:1px solid rgba(255,255,255,0.4);border-radius:5px;color:#fff;font-size:12px;font-weight:600;padding:5px 14px;cursor:pointer">
            ? Explain
        </button>
        <div id="<?php echo esc_attr($modal_id); ?>" style="display:none;position:fixed;inset:0;z-index:99999;background:rgba(0,0,0,0.6);align-items:center;justify-content:center;padding:16px">
            <div style="background:#fff;border-radius:10px;max-width:640px;width:100%;max-height:88vh;overflow-y:auto;box-shadow:0 20px 60px rgba(0,0,0,0.4)">
                <div style="background:#1a4a7a;border-radius:10px 10px 0 0;padding:16px 20px;display:flex;justify-content:space-between;align-items:center">
                    <strong style="color:#fff;font-size:15px"><?php echo esc_html($title); ?></strong>
                    <button type="button" onclick="document.getElementById('<?php echo esc_attr($modal_id); ?>').style.display='none'"
                        style="background:rgba(255,255,255,0.2);border:1px solid rgba(255,255,255,0.4);border-radius:5px;color:#fff;font-size:16px;font-weight:700;padding:2px 10px;cursor:pointer;line-height:1">✕</button>
                </div>
                <div style="padding:20px 24px;font-size:13px;line-height:1.6;color:#1d2327">
                    <?php foreach ($items as $item):
                        $rec = $item['rec'];
                        $is_on  = strpos($rec, 'Recommended') !== false;
                        $is_opt = strpos($rec, 'Optional') !== false;
                        $bg  = $is_on ? '#edfaef' : ($is_opt ? '#f6f7f7' : '#f0f6fc');
                        $col = $is_on ? '#1a7a34' : ($is_opt ? '#50575e' : '#1a4a7a');
                        $bdr = $is_on ? '#1a7a34' : ($is_opt ? '#c3c4c7' : '#2271b1');
                    ?>
                    <div style="border:1px solid #e0e0e0;border-radius:6px;padding:12px 14px;margin-bottom:10px">
                        <div style="display:flex;align-items:center;gap:10px;margin-bottom:5px;flex-wrap:wrap">
                            <strong style="font-size:13px"><?php echo esc_html($item['name']); ?></strong>
                            <span style="background:<?php echo esc_attr($bg); ?>;color:<?php echo esc_attr($col); ?>;border:1px solid <?php echo esc_attr($bdr); ?>;border-radius:4px;font-size:11px;font-weight:600;padding:1px 8px;white-space:nowrap"><?php echo esc_html($rec); ?></span>
                        </div>
                        <p style="margin:0;color:#50575e;font-size:12px;line-height:1.5"><?php echo esc_html($item['desc']); ?></p>
                    </div>
                    <?php endforeach; ?>
                </div>
                <div style="padding:12px 24px 20px;text-align:right">
                    <button type="button" onclick="document.getElementById('<?php echo esc_attr($modal_id); ?>').style.display='none'"
                        style="background:#1a4a7a;border:none;border-radius:6px;color:#fff;font-size:13px;font-weight:600;padding:8px 24px;cursor:pointer">
                        Got it
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    public function admin_footer_text($text): string {
        if (!$this->is_our_page()) return $text;
        return '';
    }

    public function admin_footer_version($text): string {
        if (!$this->is_our_page()) return $text;
        return '';
    }

    private function is_our_page(): bool {
        return isset($_GET['page']) && sanitize_key(wp_unslash($_GET['page'])) === 'cs-seo-optimizer';
    }

    public function admin_menu(): void {
        add_management_page(
            'CloudScale SEO AI Optimizer',
            'CloudScale SEO AI',
            'manage_options',
            'cs-seo-optimizer',
            [$this, 'settings_page']
        );
    }

    public function register_settings(): void {
        register_setting('cs_seo_group', self::OPT, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_opts'],
            'default'           => self::defaults(),
        ]);
        register_setting('cs_seo_ai_group', self::AI_OPT, [
            'type'              => 'array',
            'sanitize_callback' => [$this, 'sanitize_ai_opts'],
            'default'           => self::ai_defaults(),
        ]);
    }

    public function sanitize_opts($in): array {
        $in  = is_array($in) ? $in : [];
        $d   = self::defaults();

        // If the incoming data has no recognisable fields it's likely a spurious
        // call (e.g. plugin reinstall touching the option). Preserve existing data.
        $known_fields = ['site_name','enable_og','robots_txt','sitemap_post_types','enable_sitemap',
                         'home_title','person_name','block_ai_bots','noindex_search','title_suffix'];
        $has_known = false;
        foreach ($known_fields as $f) {
            if (array_key_exists($f, $in)) { $has_known = true; break; }
        }
        if (!$has_known) {
            return $this->opts ?: $d;
        }

        $out = [];

        // Merge with existing saved values — partial form submissions (e.g. Sitemap tab only)
        // must not wipe fields that live on other tabs.
        $existing = $this->opts ?: $d;

        $was_sitemap = (int)($existing['enable_sitemap'] ?? 0);
        $now_sitemap = array_key_exists('enable_sitemap', $in) ? (empty($in['enable_sitemap']) ? 0 : 1) : $was_sitemap;
        if ($now_sitemap !== $was_sitemap) {
            add_action('shutdown', 'flush_rewrite_rules');
        }
        foreach (['site_name','site_lang','title_suffix','home_title','twitter_handle','person_name','person_job_title'] as $k) {
            $out[$k] = sanitize_text_field(array_key_exists($k, $in) ? (string)$in[$k] : (string)($existing[$k] ?? $d[$k]));
        }
        foreach (['home_desc','default_desc','sameas','robots_txt','sitemap_exclude'] as $k) {
            $out[$k] = sanitize_textarea_field(array_key_exists($k, $in) ? (string)$in[$k] : (string)($existing[$k] ?? $d[$k]));
        }
        foreach (['default_og_image','person_url','person_image'] as $k) {
            $out[$k] = esc_url_raw(array_key_exists($k, $in) ? (string)$in[$k] : (string)($existing[$k] ?? $d[$k]));
        }
        foreach ([
            'enable_og','enable_schema_person','enable_schema_website','enable_schema_article',
            'enable_schema_breadcrumbs','strip_tracking_params','enable_sitemap',
            'noindex_search','noindex_404','noindex_attachment','noindex_author_archives','noindex_tag_archives',
            'block_ai_bots','sitemap_taxonomies',
        ] as $k) {
            $out[$k] = array_key_exists($k, $in) ? (empty($in[$k]) ? 0 : 1) : (int)($existing[$k] ?? $d[$k]);
        }
        // Sitemap post types — array of sanitized strings
        $allowed_types = array_map(fn($pt) => $pt->name, get_post_types(['public' => true], 'objects'));
        if (array_key_exists('sitemap_post_types', $in)) {
            $chosen = array_intersect((array)$in['sitemap_post_types'], $allowed_types);
            $out['sitemap_post_types'] = array_values($chosen) ?: ['post'];
        } else {
            $out['sitemap_post_types'] = $existing['sitemap_post_types'] ?? $d['sitemap_post_types'];
        }
        return $out;
    }

    public function sanitize_ai_opts($in): array {
        $in      = is_array($in) ? $in : [];
        $d       = self::ai_defaults();
        $current = $this->get_ai_opts(); // existing saved values — preserve anything not in $in

        $days = array_intersect(
            (array)($in['schedule_days'] ?? $current['schedule_days'] ?? []),
            ['sun','mon','tue','wed','thu','fri','sat']
        );
        $was_enabled = (int) $current['schedule_enabled'];
        $now_enabled = array_key_exists('schedule_enabled', $in) ? (empty($in['schedule_enabled']) ? 0 : 1) : $was_enabled;

        // Schedule cron when enabled, unschedule when disabled.
        if ($now_enabled && !$was_enabled) {
            if (!wp_next_scheduled('cs_seo_daily_batch')) {
                wp_schedule_event(strtotime('tomorrow midnight'), 'daily', 'cs_seo_daily_batch');
            }
        } elseif (!$now_enabled && $was_enabled) {
            wp_clear_scheduled_hook('cs_seo_daily_batch');
        }

        // Use submitted value if present, otherwise fall back to current saved value, then default.
        return [
            'anthropic_key'    => sanitize_text_field((string)(array_key_exists('anthropic_key', $in) ? $in['anthropic_key'] : $current['anthropic_key'])),
            'model'            => sanitize_text_field((string)($in['model'] ?? $current['model'] ?? $d['model'])),
            'overwrite'        => array_key_exists('overwrite', $in) ? (empty($in['overwrite']) ? 0 : 1) : ($current['overwrite'] ?? 0),
            'min_chars'        => max(100, min(160, (int)($in['min_chars'] ?? $current['min_chars'] ?? $d['min_chars']))),
            'max_chars'        => max(100, min(200, (int)($in['max_chars'] ?? $current['max_chars'] ?? $d['max_chars']))),
            'prompt'           => sanitize_textarea_field((string)($in['prompt'] ?? $current['prompt'] ?? $d['prompt'])),
            'schedule_enabled' => $now_enabled,
            'schedule_days'    => array_values($days),
        ];
    }

    // =========================================================================
    // Settings page
    // =========================================================================

    public function settings_page(): void {
        if (!current_user_can('manage_options')) return;
        // Clear OPCache for this file so updated code is always used.
        if (function_exists('opcache_invalidate')) {
            opcache_invalidate(__FILE__, true);
        }
        $o  = $this->get_opts();
        $ai = $this->get_ai_opts();
        $nonce = wp_create_nonce('cs_seo_nonce');
        ?>
        <div class="wrap">
        <h1>CloudScale SEO AI Optimizer</h1>

        <?php /* ── TAB NAV ── */ ?>
        <style>
            .ab-tabs {
                display:flex; gap:6px; margin:20px 0 0; padding:0;
                border-bottom:3px solid #1d2327;
            }
            .ab-tab {
                padding:10px 22px; cursor:pointer;
                border:none; border-radius:6px 6px 0 0;
                font-size:13px; font-weight:600; letter-spacing:0.01em;
                background:#e0e0e0; color:#50575e;
                transition:background 0.15s, color 0.15s;
                margin-bottom:0; position:relative; bottom:-1px;
            }
            .ab-tab:hover:not(.active) { background:#c3c4c7; color:#1d2327; }
            .ab-tab[data-tab="seo"].active    { background:#2271b1; color:#fff; }
            .ab-tab[data-tab="sitemap"].active { background:#1a7a34; color:#fff; }
            .ab-tab[data-tab="batch"].active  { background:#e67e00; color:#fff; }
            .ab-pane { display:none; padding-top:24px; }
            .ab-pane.active { display:block; }
            /* AI Writer styles */
            #ab-ai-writer { font-family: -apple-system, sans-serif; }
            .ab-ai-toolbar { display:flex; gap:10px; align-items:center; margin-bottom:16px; flex-wrap:wrap; }
            #ab-log { background:#1a1a2e; color:#e0e0f0; font-family:'Courier New',monospace;
                      font-size:12px; padding:14px; border-radius:6px; max-height:260px;
                      overflow-y:auto; margin:16px 0; display:none; border:1px solid #2a2a4a; }
            #ab-log.visible { display:block; }
            .ab-log-ok   { color:#00d084; }
            .ab-log-err  { color:#ff6b6b; }
            .ab-log-skip { color:#f0c040; }
            .ab-log-info { color:#8080b0; }
            .ab-progress { background:#f0f0f1; border-radius:4px; height:8px; margin:8px 0 4px; overflow:hidden; display:none; }
            .ab-progress.visible { display:block; }
            .ab-progress-fill { height:100%; background:#2271b1; border-radius:4px; transition:width 0.3s; width:0%; }
            .ab-stats { font-size:12px; color:#50575e; margin-bottom:12px; }
            .ab-stat-val { font-weight:600; color:#1d2327; }
            table.ab-posts { width:100%; border-collapse:collapse; margin-top:12px; }
            table.ab-posts th { text-align:left; padding:8px 10px; border-bottom:2px solid #c3c4c7;
                                font-size:12px; color:#50575e; font-weight:600; }
            table.ab-posts td { padding:9px 10px; border-bottom:1px solid #f0f0f1; font-size:13px; vertical-align:top; }
            table.ab-posts tr:hover td { background:#f6f7f7; }
            .ab-badge { display:inline-block; padding:2px 8px; border-radius:3px; font-size:11px; font-weight:600; }
            .ab-badge-none   { background:#e8f3fb; color:#1a4a7a; border:1px solid #b2cfe0; }
            .ab-badge-ok     { background:#edfaef; color:#1a7a34; border:1px solid #b2dfc0; }
            .ab-badge-short  { background:#fcf9e8; color:#7a5c00; border:1px solid #f0d676; }
            .ab-badge-long   { background:#fcf0ef; color:#8a2424; border:1px solid #f5bcbb; }
            .ab-badge-gen    { background:#e8f3fb; color:#1a4a7a; border:1px solid #b2cfe0; }
            .ab-desc-text { font-size:12px; color:#50575e; margin-top:3px; line-height:1.4; word-wrap:break-word; white-space:normal; }
            .ab-desc-gen  { font-size:12px; color:#1a4a7a; margin-top:4px; background:#e8f3fb;
                            border-left:3px solid #2271b1; padding:4px 8px; border-radius:0 3px 3px 0; }
            .ab-row-btn { font-size:11px; padding:3px 8px; }
            .ab-key-row { display:flex; gap:8px; align-items:center; }
            .ab-key-status { font-size:12px; font-weight:600; }
            .ab-key-ok  { color:#1a7a34; }
            .ab-key-err { color:#8a2424; }
            #ab-ai-gen-all { position:relative; }
            .ab-spinner { display:inline-block; animation:ab-spin 0.8s linear infinite; margin-right:4px; }
            @keyframes ab-spin { to { transform:rotate(360deg); } }
            .ab-pager { display:flex; gap:8px; align-items:center; margin-top:12px; }
            .ab-summary-row { display:grid; grid-template-columns:repeat(4,1fr); gap:12px; margin:12px 0; }
            .ab-summary-card { background:#f6f7f7; border:1px solid #c3c4c7; border-radius:6px;
                               padding:12px; text-align:center; }
            .ab-summary-num  { font-size:24px; font-weight:700; color:#2271b1; line-height:1; }
            .ab-summary-lbl  { font-size:11px; color:#50575e; margin-top:4px; }
            /* Section zone headers */
            /* ── Zone cards (matching CloudScale Backup style) ── */
            .ab-zone-card {
                border-radius:8px; overflow:hidden;
                box-shadow:0 2px 8px rgba(0,0,0,0.10);
                margin:24px 0 0;
            }
            .ab-zone-header {
                display:flex; align-items:center; gap:10px;
                padding:13px 20px;
                font-size:15px; font-weight:700; color:#fff;
                letter-spacing:0.01em;
            }
            .ab-zone-header .ab-zone-icon { font-size:17px; }
            .ab-zone-body {
                background:#fff;
                padding:4px 0 8px;
            }
            .ab-zone-body .form-table th { padding-left:20px; }
            .ab-zone-body .form-table td { padding-right:20px; }
            /* Colour per section — matches backup plugin palette */
            .ab-zone-card.ab-card-identity .ab-zone-header  { background:#2271b1; } /* blue   */
            .ab-zone-card.ab-card-features .ab-zone-header  { background:#1a7a34; } /* green  */
            .ab-checkbox-grid {
                display:grid; grid-template-columns:repeat(3,1fr); gap:10px 24px; padding:4px 0 8px;
            }
            .ab-checkbox-grid label { display:flex; align-items:flex-start; gap:6px; font-size:13px; cursor:pointer; }
            .ab-zone-card.ab-card-robots .ab-zone-header    { background:#b45309; } /* amber  */
            .ab-physical-robots-warn {
                display:flex; gap:16px; align-items:flex-start;
                background:#fff8e1; border:2px solid #f0ad00;
                border-radius:6px; padding:16px 20px; margin:16px 20px 4px;
                font-size:13px; line-height:1.6;
            }
            .ab-zone-card.ab-card-person .ab-zone-header    { background:#6b3fa0; } /* purple */
            .ab-zone-card.ab-card-ai .ab-zone-header        { background:#c3372b; } /* red    */
            .ab-zone-card.ab-card-ai .ab-zone-body          { background:#fdf7f7; }
            .ab-zone-card.ab-card-schedule .ab-zone-header  { background:#e67e00; } /* orange */
            .ab-zone-card.ab-card-lastrun  .ab-zone-header  { background:#1a4a7a; } /* dark blue */
            .ab-zone-card.ab-card-sitemap-settings .ab-zone-header { background:#1a7a34; }
            .ab-zone-card.ab-card-sitemap-preview .ab-zone-header  { background:#0e5229; }
            /* Sitemap preview table */
            .ab-sitemap-url { font-size:12px; color:#2271b1; word-break:break-all; }
            .ab-sitemap-type { font-size:11px; font-weight:600; padding:2px 7px; border-radius:3px; white-space:nowrap; }
            .ab-sitemap-type-home { background:#f0e6fa; color:#6b3fa0; }
            .ab-sitemap-type-post { background:#e8f3fb; color:#1a4a7a; }
            .ab-sitemap-type-page { background:#edfaef; color:#1a7a34; }
            .ab-sitemap-type-tax  { background:#fcf9e8; color:#7a5c00; }
            .ab-sitemap-type-cpt  { background:#fef0e8; color:#8a3a00; }
            table.ab-sitemap-tbl { width:100%; border-collapse:collapse; font-size:13px; }
            table.ab-sitemap-tbl th { text-align:left; padding:8px 12px; border-bottom:2px solid #c3c4c7;
                                      font-size:12px; color:#50575e; font-weight:600; }
            table.ab-sitemap-tbl td { padding:8px 12px; border-bottom:1px solid #f0f0f1; vertical-align:middle; }
            table.ab-sitemap-tbl tr:hover td { background:#f6f7f7; }
            .ab-sitemap-count { font-size:13px; color:#50575e; margin:0 0 12px; }
            .ab-sitemap-count strong { color:#1d2327; }
            .ab-zone-card.ab-card-update-posts .ab-zone-header { background:#1d2327; font-size:17px; padding:16px 22px; }
            .ab-zone-card.ab-card-update-posts .ab-zone-header .ab-zone-icon { color:#f0c040; font-size:20px; }
            /* Load Posts CTA strip */
            .ab-load-cta {
                display:flex; align-items:center; gap:18px;
                background:linear-gradient(135deg, #1d2327 0%, #2c3338 100%);
                border-radius:6px; padding:20px 24px; margin-bottom:20px;
                border-left:5px solid #f0c040;
            }
            .ab-load-cta-icon { font-size:32px; line-height:1; flex-shrink:0; }
            .ab-load-cta-text { flex:1; }
            .ab-load-cta-text strong { display:block; color:#fff; font-size:15px; margin-bottom:3px; }
            .ab-load-cta-text span { color:#a7aaad; font-size:13px; }
            .ab-load-btn {
                flex-shrink:0;
                background:#f0c040 !important; border-color:#d4a800 !important;
                color:#1d2327 !important; font-weight:700 !important;
                font-size:15px !important; padding:10px 28px !important;
                border-radius:4px; cursor:pointer; white-space:nowrap;
                box-shadow:0 2px 6px rgba(0,0,0,0.25);
            }
            .ab-load-btn:hover { background:#f5d060 !important; }
            /* Action buttons in toolbar */
            .ab-action-btn { font-size:13px !important; padding:6px 14px !important; height:auto !important; }
            .ab-fix-btn  { background:#e67e00 !important; border-color:#c26900 !important; color:#fff !important; }
            .ab-regen-btn { background:#1a7a34 !important; border-color:#155f28 !important; color:#fff !important; }
            .ab-zone-divider {
                border:none; border-top:2px solid #dcdcde;
                margin:32px 0 0; opacity:1;
            }
            /* Force black text in all plugin textareas — overrides mobile browser defaults */
            #cs-robots-txt,
            textarea[name="cs_seo_opts[sitemap_exclude]"],
            textarea[name="cs_seo_opts[home_desc]"],
            textarea[name="cs_seo_opts[default_desc]"],
            textarea[name="cs_seo_opts[sameas]"] {
                color:#1d2327 !important;
            }
            /* Field hints — italic, muted green */
            .ab-zone-body p.description,
            .ab-zone-body .description {
                color:#2a7a3a !important;
                font-style:italic !important;
                font-size:12px !important;
            }
            /* Form labels — bold, with colon */
            .ab-zone-body .form-table th,
            .ab-zone-body .form-table th label {
                font-weight:700 !important;
                color:#1d2327 !important;
            }
            .ab-api-key-warning {
                display:none; align-items:flex-start; gap:12px;
                background:#fff8e1; border:2px solid #f0ad00;
                border-radius:6px; padding:14px 18px; margin:0 0 16px;
            }
            .ab-api-key-warning.visible { display:flex; }
            .ab-api-key-warning .ab-warn-icon { font-size:22px; line-height:1; flex-shrink:0; }
            .ab-api-key-warning .ab-warn-body { font-size:13px; color:#1d2327; }
            .ab-api-key-warning .ab-warn-body strong { display:block; margin-bottom:4px; font-size:14px; }
            .ab-api-key-warning .ab-warn-body a { color:#2271b1; font-weight:600; }
        </style>

        <div class="ab-tabs">
            <button class="ab-tab active" data-tab="seo"     onclick="abTab('seo',this)">⚙ Optimise SEO</button>
            <button class="ab-tab"        data-tab="sitemap" onclick="abTab('sitemap',this)">🗺 Sitemap &amp; Robots</button>
            <button class="ab-tab"        data-tab="batch"   onclick="abTab('batch',this)">⏱ Scheduled Batch</button>
        </div>
        </div>

        <?php /* ══════════════════ SETTINGS PANE (SEO + AI combined) ══════════════════ */ ?>
        <div class="ab-pane active" id="ab-pane-seo">

            <?php /* ── SEO Settings form ── */ ?>
            <form method="post" action="options.php">
                <?php settings_fields('cs_seo_group'); ?>

                <div class="ab-zone-card ab-card-identity">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">🌐</span> Site Identity</span>
                    <?php $this->explain_btn('identity', '🌐 Site Identity — What each field does', [
                        ['rec'=>'✅ Recommended','name'=>'Site name','desc'=>'The name of your site as it appears in search results, browser tabs, and social sharing. Used in JSON-LD schema and OpenGraph tags. e.g. "Andrew Baker" or "Andrew Baker\'s Tech Blog".'],
                        ['rec'=>'✅ Recommended','name'=>'Title suffix','desc'=>'Appended to every page title in search results. e.g. if your suffix is "| Andrew Baker" then a post titled "AWS Lambda Tips" appears as "AWS Lambda Tips | Andrew Baker". Helps with brand recognition in SERPs.'],
                        ['rec'=>'✅ Recommended','name'=>'Home title','desc'=>'The SEO title for your homepage specifically. This is what Google shows as the blue link for your homepage in search results. Make it descriptive and keyword-rich — e.g. "Andrew Baker – CIO, Cloud Architect & Technology Leader".'],
                        ['rec'=>'✅ Recommended','name'=>'Home description','desc'=>'The meta description for your homepage. Shown as the snippet under your homepage title in Google. Aim for 140–155 characters. Write for humans — this is your elevator pitch to someone seeing your site for the first time.'],
                        ['rec'=>'✅ Recommended','name'=>'Default OG image URL','desc'=>'The fallback image used when a post is shared on social media and has no featured image. Should be 1200×630px. Use a branded image with your name/logo — this appears as the preview card on LinkedIn, Twitter/X, and WhatsApp.'],
                        ['rec'=>'⬜ Optional','name'=>'Locale','desc'=>'BCP 47 language tag used in OpenGraph metadata. "en-US" is fine for most English sites. Use "en-ZA" if you want to signal a South African audience to Facebook/LinkedIn. Has minimal impact on Google rankings.'],
                        ['rec'=>'⬜ Optional','name'=>'Twitter handle','desc'=>'Your Twitter/X username including the @ symbol. Added to Twitter Card metadata so when your posts are shared on X, your account gets attributed as the author. Only matters if you actively use Twitter/X.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body">
                <table class="form-table" role="presentation">
                    <tr>
                        <th><label>Site name:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[site_name]" value="<?php echo esc_attr((string)($o['site_name'] ?? '')); ?>" placeholder="My Tech Blog">
                        <p class="description">Used in JSON-LD schema and OG tags. e.g. My Tech Blog</p></td>
                        <th><label>Locale:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[site_lang]" value="<?php echo esc_attr((string)($o['site_lang'] ?? '')); ?>" placeholder="en-US">
                        <p class="description">BCP 47 language tag. e.g. en-US, en-GB, fr-FR</p></td>
                    </tr>
                    <tr>
                        <th><label>Title suffix:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[title_suffix]" value="<?php echo esc_attr((string)($o['title_suffix'] ?? '')); ?>" placeholder=" | My Tech Blog">
                        <p class="description">Appended to every page title. e.g. " | My Blog"</p></td>
                        <th><label>Twitter handle:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[twitter_handle]" value="<?php echo esc_attr((string)($o['twitter_handle'] ?? '')); ?>" placeholder="@yourhandle">
                        <p class="description">Your Twitter/X handle including the @ symbol.</p></td>
                    </tr>
                    <tr>
                        <th><label>Home title:</label></th>
                        <td><input class="regular-text" style="width:100%" name="<?php echo esc_attr(self::OPT); ?>[home_title]" value="<?php echo esc_attr((string)($o['home_title'] ?? '')); ?>" placeholder="My Blog – Tech Writer & Developer">
                        <p class="description">Full SEO title for your homepage.</p></td>
                        <th><label>Default OG image URL:</label></th>
                        <td><input class="regular-text" style="width:100%" name="<?php echo esc_attr(self::OPT); ?>[default_og_image]" value="<?php echo esc_attr($o['default_og_image']); ?>" placeholder="https://yoursite.com/wp-content/uploads/og-default.jpg">
                        <p class="description">Fallback image for social sharing. 1200×630px ideal.</p></td>
                    </tr>
                    <tr><th>Home description:</th>
                        <td colspan="3">
                            <textarea class="large-text" rows="3" name="<?php echo esc_attr(self::OPT); ?>[home_desc]" placeholder="A blog about technology, software development, and cloud architecture. Written for engineers and technical leaders."><?php echo esc_textarea($o['home_desc']); ?></textarea>
                            <p class="description">Meta description for your homepage. Aim for 140–155 characters.</p>
                        </td></tr>
                </table>
                </div>
                </div><!-- /ab-card-identity -->

                <div class="ab-zone-card ab-card-person">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">👤</span> Person Schema</span>
                    <?php $this->explain_btn('person', '👤 Person Schema — What each field does', [
                        ['rec'=>'✅ Recommended','name'=>'Full name','desc'=>'Your name as it appears in Google search results and Knowledge Graph. Use your real name exactly as you want it attributed — this is what Google uses to connect your content to you as an individual author.'],
                        ['rec'=>'✅ Recommended','name'=>'Profile URL','desc'=>'The canonical URL for your personal profile — usually your homepage (https://yoursite.com/). Google uses this as the authoritative identifier for you as a person in its Knowledge Graph.'],
                        ['rec'=>'✅ Recommended','name'=>'Job title','desc'=>'Your current job title, e.g. "Chief Information Officer". Included in your Person JSON-LD schema and helps Google understand your professional authority in your subject area.'],
                        ['rec'=>'✅ Recommended','name'=>'Person image URL','desc'=>'URL to your headshot or profile photo. Used in Person schema so Google can associate a face with your content. Ideally a square image of at least 400×400px already uploaded to your media library.'],
                        ['rec'=>'✅ Recommended','name'=>'Social profiles (sameAs)','desc'=>'One URL per line — your LinkedIn, Twitter/X, GitHub, Google Scholar etc. Google uses these to verify your identity and connect your various online presences. The more authoritative profiles you link, the stronger your author entity signal.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body">
                <table class="form-table" role="presentation">
                    <tr>
                        <th><label>Name:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[person_name]" value="<?php echo esc_attr((string)($o['person_name'] ?? '')); ?>" placeholder="Jane Smith">
                        <p class="description">Your full name as it appears in Google.</p></td>
                        <th><label>Job title:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[person_job_title]" value="<?php echo esc_attr((string)($o['person_job_title'] ?? '')); ?>" placeholder="Software Engineer">
                        <p class="description">Your current job title.</p></td>
                    </tr>
                    <tr>
                        <th><label>URL:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[person_url]" value="<?php echo esc_attr((string)($o['person_url'] ?? '')); ?>" placeholder="https://yoursite.com">
                        <p class="description">Canonical URL for your personal profile.</p></td>
                        <th><label>Person image URL:</label></th>
                        <td><input class="regular-text" name="<?php echo esc_attr(self::OPT); ?>[person_image]" value="<?php echo esc_attr($o['person_image']); ?>" placeholder="https://yoursite.com/wp-content/uploads/headshot.jpg">
                        <p class="description">URL of your profile photo for Person JSON-LD schema.</p></td>
                    </tr>
                    <tr><th>SameAs URLs (one per line):</th>
                        <td colspan="3">
                            <textarea class="large-text" rows="4" name="<?php echo esc_attr(self::OPT); ?>[sameas]" placeholder="https://www.linkedin.com/in/yourname&#10;https://twitter.com/yourhandle&#10;https://github.com/yourname"><?php echo esc_textarea($o['sameas']); ?></textarea>
                            <p class="description">Your profiles on other platforms — one URL per line. Helps Google connect your identity across the web.</p>
                        </td></tr>
                </table>
                </div>
                </div><!-- /ab-card-person -->

                <?php submit_button('Save SEO Settings'); ?>
            </form>

            <hr class="ab-zone-divider">

            <?php /* ── AI Meta Writer config form ── */ ?>
            <form method="post" action="options.php" id="ab-ai-config-form">
                <?php settings_fields('cs_seo_ai_group'); ?>

                <div class="ab-zone-card ab-card-ai">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">✦</span> AI Meta Writer — Anthropic Claude</span>
                    <?php $this->explain_btn('ai', '✦ AI Meta Writer — What each setting does', [
                        ['rec'=>'✅ Recommended','name'=>'Anthropic API key','desc'=>'Your secret key from console.anthropic.com. Required to call the Claude AI to generate meta descriptions. Keep this private — anyone with this key can use your Anthropic account. The key is stored securely in your WordPress database.'],
                        ['rec'=>'ℹ️ Info','name'=>'Claude model','desc'=>'Which version of Claude to use for generation. Claude Haiku is fast and cheap — ideal for bulk processing hundreds of posts. Claude Sonnet is slower and costs more but produces higher quality, more nuanced descriptions. For a blog with 100+ posts, Haiku is usually the right choice.'],
                        ['rec'=>'⬜ Optional','name'=>'Overwrite existing','desc'=>'When enabled, the AI will regenerate descriptions for posts that already have one. Leave OFF to only fill in missing descriptions — this protects any manually written descriptions you\'ve already crafted.'],
                        ['rec'=>'⬜ Optional','name'=>'Min / Max characters','desc'=>'Target character range for generated descriptions. Google typically shows 140–160 characters in search results before truncating. Descriptions shorter than 120 characters look thin; longer than 165 get cut off with an ellipsis.'],
                        ['rec'=>'⬜ Optional','name'=>'Custom prompt','desc'=>'Advanced: override the default instructions sent to Claude. The default prompt is tuned for technical blog posts. Only change this if you want a different tone, language, or specific instructions about what to include or exclude in descriptions.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body ab-zone-ai">
                <table class="form-table" role="presentation">
                    <tr>
                        <th>API Key:</th>
                        <td>
                            <div class="ab-key-row">
                                <input type="password" class="regular-text"
                                    name="<?php echo esc_attr(self::AI_OPT); ?>[anthropic_key]"
                                    id="ab-api-key-field"
                                    value="<?php echo esc_attr($ai['anthropic_key']); ?>"
                                    placeholder="sk-ant-api03-...">
                                <button type="button" class="button" onclick="abTestKey()">Test Key</button>
                                <span id="ab-key-status" class="ab-key-status"></span>
                            </div>
                            <p class="description">Get your key at <a href="https://console.anthropic.com/settings/keys" target="_blank">console.anthropic.com</a>. Stored in wp_options — never output to frontend.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Model:</th>
                        <td>
                            <select name="<?php echo esc_attr(self::AI_OPT); ?>[model]">
                                <?php
                                $models = [
                                    'claude-sonnet-4-20250514'  => 'Claude Sonnet 4 (recommended — best quality)',
                                    'claude-haiku-4-5-20251001' => 'Claude Haiku 4.5 (faster, cheaper)',
                                ];
                                foreach ($models as $v => $l): ?>
                                    <option value="<?php echo esc_attr($v); ?>" <?php selected($ai['model'], $v); ?>><?php echo esc_html($l); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th>Target length:</th>
                        <td>
                            <input type="number" style="width:70px" name="<?php echo esc_attr(self::AI_OPT); ?>[min_chars]" value="<?php echo esc_attr($ai['min_chars']); ?>" min="100" max="160"> min &nbsp;
                            <input type="number" style="width:70px" name="<?php echo esc_attr(self::AI_OPT); ?>[max_chars]" value="<?php echo esc_attr($ai['max_chars']); ?>" min="100" max="200"> max characters
                            <p class="description">Google shows 120–160 chars. The range you set here is automatically injected into the prompt — you do not need to mention it in the system prompt above.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>System prompt:</th>
                        <td>
                            <textarea class="large-text" rows="10"
                                id="ab-prompt-field"
                                name="<?php echo esc_attr(self::AI_OPT); ?>[prompt]"><?php echo esc_textarea($ai['prompt']); ?></textarea>
                            <button type="button" class="button" style="margin-top:4px" id="ab-reset-prompt">
                                Reset to default
                            </button>
                            <script>
                            document.getElementById('ab-reset-prompt').addEventListener('click', function() {
                                document.getElementById('ab-prompt-field').value = <?php echo wp_json_encode(self::default_prompt()); ?>;
                            });
                            </script>
                        </td>
                    </tr>
                </table>
                </div><!-- /ab-zone-body -->
                </div><!-- /ab-card-ai -->
                <?php submit_button('Save AI Settings'); ?>
            </form>

            <hr class="ab-zone-divider">

            <div class="ab-zone-card ab-card-update-posts">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">✦</span> Update Posts with AI Descriptions</span>
                    <?php $this->explain_btn('updateposts', '✦ Update Posts — How this works', [
                        ['rec'=>'ℹ️ Info','name'=>'Total Posts','desc'=>'The total number of published posts and pages on your site that are eligible for meta description generation.'],
                        ['rec'=>'ℹ️ Info','name'=>'Have Description','desc'=>'Posts that already have a meta description saved — either written manually or previously generated by the AI.'],
                        ['rec'=>'ℹ️ Info','name'=>'Unprocessed','desc'=>'Posts with no meta description yet. These are the ones Google is currently generating its own snippet for — which is often not the best representation of your content.'],
                        ['rec'=>'ℹ️ Info','name'=>'Generated This Session','desc'=>'How many descriptions have been written by the AI since you opened this page. Resets each time you load the page.'],
                        ['rec'=>'ℹ️ Info','name'=>'Generate Missing','desc'=>'Runs the AI only on posts with no description. Safe to run at any time — it will never overwrite descriptions you\'ve already written unless "Overwrite existing" is checked in AI Meta Writer settings.'],
                        ['rec'=>'⬜ Optional','name'=>'Regenerate All','desc'=>'Forces the AI to rewrite descriptions for every post, including ones that already have descriptions. Use this if you\'ve changed your prompt or want a fresh pass. Note: this will overwrite any manually written descriptions.'],
                        ['rec'=>'⬜ Optional','name'=>'Fix Long/Short','desc'=>'Finds descriptions that fall outside your target character range and rewrites only those. Useful after an initial generation pass — run this to tidy up any descriptions that were cut off or came out too short.'],
                        ['rec'=>'ℹ️ Info','name'=>'Generate (per row)','desc'=>'Rewrites the description for a single post. Click this next to any post to manually trigger the AI for just that one entry — useful for posts where the auto-generated result wasn\'t quite right.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body" style="padding:20px 24px 24px">

                <?php /* ── API key warning banner ── */ ?>
                <div class="ab-api-key-warning" id="ab-api-warn">
                    <div class="ab-warn-icon">⚠️</div>
                    <div class="ab-warn-body">
                        <strong>No Anthropic API key saved — AI generation is disabled.</strong>
                        To use the AI buttons you need to:
                        <ol style="margin:6px 0 0 16px;padding:0">
                            <li>Get a free API key at <a href="https://console.anthropic.com" target="_blank">console.anthropic.com</a></li>
                            <li>Paste it into the <strong>API Key</strong> field in the <strong>✦ AI Meta Writer</strong> section above</li>
                            <li>Click <strong>Save AI Settings</strong></li>
                            <li>Return here and reload the page</li>
                        </ol>
                    </div>
                </div>

                <?php /* ── Load Posts CTA ── */ ?>
                <div class="ab-load-cta" id="ab-load-cta">
                    <div class="ab-load-cta-icon">⬇</div>
                    <div class="ab-load-cta-text">
                        <strong>Load your posts to get started</strong>
                        <span>Fetch all published posts and pages so you can generate or fix their meta descriptions</span>
                    </div>
                    <button class="ab-load-btn" id="ab-load-posts" onclick="abLoadPosts()">Load Posts</button>
                </div>

                <?php /* ── Summary cards ── */ ?>
                <div class="ab-summary-row" id="ab-summary" style="display:none">
                    <div class="ab-summary-card"><div class="ab-summary-num" id="sum-total">0</div><div class="ab-summary-lbl">Total Posts</div></div>
                    <div class="ab-summary-card"><div class="ab-summary-num" id="sum-has" style="color:#1a7a34">0</div><div class="ab-summary-lbl">Have Description</div></div>
                    <div class="ab-summary-card"><div class="ab-summary-num" id="sum-missing" style="color:#6b3fa0">0</div><div class="ab-summary-lbl">Unprocessed</div></div>
                    <div class="ab-summary-card"><div class="ab-summary-num" id="sum-generated" style="color:#2271b1">0</div><div class="ab-summary-lbl">Generated This Session</div></div>
                </div>

                <?php /* ── Toolbar ── */ ?>
                <div class="ab-ai-toolbar" id="ab-ai-toolbar" style="display:none">
                    <button class="button button-primary ab-action-btn" id="ab-ai-gen-missing" onclick="abGenAll(0)" disabled>✦ Generate Missing</button>
                    <button class="button ab-action-btn ab-regen-btn" id="ab-ai-gen-all" onclick="abGenAll(1)" disabled>↺ Regenerate All</button>
                    <button class="button ab-action-btn ab-fix-btn" id="ab-ai-fix" onclick="abFixAll()" disabled>⚑ Fix Long/Short</button>
                    <button class="button" id="ab-load-posts-again" onclick="abLoadPosts()" style="margin-left:auto">↻ Reload</button>
                    <button class="button" id="ab-ai-stop" onclick="abStop()" style="display:none">◻ Stop</button>
                    <span id="ab-toolbar-status" style="font-size:12px;color:#50575e;"></span>
                </div>

                <?php /* ── Progress bar ── */ ?>
                <div class="ab-progress" id="ab-progress">
                    <div class="ab-progress-fill" id="ab-progress-fill"></div>
                </div>
                <div class="ab-stats" id="ab-prog-label"></div>

                <?php /* ── Log ── */ ?>
                <div id="ab-log"></div>

                <?php /* ── Post table ── */ ?>
                <div id="ab-posts-wrap"></div>
                <div class="ab-pager" id="ab-pager" style="display:none">
                    <button class="button" id="ab-prev" onclick="abPage(-1)">← Prev</button>
                    <span id="ab-page-info" style="font-size:12px;color:#50575e;"></span>
                    <button class="button" id="ab-next" onclick="abPage(1)">Next →</button>
                </div>

                </div><!-- /ab-zone-body -->
            </div><!-- /ab-card-update-posts -->

        </div><!-- /ab-pane-seo -->

        <?php /* ══════════════════ SITEMAP PANE ══════════════════ */ ?>
        <div class="ab-pane" id="ab-pane-sitemap">
            <form method="post" action="options.php">
                <?php settings_fields('cs_seo_group'); ?>

                <?php
                $pub_types = get_post_types(['public' => true], 'objects');
                $sel_types = (array)($o['sitemap_post_types'] ?? ['post', 'page']);
                ?>

                <div class="ab-zone-card ab-card-features">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">⚙</span> Features &amp; Robots</span>
                    <?php $this->explain_btn('features', '⚙ Features & Robots — What each option does', [
                        ['rec'=>'✅ Recommended','name'=>'OpenGraph + Twitter Cards','desc'=>'Adds structured metadata so your posts display with a title, description and image when shared on LinkedIn, Twitter/X, WhatsApp or any other platform. Without this, shared links look blank or use random images.'],
                        ['rec'=>'✅ Recommended','name'=>'WebSite JSON-LD (front page)','desc'=>'Tells Google the name and URL of your site in structured data format. Helps Google display your site name correctly in search results and can unlock sitelinks beneath your homepage listing.'],
                        ['rec'=>'✅ Recommended','name'=>'Person JSON-LD schema','desc'=>'Embeds your name, job title, photo, and social profiles into your site so Google can connect your content to you as an individual. Important for personal brand and author authority signals.'],
                        ['rec'=>'✅ Recommended','name'=>'BlogPosting JSON-LD schema','desc'=>'Marks up each post as an article with author, publish date, and headline. Google uses this for rich results and to better understand your content type. Can improve click-through rates in search.'],
                        ['rec'=>'⬜ Optional','name'=>'Breadcrumb JSON-LD schema','desc'=>'Adds breadcrumb trail markup to posts. Most useful on large sites with deep category hierarchies. For a flat personal blog this adds little value — Google will figure out your structure without it.'],
                        ['rec'=>'⬜ Optional','name'=>'Strip UTM params in canonical URLs','desc'=>'If you use UTM tracking parameters on your own internal links (e.g. ?utm_source=newsletter), this stops them creating duplicate pages in Google\'s index. Only needed if you track internal clicks with UTM.'],
                        ['rec'=>'✅ Recommended','name'=>'Enable /sitemap.xml','desc'=>'Generates a sitemap listing all your posts and pages. Submit this URL to Google Search Console so Google knows exactly what to crawl. Also automatically added to your robots.txt.'],
                        ['rec'=>'✅ Recommended','name'=>'noindex search results','desc'=>'Prevents Google from indexing your WordPress search result pages (e.g. /?s=keyword). These pages have no unique value and waste Google\'s crawl budget — always block them.'],
                        ['rec'=>'✅ Recommended','name'=>'noindex 404 pages','desc'=>'Stops Google indexing error pages. A 404 page has no content worth ranking — keeping these out of the index keeps your crawl budget focused on real content.'],
                        ['rec'=>'✅ Recommended','name'=>'noindex attachment pages','desc'=>'WordPress creates a separate page for every uploaded image or file. These pages are near-empty and often outrank your actual posts for image searches. Always block them.'],
                        ['rec'=>'✅ Recommended','name'=>'noindex author archives','desc'=>'On a single-author blog, your author archive page (/author/yourname/) is essentially a duplicate of your homepage. Blocking it prevents a duplicate content penalty.'],
                        ['rec'=>'✅ Recommended','name'=>'noindex tag archives','desc'=>'Tag archive pages (/tag/aws/) often duplicate post content and can dilute your rankings. Unless your tag pages have unique introductory text and real editorial value, block them.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body" style="padding:16px 20px">
                <div class="ab-checkbox-grid">
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_og]" value="1" <?php checked((int)($o['enable_og'] ?? 0), 1); ?>> OpenGraph + Twitter Cards</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_schema_website]" value="1" <?php checked((int)($o['enable_schema_website'] ?? 0), 1); ?>> WebSite JSON-LD (front page)</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_schema_person]" value="1" <?php checked((int)($o['enable_schema_person'] ?? 0), 1); ?>> Person JSON-LD schema</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_schema_article]" value="1" <?php checked((int)($o['enable_schema_article'] ?? 0), 1); ?>> BlogPosting JSON-LD schema</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_schema_breadcrumbs]" value="1" <?php checked((int)($o['enable_schema_breadcrumbs'] ?? 0), 1); ?>> Breadcrumb JSON-LD schema</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[strip_tracking_params]" value="1" <?php checked((int)($o['strip_tracking_params'] ?? 0), 1); ?>> Strip UTM params in canonical URLs</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_sitemap]" value="1" <?php checked((int)($o['enable_sitemap'] ?? 0), 1); ?>> Enable /sitemap.xml</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[noindex_search]" value="1" <?php checked((int)($o['noindex_search'] ?? 0), 1); ?>> noindex search results</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[noindex_404]" value="1" <?php checked((int)($o['noindex_404'] ?? 0), 1); ?>> noindex 404 pages</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[noindex_attachment]" value="1" <?php checked((int)($o['noindex_attachment'] ?? 0), 1); ?>> noindex attachment pages</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[noindex_author_archives]" value="1" <?php checked((int)($o['noindex_author_archives'] ?? 0), 1); ?>> noindex author archives</label>
                    <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[noindex_tag_archives]" value="1" <?php checked((int)($o['noindex_tag_archives'] ?? 0), 1); ?>> noindex tag archives</label>
                </div>
                </div>
                </div><!-- /ab-card-features -->

                <div class="ab-zone-card ab-card-sitemap-settings">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">⚙</span> Sitemap Settings</span>
                    <?php $this->explain_btn('sitemap', '⚙ Sitemap Settings — What each option does', [
                        ['rec'=>'✅ Recommended','name'=>'Enable /sitemap.xml','desc'=>'Generates a sitemap at yoursite.com/sitemap.xml listing all your published content. Submit this URL to Google Search Console so Google knows exactly what pages to crawl. Also automatically appends the sitemap URL to your robots.txt.'],
                        ['rec'=>'✅ Recommended','name'=>'Include Posts','desc'=>'Adds all your published blog posts to the sitemap. This should always be on — posts are your primary content and the main thing you want Google to discover and index.'],
                        ['rec'=>'✅ Recommended','name'=>'Include Pages','desc'=>'Adds your WordPress pages (About, Contact etc.) to the sitemap. Keep this on — pages like your About and Contact pages should be indexed.'],
                        ['rec'=>'⬜ Optional','name'=>'Taxonomy archives','desc'=>'Includes category, tag, and custom taxonomy archive pages in the sitemap. Turn this on only if your archive pages have unique introductory content and genuine value for visitors. For most blogs, leave it off — archive pages often duplicate post content.'],
                        ['rec'=>'⬜ Optional','name'=>'Exclude URLs or IDs','desc'=>'Enter specific URLs or post IDs to omit from the sitemap — one per line. Use this for thank-you pages, landing pages, privacy policy pages, or any content you don\'t want Google to prioritise. Numeric IDs (e.g. 42) refer to the WordPress post/page ID shown in the edit URL.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body">
                <table class="form-table" role="presentation">
                    <tr>
                        <th>Enable sitemap:</th>
                        <td>
                            <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[enable_sitemap]" value="1" <?php checked((int)($o['enable_sitemap'] ?? 0), 1); ?>>
                            Generate sitemap at <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank"><?php echo esc_html(home_url('/sitemap.xml')); ?></a></label>
                            <p class="description">When enabled the Sitemap URL is also appended to your robots.txt automatically.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Include post types:</th>
                        <td>
                            <div style="display:flex;gap:16px;flex-wrap:wrap">
                            <?php foreach ($pub_types as $pt): ?>
                                <?php if (in_array($pt->name, ['attachment'], true)) continue; ?>
                                <label>
                                    <input type="checkbox"
                                        name="<?php echo esc_attr(self::OPT); ?>[sitemap_post_types][]"
                                        value="<?php echo esc_attr($pt->name); ?>"
                                        <?php checked(in_array($pt->name, $sel_types, true), true); ?>>
                                    <?php echo esc_html($pt->labels->name); ?>
                                    <span style="color:#888;font-size:11px">(<?php echo esc_html($pt->name); ?>)</span>
                                </label>
                            <?php endforeach; ?>
                            </div>
                            <p class="description">Select which post types to include. Uncheck types that are not meaningful for search engines (e.g. WooCommerce order pages).</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Taxonomy archives:</th>
                        <td>
                            <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[sitemap_taxonomies]" value="1" <?php checked((int)($o['sitemap_taxonomies'] ?? 0), 1); ?>>
                            Include category, tag, and custom taxonomy archive pages</label>
                            <p class="description">Off by default. Enable if your category or tag archive pages have unique, valuable content worth indexing.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Exclude URLs or IDs:</th>
                        <td>
                            <textarea name="<?php echo esc_attr(self::OPT); ?>[sitemap_exclude]"
                                rows="6" style="width:100%;font-family:'Courier New',monospace;font-size:12px;color:#1d2327"
                                placeholder="https://yoursite.com/thank-you/<?php echo "\n"; ?>https://yoursite.com/privacy-policy/<?php echo "\n"; ?>42<?php echo "\n"; ?>156"><?php echo esc_textarea((string)($o['sitemap_exclude'] ?? '')); ?></textarea>
                            <p class="description">One entry per line. Enter full URLs or numeric post/page IDs. These will be omitted from the sitemap.</p>
                        </td>
                    </tr>
                </table>
                </div>
                </div><!-- /ab-card-sitemap-settings -->

                <?php submit_button('Save Sitemap Settings'); ?>
            </form>

            <hr class="ab-zone-divider">

            <form method="post" action="options.php">
                <?php settings_fields('cs_seo_group'); ?>

                <div class="ab-zone-card ab-card-robots">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">🤖</span> Robots.txt</span>
                    <?php $this->explain_btn('robots', '🤖 Robots.txt — What this all means', [
                        ['rec'=>'ℹ️ Info','name'=>'What is robots.txt?','desc'=>'A plain text file at yoursite.com/robots.txt that tells search engine crawlers which pages they are and aren\'t allowed to visit. It doesn\'t prevent indexing — it prevents crawling. Google respects it; malicious bots ignore it entirely.'],
                        ['rec'=>'ℹ️ Info','name'=>'Physical file warning','desc'=>'If a robots.txt file exists on disk, the web server serves it directly — bypassing WordPress and this plugin completely. You must rename or delete it to let the plugin take control. The plugin offers a one-click rename to robots.txt.bak.'],
                        ['rec'=>'⬜ Optional','name'=>'Block AI training bots','desc'=>'Adds Disallow: / rules for GPTBot, CCBot, Claude-Web, anthropic-ai and other AI training crawlers. Turn this ON if you don\'t want AI companies training their models on your content. Leave OFF if you want AI assistants to surface your content when users ask relevant questions.'],
                        ['rec'=>'✅ Recommended','name'=>'Custom robots.txt rules','desc'=>'The full content of your robots.txt file. The plugin automatically appends your sitemap URL and the AI bot blocklist (if enabled) — do not add those here manually. Changes take effect immediately on every request — there is no caching.'],
                        ['rec'=>'ℹ️ Info','name'=>'User-agent: Googlebot','desc'=>'Rules that apply specifically to Google\'s crawler. Googlebot respects these rules more strictly than other crawlers. Disallowing /wp-admin/, /wp-login.php and search pages stops Google wasting crawl budget on admin and junk pages.'],
                        ['rec'=>'ℹ️ Info','name'=>'User-agent: *','desc'=>'Rules that apply to all other crawlers not specifically named above. This is the catch-all for Bing, DuckDuckGo, and any other well-behaved search engine crawler.'],
                        ['rec'=>'ℹ️ Info','name'=>'Live preview','desc'=>'Shows exactly what search engines see when they fetch yoursite.com/robots.txt right now. If the sitemap URL appears at the bottom, everything is working correctly.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body">

                <?php
                $physical_exists   = file_exists(ABSPATH . 'robots.txt');
                $physical_writable = $physical_exists && wp_is_writable(ABSPATH . 'robots.txt');
                $physical_contents = $physical_exists ? file_get_contents(ABSPATH . 'robots.txt') : '';

                // Also check one level up in case WordPress is in a subdirectory
                $alt_path          = dirname(rtrim(ABSPATH, '/')) . '/robots.txt';
                $alt_exists        = file_exists($alt_path);
                ?>
                <div style="background:#f0f6fc;border:1px solid #c2d9f0;border-radius:4px;padding:10px 14px;margin:12px 20px;font-size:12px;font-family:monospace">
                    <strong>File detection:</strong><br>
                    ABSPATH: <code><?php echo esc_html(ABSPATH); ?></code><br>
                    Looking for: <code><?php echo esc_html(ABSPATH . 'robots.txt'); ?></code> → <?php echo $physical_exists ? '<span style="color:#1a7a34">found</span>' : '<span style="color:#c3372b">not found</span>'; ?><br>
                    <?php if (!$physical_exists): ?>
                    Also checking: <code><?php echo esc_html($alt_path); ?></code> → <?php echo $alt_exists ? '<span style="color:#e67e00">found here!</span>' : '<span style="color:#888">not found</span>'; ?>
                    <?php endif; ?>
                </div>

                <?php
                // If file not at ABSPATH, try one level up (WordPress in subdirectory)
                $robots_path = ABSPATH . 'robots.txt';
                if (!$physical_exists && $alt_exists) {
                    $robots_path     = $alt_path;
                    $physical_exists = true;
                    $physical_writable = wp_is_writable($alt_path);
                    $physical_contents = file_get_contents($alt_path);
                }
                ?>
                <?php if ($physical_exists): ?>
                <div class="ab-physical-robots-warn" id="ab-physical-robots-warn">
                    <div style="font-size:22px;flex-shrink:0">⚠️</div>
                    <div style="flex:1">
                        <strong>A physical robots.txt file exists on your server</strong><br>
                        WordPress (and this plugin) cannot control your robots.txt while a real file exists on disk — the web server serves the file directly, bypassing WordPress entirely. To let the plugin manage your robots.txt, the file needs to be renamed.<br><br>
                        <strong>Current file location:</strong> <code><?php echo esc_html($robots_path); ?></code>
                        &nbsp;·&nbsp; <strong>Writable:</strong> <?php echo $physical_writable ? '<span style="color:#1a7a34">Yes</span>' : '<span style="color:#c3372b">No</span>'; ?><br><br>
                        <?php if ($physical_contents): ?>
                        <strong>Current file contents:</strong><br>
                        <pre style="background:#f6f7f7;border:1px solid #c3c4c7;border-radius:4px;padding:10px;font-size:12px;line-height:1.6;max-height:200px;overflow-y:auto;margin:6px 0 12px"><?php echo esc_html($physical_contents); ?></pre>
                        <?php endif; ?>
                        <strong>What happens when you click Rename:</strong> The file is renamed to <code>robots.txt.bak</code> in the same directory. WordPress then takes over and this plugin generates robots.txt dynamically on every request.<br><br>
                        <?php if ($physical_writable): ?>
                        <button type="button" class="button button-primary" id="ab-rename-robots-btn" onclick="abRenameRobots()">
                            ✎ Rename robots.txt → robots.txt.bak
                        </button>
                        <span id="ab-rename-robots-status" style="margin-left:10px;font-size:13px"></span>
                        <?php else: ?>
                        <div style="background:#fef0f0;border:1px solid #f5bcbb;border-radius:4px;padding:10px;margin-top:4px">
                            <strong style="color:#c3372b">File is not writable</strong> — the web server does not have permission to rename this file.<br>
                            Fix via FTP or your host's file manager: right-click <code>robots.txt</code> → set permissions to <strong>644</strong>, then reload this page.<br><br>
                            Alternatively, rename the file manually via FTP: rename <code>robots.txt</code> to <code>robots.txt.bak</code> in your WordPress root.
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php else: ?>
                <div style="display:flex;gap:12px;align-items:flex-start;background:#edfaef;border:1px solid #1a7a34;border-radius:6px;padding:14px 18px;margin:12px 20px">
                    <div style="font-size:22px;flex-shrink:0">✅</div>
                    <div style="font-size:13px">
                        <strong>No physical robots.txt file detected</strong> — this plugin is managing your robots.txt dynamically. Search engines will see the content shown in the Live robots.txt preview below.<br><br>
                        <span style="color:#50575e">If you recently deleted or renamed the file manually, this is correct. The Live preview below shows exactly what Google will see.</span>
                    </div>
                </div>
                <?php endif; ?>

                <?php /* Live robots.txt preview */ ?>
                <div style="padding:16px 20px 4px">
                    <strong style="font-size:13px">Live robots.txt</strong>
                    &nbsp;<a href="<?php echo esc_url(home_url('/robots.txt')); ?>" target="_blank" style="font-size:12px">↗ view in browser</a>
                    <button type="button" class="button" style="float:right;font-size:11px;padding:2px 10px" onclick="abRefreshRobotsPreview()">↻ Refresh</button>
                    <pre id="ab-robots-live-preview" style="background:#1a1a2e;color:#e0e0f0;font-family:'Courier New',monospace;font-size:12px;line-height:1.6;padding:14px;border-radius:6px;max-height:320px;overflow-y:auto;margin:8px 0 0;white-space:pre-wrap;word-break:break-word">Loading…</pre>
                </div>

                <table class="form-table" role="presentation">
                    <tr>
                        <th>Block AI training bots:</th>
                        <td>
                            <label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[block_ai_bots]" value="1" <?php checked((int)($o['block_ai_bots'] ?? 1), 1); ?>>
                            Block GPTBot, ChatGPT-User, CCBot, anthropic-ai, Claude-Web, FacebookBot, Bytespider, Applebot-Extended</label>
                            <p class="description">Adds <code>Disallow: /</code> for each AI training crawler. Appended automatically after your custom rules below.</p>
                        </td>
                    </tr>
                    <tr>
                        <th><label for="cs-robots-txt">Custom robots.txt rules</label></th>
                        <td>
                            <textarea id="cs-robots-txt" name="<?php echo esc_attr(self::OPT); ?>[robots_txt]"
                                rows="16" style="width:100%;font-family:'Courier New',monospace;font-size:12px;line-height:1.6;color:#1d2327"><?php echo esc_textarea((string)($o['robots_txt'] ?? self::default_robots_txt())); ?></textarea>
                            <p class="description">Full robots.txt content. The AI bot blocklist (if enabled) and your sitemap URL are appended automatically — do not add them here. Changes take effect immediately at <a href="<?php echo esc_url(home_url('/robots.txt')); ?>" target="_blank"><?php echo esc_html(home_url('/robots.txt')); ?></a></p>
                            <div style="display:flex;justify-content:flex-end;margin-top:8px">
                                <button type="button" class="button" onclick="document.getElementById('cs-robots-txt').value=<?php echo wp_json_encode(self::default_robots_txt()); ?>">Reset to default</button>
                            </div>
                        </td>
                    </tr>
                </table>
                </div>
                </div><!-- /ab-card-robots -->

                <?php submit_button('Save Robots Settings'); ?>
            </form>

            <hr class="ab-zone-divider">

            <div class="ab-zone-card ab-card-sitemap-preview">
            <div class="ab-zone-header" style="justify-content:space-between;align-items:center">
                <span><span class="ab-zone-icon">🔍</span> Sitemap Preview</span>
                <div style="display:flex;gap:8px;align-items:center">
                    <?php $this->explain_btn('sitemappreview', '🔍 Sitemap Preview — How to use this', [
                        ['rec'=>'ℹ️ Info','name'=>'What this shows','desc'=>'A table of every URL that will appear in your sitemap.xml when Google crawls it. This is the live data — if a post appears here, it is in your sitemap. If it doesn\'t appear, Google won\'t find it via the sitemap.'],
                        ['rec'=>'ℹ️ Info','name'=>'Type badges','desc'=>'Each row shows the content type: Post (blog post), Page (WordPress page), Home (your homepage), Taxonomy (category/tag archive). Use this to verify the right content types are being included based on your Sitemap Settings.'],
                        ['rec'=>'ℹ️ Info','name'=>'Last Modified','desc'=>'The date the post was last updated. Google uses this to decide how often to re-crawl a page. Recently updated posts get re-crawled sooner. If a post shows an old date, consider updating it to signal freshness.'],
                        ['rec'=>'ℹ️ Info','name'=>'Pagination','desc'=>'Results are shown 200 at a time. Use Prev/Next to browse all your URLs. The count at the bottom right shows which URLs you\'re viewing out of the total.'],
                        ['rec'=>'ℹ️ Info','name'=>'View live sitemap','desc'=>'The link opens your actual sitemap.xml in a new tab — this is what Google sees. The index file lists all your sub-sitemaps (one per post type). Click through to see the raw XML.'],
                    ]); ?>
                    <button id="ab-sitemap-load" onclick="abLoadSitemap();return false;"
                        style="background:#f0b429;border:none;border-radius:6px;color:#1d2327;font-size:13px;font-weight:700;padding:7px 18px;cursor:pointer;letter-spacing:0.02em;box-shadow:0 2px 6px rgba(0,0,0,0.25);transition:background 0.15s">
                        ⬇ Load Preview
                    </button>
                </div>
            </div>
            <div class="ab-zone-body" style="padding:16px 20px">
                <p style="color:#50575e;margin:0 0 14px;font-size:13px">Shows all URLs that will appear in your sitemap. Paginated at 200 rows — use Prev/Next to browse. Save settings before previewing.</p>
                <div id="ab-sitemap-preview-wrap">
                    <p style="color:#a7aaad;font-size:13px">Click <strong>Load Preview</strong> to fetch the current sitemap contents.</p>
                </div>
            </div>
            </div><!-- /ab-card-sitemap-preview -->
            <script>
            (function() {
                var _ajax  = <?php echo wp_json_encode(admin_url('admin-ajax.php')); ?>;
                var _nonce = <?php echo wp_json_encode(wp_create_nonce('cs_seo_nonce')); ?>;

                function loadSitemapPreview(pg) {
                    var wrap = document.getElementById('ab-sitemap-preview-wrap');
                    var btn  = document.getElementById('ab-sitemap-load');
                    if (!wrap || !btn) return;
                    btn.disabled = true;
                    btn.textContent = '⟳ Loading...';
                    btn.style.background = '#c0882a';
                    wrap.innerHTML = '<p style="color:#666;font-size:13px">Fetching sitemap entries…</p>';

                    var body = 'action=cs_seo_sitemap_preview&nonce='+encodeURIComponent(_nonce)+'&sitemap_pg='+(pg||1);
                    fetch(_ajax, {method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body:body})
                        .then(function(r){ return r.text(); })
                        .then(function(txt) {
                            btn.disabled = false;
                            btn.textContent = '↻ Reload';
                            btn.style.background = '#f0b429';
                            var data;
                            try { data = JSON.parse(txt); } catch(e) {
                                wrap.innerHTML = '<div style="color:#c3372b;background:#fef0f0;border:1px solid #f5bcbb;padding:12px;border-radius:4px">Response was not JSON. Check for PHP errors.</div>';
                                return;
                            }
                            if (!data.success) {
                                wrap.innerHTML = '<div style="color:#c3372b;background:#fef0f0;border:1px solid #f5bcbb;padding:12px;border-radius:4px">Error: '+(data.data||'unknown')+'</div>';
                                return;
                            }
                            var d=data.data, entries=d.entries, total=d.total, page=d.page, pages=d.pages, per=d.per_page;
                            var cols={home:'#6b3fa0',post:'#1a4a7a',page:'#1a7a34',tax:'#7a5c00',cpt:'#8a3a00'};
                            var labels={home:'Home',post:'Post',page:'Page',tax:'Taxonomy',cpt:'CPT'};
                            var rows=entries.map(function(e){
                                return '<tr style="border-bottom:1px solid #f0f0f0">'+
                                    '<td style="padding:6px 8px"><a href="'+e.loc+'" target="_blank" style="font-size:12px;color:#2271b1">'+e.loc+'</a>'+(e.title?'<br><small style="color:#888;font-size:11px">'+e.title+'</small>':'')+'</td>'+
                                    '<td style="padding:6px 8px"><span style="background:'+(cols[e.type]||'#444')+';color:#fff;border-radius:3px;padding:2px 8px;font-size:11px;white-space:nowrap">'+(labels[e.type]||e.type)+'</span></td>'+
                                    '<td style="padding:6px 8px;color:#888;font-size:12px;white-space:nowrap">'+(e.lastmod||'—')+'</td></tr>';
                            }).join('');
                            var pager='';
                            if(pages>1){
                                pager='<div style="display:flex;gap:10px;align-items:center;margin-top:14px;flex-wrap:wrap">'+
                                    '<button class="button" '+(page<=1?'disabled':'onclick="window._abSitemapLoad('+(page-1)+')"')+'>← Prev</button>'+
                                    '<span style="font-size:13px;color:#50575e">Page <strong>'+page+'</strong> of <strong>'+pages+'</strong></span>'+
                                    '<button class="button button-primary" '+(page>=pages?'disabled':'onclick="window._abSitemapLoad('+(page+1)+')"')+'>Next →</button>'+
                                    '<span style="font-size:12px;margin-left:auto;color:#888">'+((page-1)*per+1)+'–'+Math.min(page*per,total)+' of '+total+' URLs</span>'+
                                    '</div>';
                            }
                            wrap.innerHTML=
                                '<p style="font-size:13px;margin:0 0 12px;color:#1d2327"><strong>'+total+'</strong> total URLs across <strong>'+pages+'</strong> sitemap file'+(pages>1?'s':'')+
                                ' &nbsp;·&nbsp; <a href="<?php echo esc_url(home_url('/sitemap.xml')); ?>" target="_blank" style="color:#2271b1">View live sitemap ↗</a></p>'+
                                '<table style="width:100%;border-collapse:collapse;font-size:13px;background:#fff;border:1px solid #e0e0e0;border-radius:4px;overflow:hidden">'+
                                '<thead><tr style="background:#f6f7f7;border-bottom:2px solid #e0e0e0">'+
                                '<th style="text-align:left;padding:8px 8px;font-size:12px;color:#50575e;font-weight:600">URL</th>'+
                                '<th style="text-align:left;padding:8px 8px;font-size:12px;color:#50575e;font-weight:600">Type</th>'+
                                '<th style="text-align:left;padding:8px 8px;font-size:12px;color:#50575e;font-weight:600">Last Modified</th></tr></thead>'+
                                '<tbody>'+rows+'</tbody></table>'+pager;
                        })
                        .catch(function(e){
                            btn.disabled=false; btn.textContent='↻ Reload'; btn.style.background='#f0b429';
                            wrap.innerHTML='<div style="color:#c3372b;background:#fef0f0;border:1px solid #f5bcbb;padding:12px;border-radius:4px">Network error: '+e.message+'</div>';
                        });
                }
                window.abLoadSitemap  = loadSitemapPreview;
                window._abSitemapLoad = loadSitemapPreview;
                document.addEventListener('DOMContentLoaded', function() {
                    var b = document.getElementById('ab-sitemap-load');
                    if (b) b.addEventListener('click', function(e){ e.preventDefault(); loadSitemapPreview(1); });
                });
            })();
            </script>

        </div><!-- /ab-pane-sitemap -->

        <?php /* ══════════════════ SCHEDULED BATCH PANE ══════════════════ */ ?>
        <div class="ab-pane" id="ab-pane-batch">
            <form method="post" action="options.php">
                <?php settings_fields('cs_seo_ai_group'); ?>

                <div class="ab-zone-card ab-card-schedule">
                <div class="ab-zone-header" style="justify-content:space-between">
                    <span><span class="ab-zone-icon">⏱</span> Scheduled Batch Generation</span>
                    <?php $this->explain_btn('schedule', '⏱ Scheduled Batch — How this works', [
                        ['rec'=>'ℹ️ Info','name'=>'What this does','desc'=>'Automatically runs the AI meta description generator on a schedule — no need to manually click Generate Missing. The batch only processes posts that don\'t yet have a description, so it never overwrites existing ones.'],
                        ['rec'=>'⬜ Optional','name'=>'Enable schedule','desc'=>'Turns the scheduled batch on or off. When enabled, the batch runs automatically at midnight (server time) on the days you select. When disabled, no automatic generation happens — you can still run it manually from the Optimise SEO tab.'],
                        ['rec'=>'⬜ Optional','name'=>'Days of the week','desc'=>'Choose which days the batch runs. For a high-volume blog that publishes daily, tick every day. For a weekly blog, once or twice a week is sufficient. The batch only does work if there are unprocessed posts — if everything is up to date, it completes instantly.'],
                        ['rec'=>'ℹ️ Info','name'=>'Midnight server time','desc'=>'The batch runs at midnight based on your server\'s timezone, not your local time. Check your WordPress timezone setting under Settings → General if the timing seems off.'],
                        ['rec'=>'ℹ️ Info','name'=>'API costs','desc'=>'Each description generated makes one API call to Anthropic Claude. At typical blog post lengths, Claude Haiku costs roughly $0.001–$0.003 per post. A full run across 100 unprocessed posts costs around $0.10–$0.30.'],
                    ]); ?>
                </div>
                <div class="ab-zone-body">
                <p style="padding:12px 20px 0;color:#50575e;margin:0">The batch runs automatically on selected days at midnight (server time). It only processes posts that do not yet have a meta description — it never overwrites existing ones.</p>
                <table class="form-table" role="presentation">
                    <tr>
                        <th>Enable schedule:</th>
                        <td>
                            <label>
                                <input type="checkbox"
                                    id="cs-sched-enabled"
                                    name="<?php echo esc_attr(self::AI_OPT); ?>[schedule_enabled]"
                                    value="1" <?php checked((int)($ai['schedule_enabled'] ?? 0), 1); ?>
                                    onchange="csToggleSchedDays(this.checked)">
                                Enable automatic scheduled batch
                            </label>
                            <p class="description">Requires an Anthropic API key saved in the Optimise SEO tab → AI Meta Writer section.</p>
                        </td>
                    </tr>
                    <tr>
                        <th>Run on these days:</th>
                        <td>
                            <div style="display:flex;gap:16px;flex-wrap:wrap" id="cs-sched-days">
                            <?php
                            $day_labels  = ['mon'=>'Monday','tue'=>'Tuesday','wed'=>'Wednesday','thu'=>'Thursday','fri'=>'Friday','sat'=>'Saturday','sun'=>'Sunday'];
                            $sched_days  = (array)($ai['schedule_days'] ?? []);
                            $sched_on    = (int)($ai['schedule_enabled'] ?? 0);
                            foreach ($day_labels as $val => $label): ?>
                                <label style="<?php echo $sched_on ? '' : 'opacity:0.4'; ?>">
                                    <input type="checkbox"
                                        class="cs-sched-day"
                                        name="<?php echo esc_attr(self::AI_OPT); ?>[schedule_days][]"
                                        value="<?php echo esc_attr($val); ?>"
                                        <?php checked(in_array($val, $sched_days, true), true); ?>
                                        <?php echo $sched_on ? '' : 'disabled'; ?>>
                                    <?php echo esc_html($label); ?>
                                </label>
                            <?php endforeach; ?>
                            </div>
                            <script>
                            function csToggleSchedDays(enabled) {
                                document.querySelectorAll('.cs-sched-day').forEach(function(cb) {
                                    cb.disabled = !enabled;
                                    cb.closest('label').style.opacity = enabled ? '1' : '0.4';
                                });
                            }
                            </script>
                            <p class="description" style="margin-top:10px">
                                <?php
                                $cron_next = wp_next_scheduled('cs_seo_daily_batch');
                                if ($cron_next && !empty($sched_days)) {
                                    $day_map = ['mon'=>1,'tue'=>2,'wed'=>3,'thu'=>4,'fri'=>5,'sat'=>6,'sun'=>0];
                                    $target_dow = array_map(fn($d) => $day_map[$d] ?? -1, $sched_days);
                                    $found = null;
                                    for ($i = 0; $i <= 7; $i++) {
                                        $ts  = strtotime("midnight +{$i} days");
                                        $dow = (int) gmdate('w', $ts);
                                        if (in_array($dow, $target_dow, true)) {
                                            $found = $ts;
                                            break;
                                        }
                                    }
                                    if ($found) {
                                        echo 'Next scheduled run: <strong>' . esc_html(gmdate('D d M Y H:i:s', $found)) . '</strong> (server time)';
                                    } else {
                                        echo 'No matching days selected.';
                                    }
                                } elseif ($cron_next && (int)($ai['schedule_enabled'] ?? 0)) {
                                    echo '<span style="color:#c3372b">No days selected — tick at least one day above.</span>';
                                } elseif ((int)($ai['schedule_enabled'] ?? 0)) {
                                    echo '<span style="color:#c3372b">No cron event found — try saving settings again.</span>';
                                } else {
                                    echo 'Schedule is disabled.';
                                } ?>
                            </p>
                        </td>
                    </tr>
                </table>
                </div><!-- /ab-zone-body -->
                </div><!-- /ab-card-schedule -->
                <?php submit_button('Save Schedule Settings'); ?>
            </form>

            <hr class="ab-zone-divider">

            <div class="ab-zone-card ab-card-lastrun">
            <div class="ab-zone-header" style="justify-content:space-between">
                <span><span class="ab-zone-icon">📋</span> Last Batch Run</span>
                <?php $this->explain_btn('lastrun', '📋 Last Batch Run — Reading the results', [
                    ['rec'=>'ℹ️ Info','name'=>'Last run time','desc'=>'When the most recent scheduled batch executed. If this shows "Never" and you have the schedule enabled, the WordPress cron may not have triggered yet — it runs at midnight server time. You can also trigger a run manually from the Optimise SEO tab.'],
                    ['rec'=>'ℹ️ Info','name'=>'Processed','desc'=>'How many posts the batch attempted to generate descriptions for in the last run. Posts that already had descriptions are skipped and not counted here.'],
                    ['rec'=>'ℹ️ Info','name'=>'Succeeded','desc'=>'Posts that were successfully updated with a new AI-generated description. These posts now have meta descriptions and will be skipped in future batch runs.'],
                    ['rec'=>'ℹ️ Info','name'=>'Errors','desc'=>'Posts where generation failed — usually due to an API error, rate limit, or the post having no readable content. The batch will retry these on the next scheduled run. Check your Anthropic API key if errors are consistently high.'],
                    ['rec'=>'ℹ️ Info','name'=>'Next scheduled run','desc'=>'When the batch will next execute automatically. If this shows "Not scheduled" but the schedule is enabled, try saving your schedule settings again — this re-registers the WordPress cron event.'],
                ]); ?>
            </div>
            <div class="ab-zone-body">
            <?php $last_batch = get_transient('cs_seo_last_batch'); ?>
            <?php if ($last_batch): ?>
                <div style="padding:16px 20px">
                    <p style="margin:0 0 8px">
                        <strong><?php echo esc_html($last_batch['day']); ?> <?php echo esc_html($last_batch['date']); ?></strong> —
                        <span style="color:#1a7a34"><?php echo (int)$last_batch['done']; ?> generated</span>,
                        <?php echo (int)$last_batch['skipped']; ?> skipped,
                        <?php if ($last_batch['errors'] > 0): ?>
                            <span style="color:#c3372b"><?php echo (int)$last_batch['errors']; ?> errors</span>,
                        <?php endif; ?>
                        <?php echo esc_html($last_batch['elapsed']); ?> minutes total
                    </p>
                    <?php if (!empty($last_batch['log'])): ?>
                    <details style="margin-top:8px">
                        <summary style="cursor:pointer;font-size:12px;color:#50575e">Show post log (<?php echo count($last_batch['log']); ?> entries)</summary>
                        <div style="background:#1a1a2e;color:#e0e0f0;font-family:'Courier New',monospace;font-size:11px;padding:10px;border-radius:4px;margin-top:8px;max-height:200px;overflow-y:auto">
                        <?php foreach ($last_batch['log'] as $entry): ?>
                            <?php if ($entry['status'] === 'ok'): ?>
                                <div style="color:#00d084">✓ <?php echo esc_html($entry['title']); ?> → <?php echo (int)$entry['chars']; ?> chars</div>
                            <?php else: ?>
                                <div style="color:#ff6b6b">✗ <?php echo esc_html($entry['title']); ?>: <?php echo esc_html($entry['message']); ?></div>
                            <?php endif; ?>
                        <?php endforeach; ?>
                        </div>
                    </details>
                    <?php endif; ?>
                </div>
            <?php else: ?>
                <p style="padding:16px 20px;margin:0;color:#50575e">No batch has run yet.</p>
            <?php endif; ?>
            </div><!-- /ab-zone-body -->
            </div><!-- /ab-card-lastrun -->

        </div><!-- /ab-pane-batch -->

        <script>
        // ── Tab switching ────────────────────────────────────────────────────
        function abTab(id, btn) {
            document.querySelectorAll('.ab-pane').forEach(p => p.classList.remove('active'));
            document.querySelectorAll('.ab-tab').forEach(b  => b.classList.remove('active'));
            document.getElementById('ab-pane-' + id).classList.add('active');
            btn.classList.add('active');
            if (id === 'sitemap') abRefreshRobotsPreview();
        }

        // ── State ────────────────────────────────────────────────────────────
        const abState = {
            posts:         [],
            page:          1,
            totalPages:    1,
            total:         0,
            totalWithDesc: 0,
            generated:     0,
            stopped:       false,
            running:       false,
        };

        const abNonce   = <?php echo wp_json_encode($nonce); ?>;
        const abAjax    = <?php echo wp_json_encode(admin_url('admin-ajax.php')); ?>;
        const abMinChar = <?php echo (int) $this->ai_opts['min_chars']; ?>;
        const abMaxChar = <?php echo (int) $this->ai_opts['max_chars']; ?>;
        const abHasApiKey = <?php echo json_encode(!empty(trim((string)($this->ai_opts['anthropic_key'] ?? '')))); ?>;

        // ── Live robots.txt preview ──────────────────────────────────────────
        function abRefreshRobotsPreview() {
            try {
            const pre = document.getElementById('ab-robots-live-preview');
            if (!pre) return;
            pre.textContent = 'Loading…';
            const params = new URLSearchParams({action: 'cs_seo_fetch_robots', nonce: abNonce});
            fetch(abAjax, {method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: params})
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        pre.textContent = data.data.content;
                    } else {
                        pre.textContent = '(error: ' + data.data + ')';
                    }
                })
                .catch(e => { pre.textContent = '(fetch error: ' + e.message + ')'; });
            } catch(e) { console.warn('abRefreshRobotsPreview error:', e); }
        }
        // Auto-load robots preview on page load if sitemap tab is active
        document.addEventListener('DOMContentLoaded', function() {
            if (document.getElementById('ab-pane-sitemap')?.classList.contains('active')) {
                abRefreshRobotsPreview();
            }
        });

        // ── Rename physical robots.txt ───────────────────────────────────────
        function abRenameRobots() {
            const btn    = document.getElementById('ab-rename-robots-btn');
            const status = document.getElementById('ab-rename-robots-status');
            btn.disabled = true;
            btn.textContent = '⟳ Renaming...';
            status.style.color = '#50575e';
            status.textContent = 'Working...';

            const params = new URLSearchParams({action: 'cs_seo_rename_robots', nonce: abNonce});
            fetch(abAjax, {method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: params})
                .then(r => r.json())
                .then(data => {
                    if (data.success) {
                        const warn = document.getElementById('ab-physical-robots-warn');
                        if (warn) {
                            warn.style.background = '#edfaef';
                            warn.style.borderColor = '#1a7a34';
                            warn.innerHTML = '<div style="font-size:22px">✅</div>' +
                                '<div><strong>Done!</strong> robots.txt has been renamed to robots.txt.bak. ' +
                                'The plugin is now managing your robots.txt. ' +
                                'Purge your Cloudflare cache, then <a href="' + window.location.href + '">reload this page</a> to confirm.</div>';
                        }
                    } else {
                        btn.disabled = false;
                        btn.textContent = 'Rename robots.txt → robots.txt.bak';
                        status.style.color = '#c3372b';
                        status.textContent = '✗ ' + data.data;
                    }
                })
                .catch(e => {
                    btn.disabled = false;
                    btn.textContent = 'Rename robots.txt → robots.txt.bak';
                    status.style.color = '#c3372b';
                    status.textContent = '✗ Network error: ' + e.message;
                });
        }
        let abSitemapPage = 1;

        function abLoadSitemap(pg) {
            abSitemapPage = pg || 1;
            const wrap = document.getElementById('ab-sitemap-preview-wrap');
            const btn  = document.getElementById('ab-sitemap-load');
            if (!wrap || !btn) {
                console.error('CloudScale SEO: sitemap preview elements not found');
                return;
            }
            console.log('CloudScale SEO: loading sitemap preview page', abSitemapPage);
            btn.disabled = true;
            btn.textContent = '⟳ Loading...';
            if (abSitemapPage === 1) {
                wrap.innerHTML = '<p style="color:#50575e;font-size:13px">Fetching sitemap entries...</p>';
            }

            const params = new URLSearchParams({
                action: 'cs_seo_sitemap_preview',
                nonce: abNonce,
                sitemap_pg: abSitemapPage,
            });
            fetch(abAjax, {method:'POST', headers:{'Content-Type':'application/x-www-form-urlencoded'}, body: params})
                .then(r => r.json())
                .then(data => {
                    btn.disabled = false;
                    btn.textContent = '↻ Reload';
                    if (!data.success) {
                        wrap.innerHTML = '<div style="background:#fef0f0;border:1px solid #f5bcbb;border-radius:4px;padding:12px;color:#c3372b"><strong>Preview failed:</strong> ' + (data.data || 'Unknown error') + '<br><small>Check that your API key is set and the plugin settings have been saved.</small></div>';
                        return;
                    }
                    const d        = data.data;
                    const entries  = d.entries;
                    const total    = d.total;
                    const page     = d.page;
                    const pages    = d.pages;
                    const per_page = d.per_page;
                    const start    = (page - 1) * per_page + 1;
                    const end      = Math.min(page * per_page, total);

                    const typeLabels = {home:'Home', post:'Post', page:'Page', tax:'Taxonomy', cpt:'CPT'};
                    const typeClass  = t => 'ab-sitemap-type ab-sitemap-type-' + (t || 'post');

                    let rows = entries.map(e =>
                        '<tr>' +
                        '<td><a class="ab-sitemap-url" href="' + e.loc + '" target="_blank">' + e.loc + '</a>' +
                        (e.title ? '<br><small style="color:#888">' + e.title + '</small>' : '') + '</td>' +
                        '<td><span class="' + typeClass(e.type) + '">' + (typeLabels[e.type] || e.type) + '</span></td>' +
                        '<td style="color:#888;font-size:12px;white-space:nowrap">' + (e.lastmod || '—') + '</td>' +
                        '</tr>'
                    ).join('');

                    // Pager
                    let pager = '';
                    if (pages > 1) {
                        const sitemapLinks = Array.from({length: pages}, (_, i) => {
                            const n = i + 1;
                            const active = n === page ? 'font-weight:700;color:#1d2327' : 'color:#2271b1;cursor:pointer';
                            return '<span style="' + active + ';padding:0 4px" ' +
                                (n !== page ? 'onclick="abLoadSitemap(' + n + ')"' : '') + '>' + n + '</span>';
                        }).join(' ');
                        pager = '<div style="display:flex;align-items:center;gap:10px;margin-top:12px;flex-wrap:wrap">' +
                            '<button class="button" ' + (page <= 1 ? 'disabled' : '') + ' onclick="abLoadSitemap(' + (page-1) + ')">← Prev</button>' +
                            '<span style="font-size:12px;color:#50575e">Page ' + page + ' of ' + pages + '</span>' +
                            '<button class="button" ' + (page >= pages ? 'disabled' : '') + ' onclick="abLoadSitemap(' + (page+1) + ')">Next →</button>' +
                            '<span style="font-size:12px;color:#888;margin-left:auto">Showing ' + start + '–' + end + ' of ' + total + ' URLs</span>' +
                            '</div>';
                    }

                    wrap.innerHTML =
                        '<p class="ab-sitemap-count"><strong>' + total + '</strong> total URLs across <strong>' + pages + '</strong> sitemap file' + (pages > 1 ? 's' : '') +
                        ' &nbsp;·&nbsp; <a href="' + <?php echo wp_json_encode(home_url('/sitemap.xml')); ?> + '" target="_blank">View sitemap index ↗</a></p>' +
                        '<table class="ab-sitemap-tbl">' +
                        '<thead><tr><th>URL</th><th>Type</th><th>Last Modified</th></tr></thead>' +
                        '<tbody>' + rows + '</tbody></table>' +
                        pager;
                })
                .catch(e => {
                    btn.disabled = false;
                    btn.textContent = '⬇ Load Preview';
                    wrap.innerHTML = '<div style="background:#fef0f0;border:1px solid #f5bcbb;border-radius:4px;padding:12px;color:#c3372b"><strong>Network error:</strong> ' + e.message + '</div>';
                    wrap.innerHTML = '<p style="color:#c3372b">Error: ' + e.message + '</p>';
                });
        }

        // ── API key guard ─────────────────────────────────────────────────────
        function abCheckApiKey() {
            if (abHasApiKey) return true;
            document.getElementById('ab-api-warn').classList.add('visible');
            abLog('⚠ No API key saved. Scroll up to the ✦ AI Meta Writer section, enter your Anthropic API key and click Save AI Settings, then reload the page.', 'err');
            return false;
        }

        // Show warning banner on page load if no key saved
        if (!abHasApiKey) {
            document.addEventListener('DOMContentLoaded', function() {
                document.getElementById('ab-api-warn').classList.add('visible');
            });
        }

        // ── Utilities ────────────────────────────────────────────────────────
        function abLog(msg, type) {
            const el = document.getElementById('ab-log');
            el.classList.add('visible');
            const ts = new Date().toLocaleTimeString('en-GB');
            el.innerHTML += '<div class="ab-log-' + (type||'info') + '">[' + ts + '] ' + abEsc(msg) + '</div>';
            el.scrollTop = el.scrollHeight;
        }

        function abEsc(s) {
            return String(s).replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;');
        }

        function abSetStatus(msg) {
            document.getElementById('ab-toolbar-status').textContent = msg;
        }

        function abSetProgress(done, total) {
            const pct = total > 0 ? Math.round(done/total*100) : 0;
            document.getElementById('ab-progress').classList.add('visible');
            document.getElementById('ab-progress-fill').style.width = pct + '%';
            document.getElementById('ab-prog-label').textContent =
                done + ' / ' + total + ' processed (' + pct + '%)';
        }

        function abUpdateSummary() {
            const total   = abState.total;
            const hasDesc = abState.totalWithDesc + abState.generated;
            const missing = Math.max(0, total - hasDesc);
            document.getElementById('sum-total').textContent     = total;
            document.getElementById('sum-has').textContent       = hasDesc;
            document.getElementById('sum-missing').textContent   = missing;
            document.getElementById('sum-generated').textContent = abState.generated;
            document.getElementById('ab-summary').style.display  = 'grid';
        }

        function abPost(action, extra) {
            const params = new URLSearchParams({action, nonce: abNonce, ...extra});
            return fetch(abAjax, {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: params
            }).then(r => r.json());
        }

        // ── Test API key ─────────────────────────────────────────────────────
        function abTestKey() {
            const status  = document.getElementById('ab-key-status');
            const keyField = document.getElementById('ab-api-key-field');
            const key     = keyField.value.trim();
            if (!key) {
                status.textContent = '✗ Enter a key first';
                status.className   = 'ab-key-status ab-key-err';
                return;
            }
            status.textContent = '⟳ Testing...';
            status.className   = 'ab-key-status';

            // Pass the key directly so we test what's in the field right now,
            // without needing to save the form first.
            fetch(abAjax, {
                method: 'POST',
                headers: {'Content-Type': 'application/x-www-form-urlencoded'},
                body: new URLSearchParams({
                    action: 'cs_seo_ai_test_key',
                    nonce:  abNonce,
                    live_key: key,
                })
            })
            .then(r => r.json())
            .then(data => {
                if (data.success) {
                    status.textContent = '✓ ' + data.data;
                    status.className   = 'ab-key-status ab-key-ok';
                } else {
                    status.textContent = '✗ ' + data.data;
                    status.className   = 'ab-key-status ab-key-err';
                }
            })
            .catch(e => {
                status.textContent = '✗ Network error: ' + e.message;
                status.className   = 'ab-key-status ab-key-err';
            });
        }

        // ── Load posts ───────────────────────────────────────────────────────
        function abLoadPosts(page) {
            page = page || 1;
            abState.page = page;
            document.getElementById('ab-load-posts').disabled = true;
            abSetStatus('Loading posts...');
            abPost('cs_seo_ai_get_posts', {page}).then(data => {
                document.getElementById('ab-load-posts').disabled = false;
                if (!data.success) { abLog('Failed to load posts: ' + data.data, 'err'); return; }
                abState.posts      = data.data.posts;
                abState.total          = data.data.total;
                abState.totalWithDesc  = data.data.total_with_desc;
                abState.totalPages     = data.data.total_pages;
                abState.page       = data.data.page;
                abUpdateSummary();
                abRenderTable();
                abSetStatus(data.data.total + ' posts loaded');
                // Hide the load CTA, show the action toolbar
                document.getElementById('ab-load-cta').style.display = 'none';
                document.getElementById('ab-ai-toolbar').style.display = 'flex';
                document.getElementById('ab-ai-gen-missing').disabled = false;
                document.getElementById('ab-ai-gen-all').disabled = false;
                document.getElementById('ab-ai-fix').disabled = false;
                // Pager
                const pager = document.getElementById('ab-pager');
                pager.style.display = abState.totalPages > 1 ? 'flex' : 'none';
                document.getElementById('ab-page-info').textContent =
                    'Page ' + abState.page + ' of ' + abState.totalPages;
                document.getElementById('ab-prev').disabled = abState.page <= 1;
                document.getElementById('ab-next').disabled = abState.page >= abState.totalPages;
            }).catch(e => {
                document.getElementById('ab-load-posts').disabled = false;
                abLog('Error: ' + e.message, 'err');
            });
        }

        function abPage(dir) {
            abLoadPosts(abState.page + dir);
        }

        // ── Render table ─────────────────────────────────────────────────────
        function abBadge(post) {
            if (!post.has_desc && !post._gen) return '<span class="ab-badge ab-badge-none">No AI description</span>';
            const desc  = post._gen || post.desc;
            const chars = desc ? desc.length : 0;
            if (post._gen) return '<span class="ab-badge ab-badge-gen">✦ Generated · ' + chars + 'c</span>';
            if (chars >= abMinChar && chars <= abMaxChar) return '<span class="ab-badge ab-badge-ok">✓ ' + chars + 'c</span>';
            if (chars > 0 && chars < abMinChar)           return '<span class="ab-badge ab-badge-short">Short · ' + chars + 'c</span>';
            if (chars > abMaxChar)                         return '<span class="ab-badge ab-badge-long">Long · ' + chars + 'c</span>';
            return '<span class="ab-badge ab-badge-none">No AI description</span>';
        }

        function abRenderTable() {
            const wrap = document.getElementById('ab-posts-wrap');
            if (!abState.posts.length) {
                wrap.innerHTML = '<p style="color:#50575e">No posts found.</p>';
                return;
            }
            let rows = abState.posts.map(p => {
                const existDesc = p.desc
                    ? '<div class="ab-desc-text">' + abEsc(p.desc) + '</div>'
                    : '';
                const genDesc = p._gen
                    ? '<div class="ab-desc-gen">✦ ' + abEsc(p._gen) + '</div>'
                    : '';
                const canGen = !p._processing;
                return '<tr id="ab-row-' + p.id + '">' +
                    '<td><strong>' + abEsc(p.title) + '</strong><br><small style="color:#888">' + p.type + ' · ' + p.date + '</small></td>' +
                    '<td>' + abBadge(p) + existDesc + genDesc + '</td>' +
                    '<td>' +
                        '<button class="button ab-row-btn" onclick="abGenOne(' + p.id + ')" ' + (canGen?'':'disabled') + ' id="ab-btn-' + p.id + '">' +
                        (p._processing ? '<span class="ab-spinner">⟳</span>' : '✦') + ' Generate</button>' +
                    '</td>' +
                '</tr>';
            }).join('');

            wrap.innerHTML = '<table class="ab-posts">' +
                '<thead><tr><th style="width:40%">Post</th><th style="width:45%">Description</th><th style="width:15%">Action</th></tr></thead>' +
                '<tbody>' + rows + '</tbody></table>';
        }

        // ── Generate one post ─────────────────────────────────────────────────
        function abGenOne(postId) {
            if (!abCheckApiKey()) return;
            const post = abState.posts.find(p => p.id === postId);
            if (!post) return;
            post._processing = true;
            abRenderTable();

            abPost('cs_seo_ai_generate_one', {post_id: postId}).then(data => {
                post._processing = false;
                if (data.success) {
                    post._gen     = data.data.description;
                    post.has_desc = true;
                    post.desc     = data.data.description;
                    abLog('✓ "' + post.title.slice(0,55) + '" → ' + data.data.chars + ' chars', 'ok');
                    abState.generated++;
                    abUpdateSummary();
                } else {
                    abLog('✗ "' + post.title.slice(0,45) + '": ' + (data.data || 'Unknown error'), 'err');
                }
                abRenderTable();
            }).catch(e => {
                post._processing = false;
                abLog('✗ Network error: ' + e.message, 'err');
                abRenderTable();
            });
        }

        // ── Generate all ──────────────────────────────────────────────────────
        async function abGenAll(overwrite) {
            if (!abCheckApiKey()) return;
            if (abState.running) return;
            abState.stopped = false;
            abState.running = true;

            document.getElementById('ab-ai-gen-missing').disabled = true;
            document.getElementById('ab-ai-gen-all').disabled = true;
            document.getElementById('ab-ai-fix').disabled = true;
            document.getElementById('ab-ai-stop').style.display = 'inline-block';

            abLog(overwrite ? 'Starting full regeneration run...' : 'Starting generation run (missing only)...', 'info');

            let allPosts = [];
            abSetStatus('Fetching full post list...');
            for (let pg = 1; pg <= abState.totalPages; pg++) {
                if (abState.stopped) break;
                try {
                    const data = await abPost('cs_seo_ai_get_posts', {page: pg});
                    if (data.success) allPosts = allPosts.concat(data.data.posts);
                } catch(e) {}
            }

            const targets = allPosts.filter(p => !p.has_desc || overwrite);
            abLog('Found ' + targets.length + ' posts to process', 'info');

            let done = 0, errors = 0, skipped = 0;

            for (const post of targets) {
                if (abState.stopped) { abLog('Stopped by user after ' + done + ' posts', 'skip'); break; }

                abSetStatus('Processing: "' + post.title.slice(0,50) + '"...');
                abSetProgress(done, targets.length);

                try {
                    const data = await abPost('cs_seo_ai_generate_all', {
                        post_id:   post.id,
                        overwrite: overwrite ? 1 : 0,
                    });

                    if (data.success) {
                        const r = data.data;
                        if (r.status === 'skipped') {
                            skipped++;
                            abLog('⊘ "' + post.title.slice(0,55) + '" — skipped (has desc)', 'skip');
                        } else {
                            done++;
                            abState.generated++;
                            abLog('✓ "' + post.title.slice(0,55) + '" → ' + r.chars + ' chars', 'ok');
                            const local = abState.posts.find(p => p.id === post.id);
                            if (local) { local._gen = r.description; local.has_desc = true; local.desc = r.description; }
                        }
                    } else {
                        errors++;
                        const msg = typeof data.data === 'object' ? data.data.message : data.data;
                        abLog('✗ "' + post.title.slice(0,45) + '": ' + msg, 'err');
                        await abSleep(12000); // longer pause on error — likely a rate limit
                    }
                } catch(e) {
                    errors++;
                    abLog('✗ Network error: ' + e.message, 'err');
                    await abSleep(12000);
                }

                abUpdateSummary();
                abRenderTable();
                await abSleep(2500); // ~24 posts/min — stays under Anthropic's 30k token/min limit
            }

            abSetProgress(done + skipped, targets.length);
            abSetStatus('Done — ' + done + ' generated, ' + skipped + ' skipped, ' + errors + ' errors');
            abLog('Run complete: ' + done + ' generated, ' + skipped + ' skipped, ' + errors + ' errors', done > 0 ? 'ok' : 'info');

            document.getElementById('ab-ai-gen-missing').disabled = false;
            document.getElementById('ab-ai-gen-all').disabled      = false;
            document.getElementById('ab-ai-fix').disabled          = false;
            document.getElementById('ab-ai-stop').style.display    = 'none';
            abState.running = false;
        }

        // ── Fix out-of-range descriptions ──────────────────────────────────────
        async function abFixAll() {
            if (!abCheckApiKey()) return;
            if (abState.running) return;
            abState.stopped = false;
            abState.running = true;

            document.getElementById('ab-ai-gen-missing').disabled = true;
            document.getElementById('ab-ai-gen-all').disabled     = true;
            document.getElementById('ab-ai-fix').disabled         = true;
            document.getElementById('ab-ai-stop').style.display   = 'inline-block';

            abLog('Starting fix run — scanning for short and long descriptions...', 'info');

            // Fetch all posts across all pages.
            let allPosts = [];
            abSetStatus('Fetching full post list...');
            for (let pg = 1; pg <= abState.totalPages; pg++) {
                if (abState.stopped) break;
                try {
                    const data = await abPost('cs_seo_ai_get_posts', {page: pg});
                    if (data.success) allPosts = allPosts.concat(data.data.posts);
                } catch(e) {}
            }

            // Target only posts that have a description but it's outside the configured range.
            const targets = allPosts.filter(p => {
                if (!p.has_desc || !p.desc) return false;
                const len = p.desc.length;
                return len < abMinChar || len > abMaxChar;
            });

            if (targets.length === 0) {
                abLog('No out-of-range descriptions found — nothing to fix.', 'info');
                abSetStatus('Nothing to fix.');
                document.getElementById('ab-ai-gen-missing').disabled = false;
                document.getElementById('ab-ai-gen-all').disabled     = false;
                document.getElementById('ab-ai-fix').disabled         = false;
                document.getElementById('ab-ai-stop').style.display   = 'none';
                abState.running = false;
                return;
            }

            abLog('Found ' + targets.length + ' descriptions to fix (' + abMinChar + '–' + abMaxChar + ' char target)', 'info');

            let done = 0, errors = 0, skipped = 0;

            for (const post of targets) {
                if (abState.stopped) { abLog('Stopped by user after ' + done + ' posts', 'skip'); break; }

                const len = post.desc ? post.desc.length : 0;
                const issue = len < abMinChar ? 'too short (' + len + 'c)' : 'too long (' + len + 'c)';
                abSetStatus('Fixing: "' + post.title.slice(0,50) + '" — ' + issue);
                abSetProgress(done, targets.length);

                try {
                    const data = await abPost('cs_seo_ai_fix_desc', {post_id: post.id});

                    if (data.success) {
                        const r = data.data;
                        if (r.status === 'skipped') {
                            skipped++;
                            abLog('⊘ "' + post.title.slice(0,55) + '" — ' + r.message, 'skip');
                        } else {
                            done++;
                            abState.generated++;
                            if (r.in_range) {
                                abLog('✓ "' + post.title.slice(0,55) + '" fixed: ' + r.was_chars + 'c → ' + r.chars + 'c', 'ok');
                            } else {
                                abLog('⚠ "' + post.title.slice(0,55) + '" still out of range after retries: ' + r.was_chars + 'c → ' + r.chars + 'c', 'err');
                            }
                            const local = abState.posts.find(p => p.id === post.id);
                            if (local) { local._gen = r.description; local.has_desc = true; local.desc = r.description; }
                        }
                    } else {
                        errors++;
                        const msg = typeof data.data === 'object' ? data.data.message : data.data;
                        abLog('✗ "' + post.title.slice(0,45) + '": ' + msg, 'err');
                        await abSleep(12000);
                    }
                } catch(e) {
                    errors++;
                    abLog('✗ Network error: ' + e.message, 'err');
                    await abSleep(12000);
                }

                abUpdateSummary();
                abRenderTable();
                await abSleep(2500);
            }

            abSetProgress(done + skipped, targets.length);
            abSetStatus('Fix run done — ' + done + ' fixed, ' + skipped + ' skipped, ' + errors + ' errors');
            abLog('Fix run complete: ' + done + ' fixed, ' + skipped + ' skipped, ' + errors + ' errors', done > 0 ? 'ok' : 'info');

            document.getElementById('ab-ai-gen-missing').disabled = false;
            document.getElementById('ab-ai-gen-all').disabled     = false;
            document.getElementById('ab-ai-fix').disabled         = false;
            document.getElementById('ab-ai-stop').style.display   = 'none';
            abState.running = false;
        }

        function abStop() { abState.stopped = true; abSetStatus('Stopping...'); }
        function abSleep(ms) { return new Promise(r => setTimeout(r, ms)); }
        </script>
        </div><!-- /wrap -->
        <?php
    }

    private function tr_text(string $k, string $label, array $o, string $placeholder = '', string $hint = ''): void { ?>
        <tr><th><label><?php echo esc_html($label); ?></label></th>
            <td>
                <input class="regular-text"
                    name="<?php echo esc_attr(self::OPT); ?>[<?php echo esc_attr($k); ?>]"
                    value="<?php echo esc_attr((string)($o[$k] ?? '')); ?>"
                    <?php if ($placeholder) echo 'placeholder="' . esc_attr($placeholder) . '"'; ?>>
                <?php if ($hint): ?>
                    <p class="description"><?php echo esc_html($hint); ?></p>
                <?php endif; ?>
            </td></tr>
    <?php }

    private function tr_bool(string $k, string $label, array $o): void { ?>
        <tr><th><?php echo esc_html($label); ?></th>
            <td><label><input type="checkbox" name="<?php echo esc_attr(self::OPT); ?>[<?php echo esc_attr($k); ?>]" value="1" <?php checked((int)($o[$k] ?? 0), 1); ?>> Enabled</label></td></tr>
    <?php }

    // =========================================================================
    public function ajax_fetch_robots(): void {
        check_ajax_referer('cs_seo_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorised');
        }
        // Read directly from disk if a physical file exists, otherwise generate
        $physical = ABSPATH . 'robots.txt';
        if (file_exists($physical)) {
            $content = file_get_contents($physical);
            wp_send_json_success(['content' => $content, 'source' => 'file']);
        } else {
            // Generate what WordPress/plugin would serve
            // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedHooknameFound -- 'robots_txt' is a WordPress core filter
            $content = apply_filters('robots_txt', '', (bool) get_option('blog_public'));
            wp_send_json_success(['content' => $content ?: '(empty — WordPress default)', 'source' => 'dynamic']);
        }
    }

    public function ajax_rename_robots(): void {
        check_ajax_referer('cs_seo_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorised');
        }
        // Check ABSPATH first, then one level up for subdirectory installs
        $physical = ABSPATH . 'robots.txt';
        if (!file_exists($physical)) {
            $alt = dirname(rtrim(ABSPATH, '/')) . '/robots.txt';
            if (file_exists($alt)) {
                $physical = $alt;
            }
        }
        $backup = preg_replace('/robots\.txt$/', 'robots.txt.bak', $physical);
        if (!file_exists($physical)) {
            wp_send_json_error('No physical robots.txt file found — nothing to rename.');
        }
        if (!wp_is_writable(dirname($physical))) {
            wp_send_json_error('robots.txt exists but is not writable. Check file permissions (should be 644).');
        }
        $old_content = file_get_contents($physical); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        if ($wp_filesystem->move($physical, $backup, true)) {
            update_option('cs_seo_robots_bak', $old_content);
            wp_send_json_success(['message' => 'robots.txt renamed to robots.txt.bak. The plugin is now managing your robots.txt.']);
        } else {
            wp_send_json_error('rename() failed — check that the web server has write access to the WordPress root directory.');
        }
    }

    // Sitemap  (paginated index — 200 URLs per child sitemap)
    // =========================================================================

    const SITEMAP_PER_FILE    = 5000; // URLs per XML sitemap file served to Google
    const SITEMAP_PREVIEW_PER = 200;  // Rows per page in the admin preview table

    public function maybe_register_sitemap(): void {
        if (!(int) $this->opts['enable_sitemap']) return;
        // /sitemap.xml          → sitemap index listing all child sitemaps
        // /sitemap-1.xml etc.   → child sitemaps with up to 200 URLs each
        add_rewrite_rule('^sitemap\.xml$',       'index.php?cs_seo_sitemap=index', 'top');
        add_rewrite_rule('^sitemap-(\d+)\.xml$', 'index.php?cs_seo_sitemap=page&cs_seo_sitemap_pg=$matches[1]', 'top');
        add_rewrite_tag('%cs_seo_sitemap%',    '(index|page)');
        add_rewrite_tag('%cs_seo_sitemap_pg%', '\d+');
        add_action('template_redirect', [$this, 'maybe_render_sitemap']);
    }

    public function maybe_render_sitemap(): void {
        $mode = get_query_var('cs_seo_sitemap');
        if (!$mode) return;
        header('Content-Type: application/xml; charset=utf-8');
        header('X-Robots-Tag: noindex, follow');
        header('Cache-Control: public, max-age=3600');
        if ($mode === 'index') {
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $this->build_sitemap_index();
        } else {
            $pg = max(1, (int) get_query_var('cs_seo_sitemap_pg'));
            // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
            echo $this->build_sitemap_page($pg);
        }
        exit;
    }

    // Shared: build full URL list from settings (used by both XML output and preview)
    private function get_all_sitemap_urls(): array {
        $post_types  = (array)($this->opts['sitemap_post_types'] ?? ['post', 'page']);
        $inc_tax     = (int)($this->opts['sitemap_taxonomies'] ?? 0);
        $exclude_raw = trim((string)($this->opts['sitemap_exclude'] ?? ''));

        $exclude_urls = [];
        $exclude_ids  = [];
        if ($exclude_raw !== '') {
            foreach (preg_split('/\r?\n/', $exclude_raw) as $line) {
                $line = trim($line);
                if ($line === '') continue;
                if (is_numeric($line)) {
                    $exclude_ids[] = (int) $line;
                } else {
                    $exclude_urls[] = trailingslashit($line);
                }
            }
        }

        $urls = [['loc' => home_url('/'), 'lastmod' => gmdate('c'), 'type' => 'home', 'title' => 'Homepage']];

        if (!empty($post_types)) {
            $q = new WP_Query([
                'post_type'           => $post_types,
                'post_status'         => 'publish',
                'posts_per_page'      => -1,
                'no_found_rows'       => true,
                'ignore_sticky_posts' => true,
                'orderby'             => 'modified',
                'order'               => 'DESC',
                'post__not_in'        => $exclude_ids ?: [0], // phpcs:ignore WordPressVIPMinimum.Performance.WPQueryParams.PostNotIn_post__not_in -- required for user-defined URL exclusions
                'fields'              => 'ids',
            ]);
            foreach ($q->posts as $pid) {
                $pid       = (int) $pid;
                $permalink = get_permalink($pid);
                if (in_array(trailingslashit($permalink), $exclude_urls, true)) continue;
                $pt   = get_post_type($pid);
                $type = $pt === 'page' ? 'page' : ($pt === 'post' ? 'post' : 'cpt');
                $urls[] = [
                    'loc'     => $permalink,
                    'lastmod' => get_post_modified_time('c', true, $pid),
                    'type'    => $type,
                    'title'   => get_the_title($pid),
                ];
            }
        }

        if ($inc_tax) {
            foreach (get_taxonomies(['public' => true], 'names') as $tax) {
                $terms = get_terms(['taxonomy' => $tax, 'hide_empty' => true, 'number' => 0]);
                if (is_wp_error($terms)) continue;
                foreach ($terms as $term) {
                    $link = get_term_link($term);
                    if (is_wp_error($link)) continue;
                    if (in_array(trailingslashit($link), $exclude_urls, true)) continue;
                    $urls[] = [
                        'loc'     => $link,
                        'lastmod' => '',
                        'type'    => 'tax',
                        'title'   => $term->name . ' (' . $tax . ')',
                    ];
                }
            }
        }

        return $urls;
    }

    private function build_sitemap_index(): string {
        $all        = $this->get_all_sitemap_urls();
        $total      = count($all);
        $per_page   = self::SITEMAP_PER_FILE;
        $page_count = max(1, (int) ceil($total / $per_page));

        $xml  = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
        $xml .= "<sitemapindex xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
        for ($i = 1; $i <= $page_count; $i++) {
            $xml .= "  <sitemap>\n";
            $xml .= "    <loc>" . esc_url(home_url("/sitemap-{$i}.xml")) . "</loc>\n";
            $xml .= "    <lastmod>" . esc_html(gmdate('c')) . "</lastmod>\n";
            $xml .= "  </sitemap>\n";
        }
        $xml .= "</sitemapindex>\n";
        return $xml;
    }

    private function build_sitemap_page(int $pg): string {
        $all      = $this->get_all_sitemap_urls();
        $per_page = self::SITEMAP_PER_FILE;
        $slice    = array_slice($all, ($pg - 1) * $per_page, $per_page);

        $xml  = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
        $xml .= "<urlset xmlns=\"http://www.sitemaps.org/schemas/sitemap/0.9\">\n";
        foreach ($slice as $u) {
            $xml .= "  <url>\n";
            $xml .= "    <loc>" . esc_url($u['loc']) . "</loc>\n";
            if (!empty($u['lastmod'])) {
                $xml .= "    <lastmod>" . esc_html($u['lastmod']) . "</lastmod>\n";
            }
            $xml .= "  </url>\n";
        }
        $xml .= "</urlset>\n";
        return $xml;
    }

    // AJAX preview — returns paginated entries for the UI table
    public function ajax_sitemap_preview(): void {
        check_ajax_referer('cs_seo_nonce', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorised');
        }
        // Preview works regardless of enable_sitemap so you can check before enabling
        $all      = $this->get_all_sitemap_urls();
        $total    = count($all);
        $per_page = self::SITEMAP_PREVIEW_PER;
        $pages    = max(1, (int) ceil($total / $per_page));
        $pg       = max(1, min($pages, absint(wp_unslash($_POST['sitemap_pg'] ?? 1))));
        $slice    = array_slice($all, ($pg - 1) * $per_page, $per_page);

        // Normalise lastmod to Y-m-d for display
        $entries = array_map(function($u) {
            $lm = $u['lastmod'] ?? '';
            if ($lm) {
                $ts = strtotime($lm);
                $lm = $ts ? gmdate('Y-m-d', $ts) : '';
            }
            return [
                'loc'     => $u['loc'],
                'type'    => $u['type'],
                'lastmod' => $lm,
                'title'   => $u['title'] ?? '',
            ];
        }, $slice);

        wp_send_json_success([
            'entries'   => $entries,
            'total'     => $total,
            'page'      => $pg,
            'pages'     => $pages,
            'per_page'  => $per_page,
        ]);
    }

    // =========================================================================
    // Robots.txt
    // =========================================================================

    public function filter_robots_txt(string $output, bool $public): string {
        if (!$public) {
            return "User-agent: *\nDisallow: /\n";
        }

        // Use saved custom robots.txt content, falling back to default.
        $custom = trim((string)($this->opts['robots_txt'] ?? ''));
        if ($custom === '') {
            $custom = self::default_robots_txt();
        }

        $lines = explode("\n", $custom);

        // Append AI training bot blocklist if enabled.
        if ((int)($this->opts['block_ai_bots'] ?? 1)) {
            $lines[] = '';
            foreach ([
                'GPTBot', 'ChatGPT-User', 'CCBot', 'anthropic-ai', 'Claude-Web',
                'Omgilibot', 'FacebookBot', 'Bytespider', 'Applebot-Extended',
            ] as $bot) {
                $lines[] = 'User-agent: ' . $bot;
                $lines[] = 'Disallow: /';
                $lines[] = '';
            }
        }

        // Append sitemap directive if enabled.
        if ((int)($this->opts['enable_sitemap'] ?? 0)) {
            $lines[] = '';
            $lines[] = 'Sitemap: ' . home_url('/sitemap.xml');
        }

        $content = implode("\n", $lines);
        $content = preg_replace('/[ \t]+$/m', '', $content);
        return rtrim($content) . "\n";
    }
}

// Flush rewrite rules on activation so sitemap URLs work immediately,
// and on deactivation to clean up.
register_activation_hook(__FILE__, function(): void {
    // If a physical robots.txt exists in the WordPress root, rename it so
    // WordPress's robots_txt filter can take over. We keep the original as
    // robots.txt.bak so it can be restored if needed.
    $root      = ABSPATH;
    $physical  = $root . 'robots.txt';
    $backup    = $root . 'robots.txt.bak';
    if (file_exists($physical) && wp_is_writable($physical)) {
        // Save the old content into plugin options so the user can review it.
        $old_content = file_get_contents($physical); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
        update_option('cs_seo_robots_bak', $old_content);
        global $wp_filesystem;
        if (empty($wp_filesystem)) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
            WP_Filesystem();
        }
        $wp_filesystem->move($physical, $backup, true);
    }
    // Register rewrites first so flush has something to work with.
    if (get_option('cs_seo_opts')['enable_sitemap'] ?? 0) {
        add_rewrite_rule('^sitemap\.xml$',       'index.php?cs_seo_sitemap=index', 'top');
        add_rewrite_rule('^sitemap-(\d+)\.xml$', 'index.php?cs_seo_sitemap=page&cs_seo_sitemap_pg=$matches[1]', 'top');
    }
    flush_rewrite_rules();
});

register_deactivation_hook(__FILE__, function(): void {
    flush_rewrite_rules();
    wp_clear_scheduled_hook('cs_seo_daily_batch');
});

new CloudScale_SEO_AI_Optimizer();
