/**
 * CloudScale Code Block — Frontend Script v1.4.0
 */

( function() {
    'use strict';

    var defaultTheme = ( window.csCodeConfig && window.csCodeConfig.defaultTheme ) || 'dark';

    var langNames = {
        'bash':       'Bash',
        'shell':      'Shell',
        'sh':         'Shell',
        'zsh':        'Zsh',
        'python':     'Python',
        'py':         'Python',
        'java':       'Java',
        'javascript': 'JavaScript',
        'js':         'JavaScript',
        'typescript': 'TypeScript',
        'ts':         'TypeScript',
        'csharp':     'C#',
        'cs':         'C#',
        'c':          'C',
        'cpp':        'C++',
        'rust':       'Rust',
        'rs':         'Rust',
        'php':        'PHP',
        'go':         'Go',
        'ruby':       'Ruby',
        'rb':         'Ruby',
        'swift':      'Swift',
        'kotlin':     'Kotlin',
        'sql':        'SQL',
        'json':       'JSON',
        'yaml':       'YAML',
        'yml':        'YAML',
        'xml':        'XML',
        'html':       'HTML',
        'css':        'CSS',
        'scss':       'SCSS',
        'less':       'Less',
        'markdown':   'Markdown',
        'md':         'Markdown',
        'makefile':   'Makefile',
        'dockerfile': 'Dockerfile',
        'lua':        'Lua',
        'perl':       'Perl',
        'r':          'R',
        'ini':        'INI',
        'toml':       'TOML',
        'diff':       'Diff',
        'graphql':    'GraphQL',
        'plaintext':  'Text',
        'objectivec': 'Objective-C',
        'vbnet':      'VB.NET',
        'wasm':       'WebAssembly'
    };

    function getThemeLinks() {
        var links = document.querySelectorAll( 'link[rel="stylesheet"]' );
        var dark = null, light = null;
        for ( var i = 0; i < links.length; i++ ) {
            var href = links[i].getAttribute( 'href' ) || '';
            if ( href.indexOf( 'atom-one-dark' ) !== -1 ) dark = links[i];
            if ( href.indexOf( 'atom-one-light' ) !== -1 ) light = links[i];
        }
        return { dark: dark, light: light };
    }

    function applyGlobalTheme( activeTheme ) {
        var links = getThemeLinks();
        if ( activeTheme === 'light' ) {
            if ( links.dark ) links.dark.disabled = true;
            if ( links.light ) links.light.disabled = false;
        } else {
            if ( links.dark ) links.dark.disabled = false;
            if ( links.light ) links.light.disabled = true;
        }
    }

    function syncGlobalTheme() {
        var wrappers = document.querySelectorAll( '.cs-code-wrapper' );
        var darkCount = 0, lightCount = 0;
        for ( var i = 0; i < wrappers.length; i++ ) {
            if ( wrappers[i].getAttribute( 'data-active-theme' ) === 'light' ) lightCount++;
            else darkCount++;
        }
        applyGlobalTheme( lightCount > darkCount ? 'light' : 'dark' );
    }

    /**
     * Build or rebuild line numbers gutter for a code block.
     */
    function buildLineNumbers( wrapper ) {
        var codeEl = wrapper.querySelector( 'pre code' );
        if ( ! codeEl ) return;

        // Remove existing gutter if present
        var existing = wrapper.querySelector( '.cs-code-line-numbers' );
        if ( existing ) existing.remove();

        // Only build if line numbers are active
        if ( ! wrapper.classList.contains( 'cs-lines-active' ) ) return;

        var text      = codeEl.textContent || '';
        var lineCount = text.split( '\n' ).length;

        // If code ends with a newline, the last split entry is empty so don't count it
        if ( text.charAt( text.length - 1 ) === '\n' ) {
            lineCount--;
        }
        if ( lineCount < 1 ) lineCount = 1;

        var gutter = document.createElement( 'div' );
        gutter.className = 'cs-code-line-numbers';
        gutter.setAttribute( 'aria-hidden', 'true' );

        var nums = [];
        for ( var i = 1; i <= lineCount; i++ ) {
            nums.push( '<span>' + i + '</span>' );
        }
        gutter.innerHTML = nums.join( '\n' );

        // Insert gutter inside the .cs-code-body before the <pre>
        var body = wrapper.querySelector( '.cs-code-body' );
        var pre  = wrapper.querySelector( 'pre' );
        if ( body && pre ) {
            body.insertBefore( gutter, pre );
        }
    }

    function initBlock( wrapper ) {
        var codeEl = wrapper.querySelector( 'pre code' );
        if ( !codeEl ) return;

        // Set theme: per block override > global default
        var blockTheme = wrapper.getAttribute( 'data-theme' ) || defaultTheme;
        wrapper.setAttribute( 'data-active-theme', blockTheme );

        // Run highlight.js
        hljs.highlightElement( codeEl );

        // Language badge
        var detectedLang = '';
        var match = ( codeEl.className || '' ).match( /language-(\S+)/ );
        if ( match ) detectedLang = match[1];

        var badge = wrapper.querySelector( '.cs-code-lang-badge' );
        if ( badge && detectedLang ) {
            badge.textContent = langNames[ detectedLang.toLowerCase() ] || detectedLang;
        }

        // Copy button
        var copyBtn = wrapper.querySelector( '.cs-code-copy' );
        if ( copyBtn ) {
            copyBtn.addEventListener( 'click', function() {
                var text = codeEl.textContent || codeEl.innerText;
                copyToClipboard( text ).then( function() {
                    copyBtn.classList.add( 'copied' );
                    setTimeout( function() {
                        copyBtn.classList.remove( 'copied' );
                    }, 2000 );
                } );
            } );
        }

        // Theme toggle
        var toggleBtn = wrapper.querySelector( '.cs-code-theme-toggle' );
        if ( toggleBtn ) {
            toggleBtn.addEventListener( 'click', function() {
                var current = wrapper.getAttribute( 'data-active-theme' );
                var next = ( current === 'dark' ) ? 'light' : 'dark';
                wrapper.setAttribute( 'data-active-theme', next );

                // Switch the global hljs stylesheet so syntax colors update
                applyGlobalTheme( next );

                // Re-highlight
                var raw = codeEl.textContent;
                codeEl.removeAttribute( 'data-highlighted' );
                codeEl.classList.remove( 'hljs' );
                codeEl.textContent = raw;
                hljs.highlightElement( codeEl );
            } );
        }

        // Line numbers toggle
        var linesBtn = wrapper.querySelector( '.cs-code-lines-toggle' );
        if ( linesBtn ) {
            linesBtn.addEventListener( 'click', function() {
                wrapper.classList.toggle( 'cs-lines-active' );
                buildLineNumbers( wrapper );
            } );
        }
    }

    function copyToClipboard( text ) {
        if ( navigator.clipboard && window.isSecureContext ) {
            return navigator.clipboard.writeText( text );
        }
        var ta = document.createElement( 'textarea' );
        ta.value = text;
        ta.style.position = 'fixed';
        ta.style.left = '-9999px';
        document.body.appendChild( ta );
        ta.select();
        try { document.execCommand( 'copy' ); } catch ( e ) { /* silent */ }
        document.body.removeChild( ta );
        return Promise.resolve();
    }

    function boot() {
        if ( typeof hljs === 'undefined' ) {
            // hljs not loaded yet, retry
            setTimeout( boot, 100 );
            return;
        }

        hljs.configure( {
            languages: [
                'bash', 'shell', 'python', 'java', 'javascript', 'typescript',
                'csharp', 'c', 'cpp', 'rust', 'php', 'go', 'ruby', 'swift',
                'kotlin', 'sql', 'json', 'yaml', 'xml', 'html', 'css', 'scss',
                'markdown', 'makefile', 'lua', 'perl', 'r', 'ini', 'diff',
                'graphql', 'objectivec', 'vbnet'
            ]
        } );

        // Set global theme stylesheet before highlighting
        applyGlobalTheme( defaultTheme );

        var wrappers = document.querySelectorAll( '.cs-code-wrapper' );
        for ( var i = 0; i < wrappers.length; i++ ) {
            initBlock( wrappers[i] );
        }

        syncGlobalTheme();
    }

    if ( document.readyState === 'loading' ) {
        document.addEventListener( 'DOMContentLoaded', boot );
    } else {
        boot();
    }
} )();
