/**
 * CloudScale Code Block Migrator - Admin JS
 */
(function () {
    'use strict';

    var scanBtn      = document.getElementById('cs-scan-btn');
    var migrateAllBtn = document.getElementById('cs-migrate-all-btn');
    var statusEl     = document.getElementById('cs-scan-status');
    var resultsArea  = document.getElementById('cs-results-area');
    var modal        = document.getElementById('cs-preview-modal');
    var modalTitle   = document.getElementById('cs-modal-title');
    var modalBody    = document.getElementById('cs-modal-body');
    var modalMigrateBtn = document.getElementById('cs-modal-migrate-btn');

    var scannedPosts = [];

    // =========================================================================
    //  Helpers
    // =========================================================================

    function ajax(action, data, callback) {
        var fd = new FormData();
        fd.append('action', action);
        fd.append('nonce', csMigrate.nonce);
        if (data) {
            Object.keys(data).forEach(function (k) {
                fd.append(k, data[k]);
            });
        }
        fetch(csMigrate.ajaxUrl, { method: 'POST', body: fd })
            .then(function (r) { return r.json(); })
            .then(function (resp) {
                if (resp.success) {
                    callback(null, resp.data);
                } else {
                    callback(resp.data || 'Unknown error');
                }
            })
            .catch(function (err) {
                callback(err.message || 'Network error');
            });
    }

    function setStatus(msg, type) {
        statusEl.textContent = msg;
        statusEl.className = 'cs-status' + (type ? ' ' + type : '');
    }

    function escHtml(str) {
        var div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    }

    // =========================================================================
    //  Scan
    // =========================================================================

    function doScan() {
        scanBtn.disabled = true;
        setStatus('Scanning...', '');
        resultsArea.innerHTML = '<p class="cs-loading"><span class="cs-spinner"></span> Scanning all posts for legacy code blocks...</p>';

        ajax('cs_migrate_scan', {}, function (err, data) {
            scanBtn.disabled = false;

            if (err) {
                setStatus('Scan failed: ' + err, 'error');
                return;
            }

            scannedPosts = data.posts;

            if (data.total_posts === 0) {
                setStatus('No legacy code blocks found.', 'success');
                resultsArea.innerHTML = '<p class="cs-migrate-hint">No posts with legacy code blocks were found. Everything is already migrated.</p>';
                migrateAllBtn.disabled = true;
                return;
            }

            setStatus('Found ' + data.total_blocks + ' block(s) across ' + data.total_posts + ' post(s).', 'success');
            migrateAllBtn.disabled = false;
            renderResults(data);
        });
    }

    function renderResults(data) {
        var html = '';

        // Summary
        html += '<div class="cs-migrate-summary">';
        html += '<div class="cs-stat"><strong>' + data.total_posts + '</strong>Posts with legacy blocks</div>';
        html += '<div class="cs-stat"><strong>' + data.total_blocks + '</strong>Total code blocks to migrate</div>';
        html += '</div>';

        // Table
        html += '<table class="cs-migrate-table">';
        html += '<thead><tr>';
        html += '<th>Post</th>';
        html += '<th style="width:90px;text-align:center;">Blocks</th>';
        html += '<th style="width:80px;">Status</th>';
        html += '<th style="width:200px;">Actions</th>';
        html += '</tr></thead>';
        html += '<tbody>';

        data.posts.forEach(function (post) {
            html += '<tr id="cs-row-' + post.id + '">';
            html += '<td>';
            html += '<div class="cs-post-title"><a href="' + escHtml(post.view_url) + '" target="_blank">' + escHtml(post.title) + '</a></div>';
            html += '<div class="cs-post-meta">' + escHtml(post.date) + ' &middot; ' + escHtml(post.status) + ' &middot; ID: ' + post.id + '</div>';
            html += '</td>';
            html += '<td style="text-align:center;"><span class="cs-block-count">' + post.block_count + '</span></td>';
            html += '<td class="cs-status-cell">Pending</td>';
            html += '<td class="cs-actions">';
            html += '<button class="button button-small cs-preview-btn" data-post-id="' + post.id + '">Preview</button> ';
            html += '<button class="button button-primary button-small cs-single-migrate-btn" data-post-id="' + post.id + '">Migrate</button>';
            html += '</td>';
            html += '</tr>';
        });

        html += '</tbody></table>';

        resultsArea.innerHTML = html;

        // Bind preview buttons
        resultsArea.querySelectorAll('.cs-preview-btn').forEach(function (btn) {
            btn.addEventListener('click', function () {
                openPreview(parseInt(this.getAttribute('data-post-id')));
            });
        });

        // Bind single migrate buttons
        resultsArea.querySelectorAll('.cs-single-migrate-btn').forEach(function (btn) {
            btn.addEventListener('click', function () {
                migrateSingle(parseInt(this.getAttribute('data-post-id')), this);
            });
        });
    }

    // =========================================================================
    //  Preview Modal
    // =========================================================================

    function openPreview(postId) {
        modal.style.display = 'flex';
        modalTitle.textContent = 'Loading preview...';
        modalBody.innerHTML = '<p class="cs-loading"><span class="cs-spinner"></span> Loading block preview...</p>';
        modalMigrateBtn.setAttribute('data-post-id', postId);

        ajax('cs_migrate_preview', { post_id: postId }, function (err, data) {
            if (err) {
                modalBody.innerHTML = '<p style="color:#d63638;">Error: ' + escHtml(err) + '</p>';
                return;
            }

            modalTitle.textContent = data.title + ' (' + data.block_count + ' block' + (data.block_count !== 1 ? 's' : '') + ')';

            var html = '';
            data.blocks.forEach(function (block) {
                html += '<div class="cs-preview-block">';
                html += '<div class="cs-preview-block-header">';
                html += '<span class="cs-block-num">' + block.index + '</span>';
                html += '<span class="cs-block-lang">' + escHtml(block.language) + '</span>';
                html += '<span class="cs-block-lang" style="background:#f3e8ff;color:#7c3aed;">' + escHtml(block.type || 'wp:code') + '</span>';
                html += '<span class="cs-block-firstline">' + escHtml(block.first_line) + '</span>';
                html += '</div>';
                html += '<div class="cs-preview-diff">';
                html += '<div class="cs-preview-side cs-before">';
                html += '<div class="cs-preview-side-label">Before (wp:code)</div>';
                html += '<pre>' + block.original + '</pre>';
                html += '</div>';
                html += '<div class="cs-preview-side cs-after">';
                html += '<div class="cs-preview-side-label">After (cloudscale/code-block)</div>';
                html += '<pre>' + block.converted + '</pre>';
                html += '</div>';
                html += '</div>';
                html += '</div>';
            });

            modalBody.innerHTML = html;
        });
    }

    function closeModal() {
        modal.style.display = 'none';
    }

    // Modal close handlers
    modal.querySelector('.cs-modal-backdrop').addEventListener('click', closeModal);
    modal.querySelectorAll('.cs-modal-close, .cs-modal-close-btn').forEach(function (el) {
        el.addEventListener('click', closeModal);
    });

    // Migrate from modal
    modalMigrateBtn.addEventListener('click', function () {
        var postId = parseInt(this.getAttribute('data-post-id'));
        if (!postId) return;

        this.disabled = true;
        this.textContent = 'Migrating...';

        ajax('cs_migrate_single', { post_id: postId }, function (err, data) {
            modalMigrateBtn.disabled = false;
            modalMigrateBtn.innerHTML = '<span class="dashicons dashicons-yes-alt"></span> Migrate This Post';

            if (err) {
                alert('Migration failed: ' + err);
                return;
            }

            closeModal();
            markRowMigrated(postId, data.blocks_migrated);
            setStatus(data.message, 'success');
        });
    });

    // =========================================================================
    //  Single Migrate (from table button)
    // =========================================================================

    function migrateSingle(postId, btn) {
        if (!confirm('Migrate all code blocks in this post to CloudScale format?')) return;

        btn.disabled = true;
        btn.textContent = 'Migrating...';

        ajax('cs_migrate_single', { post_id: postId }, function (err, data) {
            if (err) {
                btn.disabled = false;
                btn.textContent = 'Migrate';
                alert('Migration failed: ' + err);
                return;
            }

            markRowMigrated(postId, data.blocks_migrated);
            setStatus(data.message, 'success');
        });
    }

    function markRowMigrated(postId, blockCount) {
        var row = document.getElementById('cs-row-' + postId);
        if (!row) return;

        row.classList.add('cs-migrated');

        var statusCell = row.querySelector('.cs-status-cell');
        if (statusCell) {
            statusCell.innerHTML = '<span class="cs-migrated-badge"><span class="dashicons dashicons-yes"></span> Done</span>';
        }

        var actionsCell = row.querySelector('.cs-actions');
        if (actionsCell) {
            actionsCell.innerHTML = '<a href="' + getViewUrl(postId) + '" target="_blank" class="button button-small">View Post</a>';
        }

        // Check if all rows are migrated
        var pending = document.querySelectorAll('.cs-single-migrate-btn');
        if (pending.length === 0) {
            migrateAllBtn.disabled = true;
            setStatus('All posts migrated successfully!', 'success');
        }
    }

    function getViewUrl(postId) {
        for (var i = 0; i < scannedPosts.length; i++) {
            if (scannedPosts[i].id === postId) return scannedPosts[i].view_url;
        }
        return '#';
    }

    // =========================================================================
    //  Migrate All
    // =========================================================================

    function doMigrateAll() {
        var pending = document.querySelectorAll('.cs-single-migrate-btn');
        var count = pending.length;

        if (count === 0) {
            alert('No remaining posts to migrate.');
            return;
        }

        if (!confirm('This will migrate ' + count + ' remaining post(s) to CloudScale Code Blocks. Continue?')) {
            return;
        }

        migrateAllBtn.disabled = true;
        migrateAllBtn.textContent = 'Migrating all...';
        setStatus('Migrating all remaining posts...', '');

        ajax('cs_migrate_all', {}, function (err, data) {
            migrateAllBtn.innerHTML = '<span class="dashicons dashicons-update"></span> Migrate All Remaining';

            if (err) {
                migrateAllBtn.disabled = false;
                setStatus('Batch migration failed: ' + err, 'error');
                return;
            }

            setStatus(
                'Migrated ' + data.migrated_blocks + ' block(s) across ' + data.migrated_posts + ' post(s).',
                'success'
            );

            // Mark all rows as done
            data.details.forEach(function (detail) {
                var match = detail.match(/^#(\d+)/);
                if (match) {
                    markRowMigrated(parseInt(match[1]), 0);
                }
            });

            migrateAllBtn.disabled = true;
        });
    }

    // =========================================================================
    //  Event Bindings
    // =========================================================================

    scanBtn.addEventListener('click', doScan);
    migrateAllBtn.addEventListener('click', doMigrateAll);

    // Escape key closes modal
    document.addEventListener('keydown', function (e) {
        if (e.key === 'Escape' && modal.style.display !== 'none') {
            closeModal();
        }
    });

})();
